# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for the base of external apps API."""

from __future__ import annotations

__all__ = ()

import typing
from unittest import mock
import attr
import pytest

if typing.TYPE_CHECKING:
    from _pytest.mark import structures as _pytest_structures

    from anaconda_navigator.api.external_apps import parsing_utils as t_parsing_utils


@pytest.fixture(scope='function')
def context() -> typing.Iterator['t_parsing_utils.ParsingContext']:
    """Prepare context for parsing routines."""
    from anaconda_navigator.api.external_apps import bundle
    from anaconda_navigator.api.external_apps import parsing_utils

    backup: typing.Any = bundle.jetbrains_is_available
    bundle.jetbrains_is_available = lambda: False  # pylint: disable=assigning-non-slot

    yield parsing_utils.ParsingContext(
        process_api=mock.Mock(),
        user_configuration=mock.Mock(),
    )

    bundle.jetbrains_is_available = backup  # pylint: disable=assigning-non-slot


def validate_equal(value: typing.Any) -> typing.Callable[[typing.Any], bool]:
    """Check that `item` is equal to `value`."""
    def result(item: typing.Any) -> bool:
        if value is None:
            return item is None
        return item == value
    return result


def validate_endswith(value: typing.Any) -> typing.Callable[[typing.Any], bool]:
    """Check that `item` ends with `value`."""
    def result(item: typing.Any) -> bool:
        if value is None:
            return item is None
        return item.endswith(value)
    return result


def validate_is_available(value: typing.Any) -> typing.Callable[[typing.Any], bool]:
    """Check that `item` ends with `value`."""
    def result(item: typing.Any) -> bool:
        if value == '::jetbrains_is_available':
            return item is False
        return item is value
    return result


@attr.s(auto_attribs=True, cache_hash=True, frozen=True, order=False, slots=True)
class Value:  # pylint: disable=too-few-public-methods
    """Description of a single value in description dictionary."""

    key: str
    values: typing.Sequence[typing.Any]
    required: bool = attr.ib(default=False)

    attribute: str = attr.ib(default=None)
    default: typing.Any = attr.ib(default=None)
    validate: typing.Callable[[typing.Any], typing.Callable[[typing.Any], bool]] = attr.ib(default=validate_equal)

    def __attrs_post_init__(self) -> None:
        """Additional initialization of the class."""
        if self.attribute is None:
            object.__setattr__(self, 'attribute', self.key)  # type: ignore


class Cases:
    """Container for :class:`~Value` instances, that is able to generate test cases"""

    __slots__ = ('__content',)

    def __init__(self, *args: Value) -> None:
        """Initialize new :class:`~Cases` instance."""
        self.__content: typing.Final[typing.Tuple[Value, ...]] = args

    @property
    def keys(self) -> typing.Sequence[str]:  # noqa: D401
        """All registered value keys."""
        return tuple(
            item.key
            for item in self.__content
        )

    @property
    def optional(self) -> typing.Sequence[str]:  # noqa: D401
        """List of optional keys."""
        return tuple(
            item.key
            for item in self.__content
            if item.required is False
        )

    @property
    def required(self) -> typing.Sequence[str]:  # noqa: D401
        """List of required values."""
        return tuple(
            item.key
            for item in self.__content
            if item.required is True
        )

    def configuration(
            self,
            *,
            optional: bool = False,
            include: typing.Collection[str] = (),
            exclude: typing.Collection[str] = (),
    ) -> typing.Iterator[typing.Mapping[str, typing.Any]]:
        """Prepare configurations for a specification."""
        result: typing.List[typing.Mapping[str, typing.Any]] = [{}]

        item: Value
        for item in self.__content:
            if (item.key in exclude) or not (item.required or optional or (item.key in include)):
                continue
            result = [
                {**key, item.key: value}
                for key in result
                for value in item.values
            ]

        yield from result

    def pair(
            self,
            *,
            optional: bool = False,
            include: typing.Collection[str] = (),
            exclude: typing.Collection[str] = (),
    ) -> typing.Iterator[
        typing.Tuple[typing.Mapping[str, typing.Any], typing.Mapping[str, typing.Callable[[typing.Any], bool]]]
    ]:
        """Prepare a pair of configuration and validators for a specification."""
        result: typing.List[
            typing.Tuple[
                typing.MutableMapping[str, typing.Any],
                typing.MutableMapping[str, typing.Callable[[typing.Any], bool]],
            ]
        ] = [({}, {})]

        item: Value
        for item in self.__content:
            if (item.key in exclude) or not (item.required or optional or (item.key in include)):
                result = [
                    (first, {**second, item.attribute: item.validate(item.default)})
                    for first, second in result
                ]
            else:
                result = [
                    ({**first, item.key: value}, {**second, item.key: item.validate(value)})
                    for first, second in result
                    for value in item.values
                ]

        yield from result

    def positive_cases(self) -> typing.Iterator['_pytest_structures.ParameterSet']:
        """Generate positive cases for a test."""
        index: int
        pair: typing.Tuple[typing.Mapping[str, typing.Any], typing.Mapping[str, typing.Callable[[typing.Any], bool]]]

        for index, pair in enumerate(self.pair(optional=False), start=1):
            yield pytest.param(*pair, id=f'minimal_{index:03d}')

        for index, pair in enumerate(self.pair(optional=True), start=1):
            yield pytest.param(*pair, id=f'complete_{index:03d}')

        key: str
        for key in self.optional:
            for index, pair in enumerate(self.pair(optional=False, include={key}), start=1):
                yield pytest.param(*pair, id=f'minimal_plus_{key}_{index:03d}')

            for index, pair in enumerate(self.pair(optional=True, exclude={key}), start=1):
                yield pytest.param(*pair, id=f'complete_but_{key}_{index:03d}')

    def negative_cases(self) -> typing.Iterator['_pytest_structures.ParameterSet']:
        """Generate negative cases for a test."""
        index: int
        configuration: typing.Mapping[str, typing.Any]

        extra: typing.MutableMapping[str, typing.Any] = {
            'unexpected_field': '',
        }

        key: str
        for key in self.required:
            for index, configuration in enumerate(self.configuration(optional=False, exclude={key}), start=1):
                yield pytest.param(configuration, id=f'minimal_but_{key}_{index:03d}')
            for index, configuration in enumerate(self.configuration(optional=True, exclude={key}), start=1):
                yield pytest.param(configuration, id=f'complete_but_{key}_{index:03d}')

        for key in self.keys:
            extra.pop(key, None)
            for index, configuration in enumerate(self.configuration(optional=False), start=1):
                yield pytest.param({**configuration, key: object()}, id=f'minimal_broken_{key}_{index:03d}')
            for index, configuration in enumerate(self.configuration(optional=True), start=1):
                yield pytest.param({**configuration, key: object()}, id=f'complete_broken_{key}_{index:03d}')

        value: typing.Any
        for key, value in extra.items():
            for index, configuration in enumerate(self.configuration(optional=False), start=1):
                yield pytest.param({**configuration, key: value}, id=f'minimal_plus_{key}_{index:03d}')
            for index, configuration in enumerate(self.configuration(optional=True), start=1):
                yield pytest.param({**configuration, key: value}, id=f'complete_plus_{key}_{index:03d}')


APP_PATCH_DATA: typing.Final[Cases] = Cases(
    Value(key='display_name', values=['NEW_DISPLAY_NAME'], default=None),
    Value(key='description', values=['NEW_DESCRIPTION'], default=None),
    Value(key='image_path', values=['python.png'], default=None, validate=validate_endswith),
    Value(
        key='is_available',
        values=[True, False, '::jetbrains_is_available'],
        default=True,
        validate=validate_is_available,
    ),
    Value(key='rank', values=[123], default=None),
)


@pytest.mark.parametrize(['configuration', 'validators'], APP_PATCH_DATA.positive_cases())
def test_positive_patch_parsing(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
        validators: typing.Mapping[str, typing.Callable[[typing.Any], bool]],
) -> None:
    """Perform a set of positive checks for :class:`~anaconda_navigator.api.external_apps.base.AppPatch`."""
    from anaconda_navigator.api.external_apps import base

    result: typing.Union[None, base.BaseApp, base.AppPatch] = base.BaseApp.parse_configuration(
        context=context,
        configuration=configuration,
        app_name='APP_PATCH',
    )
    assert isinstance(result, base.AppPatch)

    key: str
    validator: typing.Callable[[typing.Any], bool]
    for key, validator in validators.items():
        value: typing.Any = getattr(result, key)
        assert validator(value), f'unexpected {key}={value!r} for {configuration!r}'


@pytest.mark.parametrize('configuration', APP_PATCH_DATA.negative_cases())
def test_negative_patch_parsing(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
) -> None:
    """Perform a set of positive checks for :class:`~anaconda_navigator.api.external_apps.base.AppPatch`."""
    from anaconda_navigator.api.external_apps import base
    from anaconda_navigator.api.external_apps import exceptions

    with pytest.raises(exceptions.ValidationError):
        base.BaseApp.parse_configuration(
            context=context,
            configuration=configuration,
            app_name='APP_PATCH',
        )


WEB_APP_DATA: typing.Final[Cases] = Cases(
    Value(key='app_type', values=['web_app'], required=True),
    Value(key='display_name', values=['NEW_DISPLAY_NAME'], required=True),
    Value(key='description', values=['NEW_DESCRIPTION'], default=''),
    Value(key='image_path', values=['python.png'], default='anaconda-icon-256x256.png', validate=validate_endswith),
    Value(key='is_available', values=[True], default=True, validate=validate_is_available),
    Value(key='rank', values=[123], default=0),
    Value(key='url', values=['NEW_URL'], required=True),
)


@pytest.mark.parametrize(['configuration', 'validators'], WEB_APP_DATA.positive_cases())
def test_positive_web_parsing(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
        validators: typing.Mapping[str, typing.Callable[[typing.Any], bool]],
) -> None:
    """Perform a set of positive checks for :class:`~anaconda_navigator.api.external_apps.base.BaseWebApp`."""
    from anaconda_navigator.api.external_apps import base

    result: typing.Union[None, base.BaseApp, base.AppPatch] = base.BaseApp.parse_configuration(
        context=context,
        configuration=configuration,
        app_name='WEB_APP',
    )
    assert isinstance(result, base.BaseWebApp)

    key: str
    validator: typing.Callable[[typing.Any], bool]
    for key, validator in validators.items():
        value: typing.Any = getattr(result, key)
        assert validator(value), f'unexpected {key}={value!r} for {configuration!r}'


@pytest.mark.parametrize('configuration', WEB_APP_DATA.negative_cases())
def test_negative_web_parsing(
        context: 't_parsing_utils.ParsingContext',  # pylint: disable=redefined-outer-name
        configuration: typing.Mapping[str, typing.Any],
) -> None:
    """Perform a set of positive checks for :class:`~anaconda_navigator.api.external_apps.base.BaseWebApp`."""
    from anaconda_navigator.api.external_apps import base
    from anaconda_navigator.api.external_apps import exceptions

    with pytest.raises(exceptions.ValidationError):
        base.BaseApp.parse_configuration(
            context=context,
            configuration=configuration,
            app_name='WEB_APP',
        )
