# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2021 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for env selector dialog."""

from __future__ import annotations

__all__ = ()

import types
import typing
from unittest import mock
import requests


class CustomCloudAPI:
    """Mocked CloudAPI instance."""

    __slots__ = ('list_environments', '__contexts', '__depth')

    def __init__(self) -> None:
        """Initialize new :class:`~CustomCloudAPI` instance."""
        from anaconda_navigator.utils import workers

        self.list_environments: typing.Final[workers.Task] = workers.Task(
            mock.Mock(return_value={'items': [], 'total': 0}),
        )

        self.__contexts: typing.Final[typing.List[typing.ContextManager]] = []
        self.__depth: int = 0

    def __enter__(self) -> 'CustomCloudAPI':
        """Enter into CloudAPI mock context."""
        if self.__depth <= 0:
            patch: typing.ContextManager = mock.patch(
                'anaconda_navigator.api.cloud.CloudAPI',
                mock.Mock(return_value=self),
            )
            patch.__enter__()
            self.__contexts.append(patch)

            patch = mock.patch(
                'anaconda_navigator.utils.workers.TaskWorker.start',
                lambda instance: instance.run(),
            )
            patch.__enter__()
            self.__contexts.append(patch)

            self.__depth += 1
        return self

    def __exit__(
            self,
            exc_type: typing.Optional[typing.Type[BaseException]],
            exc_val: typing.Optional[BaseException],
            exc_tb: typing.Optional[types.TracebackType],
    ) -> bool:
        """Exit from CloudAPI mock context."""
        self.__depth -= 1
        if self.__depth <= 0:
            result: bool = any(
                context.__exit__(exc_type, exc_val, exc_tb)
                for context in self.__contexts
            )
            self.__contexts.clear()
            return result
        return False


def generate_names(start: int, stop: int) -> typing.List[str]:
    """Generate list of environment names."""
    return [f'env_{index:04d}' for index in range(start + 1, stop + 1)]


def generate_response(start: int, stop: int, total: int) -> typing.Dict[str, typing.Any]:
    """Generate environments in an API response format."""
    return {
        'items': [{'name': name} for name in generate_names(start=start, stop=stop)],
        'total': total,
    }


def test_get_env_names():  # pylint: disable=missing-function-docstring
    from anaconda_navigator.widgets.dialogs.environment import import_dialogs

    source = generate_response(0, 20, 30)
    expectation = generate_names(0, 20)
    assert import_dialogs.EnvSelectorDialog.get_env_names(source) == expectation


def test_fetched_environments(qtbot, qt_styles):  # pylint: disable=missing-function-docstring,unused-argument
    from anaconda_navigator.utils import styles
    from anaconda_navigator.widgets.dialogs.environment import import_dialogs

    api: CustomCloudAPI
    with CustomCloudAPI() as api:
        api.list_environments.function.return_value = generate_response(0, 10, 17)

        dialog = import_dialogs.EnvSelectorDialog()
        dialog.setStyleSheet(styles.load_style_sheet())
        dialog.show()
        qtbot.addWidget(dialog)

        api.list_environments.function.assert_called_once_with(
            limit=import_dialogs.ENVIRONMENT_FETCH_LIMIT,
            offset=0,
            sort=import_dialogs.ENVIRONMENT_ORDER,
        )
        api.list_environments.function.reset_mock()
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 10)
        assert dialog.error_label.text == ''

        # Second page

        api.list_environments.function.return_value = generate_response(10, 17, 17)

        scroll_bar = dialog.list.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

        api.list_environments.function.assert_called_once_with(
            limit=import_dialogs.ENVIRONMENT_FETCH_LIMIT,
            offset=10,
            sort=import_dialogs.ENVIRONMENT_ORDER,
        )
        api.list_environments.function.reset_mock()
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 17)
        assert dialog.error_label.text == ''

        # Third page

        api.list_environments.function.return_value = generate_response(17, 17, 17)

        scroll_bar = dialog.list.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 17)
        assert dialog.error_label.text == ''


def test_provided_environments(qtbot, qt_styles):  # pylint: disable=missing-function-docstring,unused-argument
    from anaconda_navigator.utils import styles
    from anaconda_navigator.widgets.dialogs.environment import import_dialogs

    api: CustomCloudAPI
    with CustomCloudAPI() as api:
        api.list_environments.function.return_value = generate_response(0, 3, 3)

        dialog = import_dialogs.EnvSelectorDialog(envs=generate_response(0, 3, 3))
        dialog.setStyleSheet(styles.load_style_sheet())
        dialog.show()
        qtbot.addWidget(dialog)

        api.list_environments.function.assert_not_called()
        api.list_environments.function.reset_mock()
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 3)
        assert dialog.error_label.text == ''

        # Second page

        api.list_environments.function.return_value = generate_response(3, 3, 3)

        scroll_bar = dialog.list.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 3)
        assert dialog.error_label.text == ''


def test_mixed_environments(qtbot, qt_styles):  # pylint: disable=missing-function-docstring,unused-argument
    from anaconda_navigator.utils import styles
    from anaconda_navigator.widgets.dialogs.environment import import_dialogs

    api: CustomCloudAPI
    with CustomCloudAPI() as api:
        api.list_environments.function.return_value = generate_response(0, 10, 19)

        dialog = import_dialogs.EnvSelectorDialog(envs=generate_response(0, 10, 19))
        dialog.setStyleSheet(styles.load_style_sheet())
        dialog.show()
        qtbot.addWidget(dialog)

        api.list_environments.function.assert_not_called()
        api.list_environments.function.reset_mock()
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 10)
        assert dialog.error_label.text == ''

        # Second page

        api.list_environments.function.return_value = generate_response(10, 19, 19)

        scroll_bar = dialog.list.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

        api.list_environments.function.assert_called_once_with(
            limit=import_dialogs.ENVIRONMENT_FETCH_LIMIT,
            offset=10,
            sort=import_dialogs.ENVIRONMENT_ORDER,
        )
        api.list_environments.function.reset_mock()
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 19)
        assert dialog.error_label.text == ''

        # Third page

        api.list_environments.function.return_value = generate_response(19, 19, 19)

        scroll_bar = dialog.list.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 19)
        assert dialog.error_label.text == ''


def test_fetch_exception(qtbot, qt_styles):  # pylint: disable=missing-function-docstring,unused-argument
    from anaconda_navigator.utils import styles
    from anaconda_navigator.widgets.dialogs.environment import import_dialogs

    api: CustomCloudAPI
    with CustomCloudAPI() as api:
        api.list_environments.function.return_value = generate_response(0, 10, 12)

        dialog = import_dialogs.EnvSelectorDialog()
        dialog.setStyleSheet(styles.load_style_sheet())
        dialog.show()
        qtbot.addWidget(dialog)

        api.list_environments.function.assert_called_once_with(
            limit=import_dialogs.ENVIRONMENT_FETCH_LIMIT,
            offset=0,
            sort=import_dialogs.ENVIRONMENT_ORDER,
        )
        api.list_environments.function.reset_mock()
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 10)
        assert dialog.error_label.text == ''

        # Second page

        api.list_environments.function.side_effect = requests.RequestException()

        scroll_bar = dialog.list.verticalScrollBar()
        scroll_bar.setValue(scroll_bar.maximum())

        api.list_environments.function.assert_called_once_with(
            limit=import_dialogs.ENVIRONMENT_FETCH_LIMIT,
            offset=10,
            sort=import_dialogs.ENVIRONMENT_ORDER,
        )
        assert dialog.current_item.name == 'env_0001'
        assert [item.name for item in dialog.list.items()] == generate_names(0, 10)
        assert dialog.error_label.text != ''
