# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for environment-related dialogs."""

from __future__ import annotations

__all__ = ()

import typing
import pytest
from qtpy.QtCore import Qt
from tests.current.widgets import common_extra


class WorkerMock():  # pylint: disable=missing-class-docstring,too-few-public-methods
    prefix = 'some/prefix'


PACKAGES = ['boom']
OUTPUT_INSTALL_SUCCESS: typing.Mapping[str, typing.Any] = {
    'success': True,
    'error': '',
    'exception_name': '',
    'actions': [
        {
            'LINK': [
                {
                    'name': 'boom',
                    'version': '1.5',
                    'channel': 'goanpeca',
                    'dist_name': 'boom-1.5-build0',
                },
                {
                    'name': 'dep-1',
                    'version': '1.6',
                    'channel': 'goanpeca',
                    'dist_name': 'dep-1-1.6-build0',
                },
            ],
            'UNLINK': [
                {
                    'name': 'boom',
                    'version': '1.4',
                    'channel': 'goanpeca',
                    'dist_name': 'boom-1.4-build0',
                },
                {
                    'name': 'dep-1',
                    'version': '1.7',
                    'channel': 'goanpeca',
                    'dist_name': 'dep-1-1.7-build0'
                },
            ],
        },
    ],
}

OUTPUT_REMOVE_SUCCESS: typing.Mapping[str, typing.Any] = {
    'success':
    True,
    'error':
    '',
    'exception_name':
    '',
    'actions': [
        {
            'LINK': [],
            'UNLINK': [
                {
                    'name': 'boom',
                    'version': '1.4',
                    'channel': 'goanpeca',
                    'dist_name': 'boom-1.4-build1',
                },
            ],
        },
    ],
}

OUTPUT_INSTALL_FAIL: typing.Mapping[str, typing.Any] = {
    'success': False,
    'error': 'Something happened!',
    'exception_name': 'WeirdException',
    'actions': [],
}

OUTPUT_EMPTY: typing.Mapping[str, typing.Any] = {}


@pytest.fixture(scope='function')
def botpackages(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs import packages

    widget = packages.PackagesDialog(packages=PACKAGES, remove_only=False)
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botpackagesremove(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs import packages

    widget = packages.PackagesDialog(packages=PACKAGES, remove_only=True)
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


def test_install_success(botpackages):  # pylint: disable=redefined-outer-name
    botpackages.widget.setup(WorkerMock(), OUTPUT_INSTALL_SUCCESS, None)
    assert botpackages.widget.button_ok.isEnabled()
    assert botpackages.widget.button_cancel.isEnabled()
    with botpackages.qtbot.waitSignal(botpackages.widget.accepted, timeout=5000, raising=True):
        botpackages.qtbot.keyPress(botpackages.widget.button_ok, Qt.Key_Enter)


def test_remove_success(botpackagesremove):  # pylint: disable=redefined-outer-name
    botpackagesremove.widget.setup(WorkerMock(), OUTPUT_REMOVE_SUCCESS, None)
    with botpackagesremove.qtbot.waitSignal(botpackagesremove.widget.accepted, timeout=10000, raising=False):
        pass
    assert botpackagesremove.widget.button_ok.isEnabled()
    assert botpackagesremove.widget.button_cancel.isEnabled()
    with botpackagesremove.qtbot.waitSignal(botpackagesremove.widget.accepted, timeout=5000, raising=True):
        botpackagesremove.qtbot.keyPress(botpackagesremove.widget.button_ok, Qt.Key_Enter)


def test_fail(botpackages):  # pylint: disable=redefined-outer-name
    botpackages.widget.setup(WorkerMock(), OUTPUT_INSTALL_FAIL, None)
    assert not botpackages.widget.button_ok.isEnabled()
    assert botpackages.widget.button_cancel.isEnabled()
    with botpackages.qtbot.waitSignal(botpackages.widget.rejected, timeout=5000, raising=True):
        botpackages.widget.reject()


def test_output_empty(botpackages):  # pylint: disable=redefined-outer-name
    botpackages.widget.setup(WorkerMock(), OUTPUT_EMPTY, None)
    assert not botpackages.widget.button_ok.isEnabled()
    assert botpackages.widget.button_cancel.isEnabled()
    with botpackages.qtbot.waitSignal(botpackages.widget.rejected, timeout=5000, raising=True):
        botpackages.widget.reject()


def test_output_wrong(botpackages):  # pylint: disable=redefined-outer-name
    botpackages.widget.setup(WorkerMock(), [], None)
    assert not botpackages.widget.button_ok.isEnabled()
    assert botpackages.widget.button_cancel.isEnabled()
    with botpackages.qtbot.waitSignal(botpackages.widget.rejected, timeout=5000, raising=True):
        botpackages.widget.reject()
