# -*- coding: utf-8 -*-

"""
These tests are about testing the actual application behavior as close to the user experience as we can get.

Such tests may and should look like an actual usage scenarios. E.g. open Navigator, press on the "connect" button, press
on the "sign in" button for the Anaconda.org account, wait for the dialog, enter username, etc.

The main thing about these tests - as we should not only replicate user scenarios for the testing, but also users
behavior. E.g. users don't usually wait for a Qt signal, but they rather wait for a button to become active, or for a
status bar to fill, item to appear in a list, etc.

Where to put tests
==================

All tests should be created in :mod:`~tests.end_to_end_tests`. You may use packages or modules to group tests in a
structure that should be easy to navigate.

Each test should have a descriptive name for easy understanding which scenario this test implements.

Advices
=======

Interactions with the Application
---------------------------------

As already mentioned, end-to-end tests should be pretty similar to user scenarios.

If in any other test you wait for a signal to be raised
(e.g. :attr:`~anaconda_navigator.widgets.main_window.MainWindow.sig_ready` to make sure main window is available), in
end-to-end tests you want to wait for a widgets to be in expected state (MainWindow is visible).

For all such interactions you may use a :code:`qt_helper` fixture. You can find more details about available functions
in :code:`~tests.utilities.qt_utils.qt_helper.QtHelper` documentation.

Going back to the MainWindow example, with :code:`qt_helper` this would look like:

.. code-block:: python

    main_window = qt_helper.wait_for_one(
        qt_utils.OfType('MainWindow') & qt_utils.IsVisible(),
        timeout=300_000,
    )

Such logic should be applied to any user interactions. If you want to wait while Navigator finishes heavy operations
(e.g. modification of some environment), you may want to wait for corresponding progress bar to appear and then
disappear:

.. code-block:: python

    progress_bar = qt_helper.wait_for_one(
        qt_utils.OfType('QProgressBar') & qt_utils.IsVisible(),
        timeout=30_000,
    )
    qt_helper.wait_for_none(
        progress_bar,
        qt_utils.IsVisible(),
        max_depth=1,
        timeout=1_200_000,
        wait_after=1_000,
    )

.. note::

    In this case, :code:`wait_after` makes sure that progress bar won't disappear for a glimpse between multiple
    actions.

.. note::

    In a lot of cases, you may want to wait for a startup operations (if any) right after main window is shown:

    .. code-block:: python

        try:
            progress_bar = qt_helper.wait_for_one(
                qt_utils.OfType('QProgressBar') & qt_utils.IsVisible(),
                timeout=30_000,
            )
        except TimeoutError:
            pass
        else:
            qt_helper.wait_for_none(
                progress_bar,
                qt_utils.IsVisible(),
                max_depth=1,
                timeout=1_200_000,
                wait_after=1_000,
            )

You may also want to wait for some controls to become enabled:

.. code-block:: python

    checkboxes = qt_helper.find_many(
        create_environment_dialog,
        qt_utils.OfType('CheckBoxBase'),  # type: ignore
    )
    qt_helper.wait_for_many(
        checkboxes,
        qt_utils.IsEnabled(),
        at_least=len(checkboxes),
        max_depth=1,
        timeout=60_000,
    )

If you want to interact with controls, you may call corresponding methods directly on them:

.. code-block:: python

    environment_name_entry.setText(environment_name)
    qt_helper.sleep()

    create_environment_button.click()
    qt_helper.sleep()

.. note::

    It is suggested to call :meth:`~tests.utilities.qt_utils.qt_helper.QtHelper.sleep` after any interaction.

Operation timeouts
------------------

Each waiting operation has its own timeout. Depending on a type of operation, you may want to wait for a different
amount of time. Below is a table with common suggestions:

+----------------------------------------------------+-----------+
| What to wait for                                   | Timeout   |
+====================================================+===========+
| Main window to appear                              | 300_000   |
+----------------------------------------------------+-----------+
| Dialog to appear                                   | 30_000    |
+----------------------------------------------------+-----------+
| Controls to be available (no background operation) | 10_000    |
+----------------------------------------------------+-----------+
| Light background operation (conda)                 | 60_000    |
+----------------------------------------------------+-----------+
| Heavy background operation (conda)                 | 1_200_000 |
+----------------------------------------------------+-----------+

Working with conda environments
-------------------------------

After all interactions with Navigator are done, it is also strongly suggested to check the environment. If you installed
some packages in the environment - make sure these packages are listed in the environment after test is finished using
:code:`conda` or any other applicable tool.

.. note::

    It is also suggested to create separate conda environments for each tests where it is applicable. And to remove any
    new environment after the test is done.

    Conda environments are still located outside testing sandbox, which may mean shared context that should be avoided.
    And it is just a nice thing to keep a development environment clean.
"""
