# -*- coding: utf-8 -*-
# pylint: disable=import-outside-toplevel

"""Tests for repodata parsing component."""

from __future__ import annotations

__all__ = ()

import itertools

import functools

import os
import pathlib
import typing

from tests.utilities import parameterization_utils


DATA_ROOT: typing.Final[str] = str(pathlib.Path(__file__).absolute().parent / 'data')


def normalize_files(
        value: typing.Iterable[typing.Iterable[str]],
        *,
        root: str = '',
) -> typing.Set[typing.Tuple[str, ...]]:
    """Normalize repodata file groups."""
    return {
        tuple(os.path.abspath(os.path.join(root, item)) for item in group)
        for group in value
    }


@parameterization_utils.cases('name_repodata')
def test_name(url: str, expectation: str) -> None:
    """
    Check repodata name detection.

    Target: :meth:`~anaconda_navigator.api.conda_api.repodata.RepoDataCache.name`.
    """
    from anaconda_navigator.api.conda_api.repodata import REPO_CACHE

    assert REPO_CACHE.name(url) == expectation


@parameterization_utils.cases('find_repodata')
def test_find_repodata(
        root: str,
        directories: typing.Union[str, typing.Iterable[str]],
        channels: typing.Union[None, str, typing.Iterable[str]],
        expectation: typing.List[typing.List[str]],
) -> None:
    """
    Check repodata files detection.

    Target: :meth:`~anaconda_navigator.api.conda_api.repodata.RepoDataCache._find_repodata`.
    """
    from anaconda_navigator.api.conda_api.repodata import REPO_CACHE

    root = os.path.join(DATA_ROOT, root)

    if isinstance(directories, str):
        directories = os.path.join(root, directories)
    else:
        directories = [os.path.join(root, item) for item in directories]

    # pylint: disable=protected-access
    output: typing.List[typing.List[str]] = REPO_CACHE._find_repodata(directories=directories, channels=channels)
    assert normalize_files(output) == normalize_files(expectation, root=root)


@parameterization_utils.cases('load_repodata')
def test_load_repodata(
        root: str,
        directories: typing.Union[str, typing.Iterable[str]],
        channels: typing.Union[None, str, typing.Iterable[str]],
        expectation: typing.Mapping[str, typing.Any],
) -> None:
    """
    Check repodata files parsing.

    Target: :meth:`~anaconda_navigator.api.conda_api.repodata.RepoDataCache.collect`.
    """
    from anaconda_navigator.api.conda_api.repodata import REPO_CACHE

    root = os.path.join(DATA_ROOT, root)

    if isinstance(directories, str):
        directories = os.path.join(root, directories)
    else:
        directories = [os.path.join(root, item) for item in directories]

    output: typing.Mapping[str, typing.Any] = REPO_CACHE.collect(directories=directories, channels=channels)
    assert output == expectation


@parameterization_utils.cases('repodata_modification_time')
def test_modification_time(
        root: str,
        directories: typing.Union[str, typing.Iterable[str]],
        channels: typing.Union[None, str, typing.Iterable[str]],
) -> None:
    """
    Check detection of the repodata modification time.

    Target: :meth:`~anaconda_navigator.api.conda_api.repodata.RepoDataCache.modification_time`.
    """
    from anaconda_navigator.api.conda_api.repodata import REPO_CACHE

    root = os.path.join(DATA_ROOT, root)

    if isinstance(directories, str):
        directories = os.path.join(root, directories)
    else:
        directories = [os.path.join(root, item) for item in directories]

    # pylint: disable=protected-access
    files: typing.List[typing.List[str]] = REPO_CACHE._find_repodata(directories=directories, channels=channels)
    output: float = REPO_CACHE.modification_time(directories=directories, channels=channels)
    expectation: float = functools.reduce(max, map(os.path.getmtime, itertools.chain.from_iterable(files)), 0.0)
    assert output == expectation
