# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Mocks for CloudAPI."""

from __future__ import annotations

__all__ = ['mocked_cloud_api']

import collections
import datetime
import operator
import typing
import uuid

import attr
import pytest
from qtpy import QtCore

from tests.utilities import requests_utils

if typing.TYPE_CHECKING:
    import pytest_mock

    from anaconda_navigator.api.cloud import types as cloud_types


@attr.s(auto_attribs=True, eq=False, slots=True)
class Control:  # pylint: disable=too-few-public-methods
    """Configuration for a mocked :class:`~anaconda_navigator.api.cloud.api._CloudAPI` instance."""

    environment_limit: int = 1000
    online: bool = True
    valid: bool = True


class EnvironmentRecord(typing.NamedTuple):
    """Single record about backed up environment in :class:`~Storage`."""

    record: 'cloud_types.CloudEnvironmentRecord'
    content: str


class Storage:
    """Collection of all data stored in the mocked :class:`~anaconda_navigator.api.cloud.api._CloudAPI` instance."""

    def __init__(self, control: Control) -> None:
        """Initialize new :class:`~Storage` instance."""
        self.__environments: typing.Final[typing.DefaultDict[str, typing.Dict[str, EnvironmentRecord]]] = (
            collections.defaultdict(dict)
        )
        self.__control: typing.Final[Control] = control

    def __push(self, username: str, record: EnvironmentRecord) -> None:
        """Push new environment in the storage."""
        if record.record['name'] in self.__environments[username]:
            requests_utils.raise_http(
                status_code=409,
                content={
                    'error': {
                        'code': 'environment_already_exists',
                        'data': None,
                        'message': '$environment_already_exists$',
                    },
                },
            )

        if len(self.__environments[username]) > self.__control.environment_limit:
            requests_utils.raise_http(
                status_code=409,
                content={
                    'error': {
                        'code': 'environment_max_count_reached',
                        'data': None,
                        'message': '$environment_max_count_reached$',
                    },
                },
            )

        self.__environments[username][record.record['name']] = record

    def create(self, username: str, name: str, content: str) -> None:
        """Create new environment."""
        self.__push(
            username=username,
            record=EnvironmentRecord(
                record={
                    'id': str(uuid.uuid4()),
                    'name': name,
                    'yaml_ref': f'{name}.yml',
                    'revision': 1,
                    'created_at': datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.%f+00:00'),
                    'updated_at': datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.%f+00:00'),
                },
                content=content,
            )
        )

    def delete(self, username: str, name: str) -> None:
        """Delete existing environment."""
        try:
            del self.__environments[username][name]
        except KeyError:
            requests_utils.raise_http(status_code=404)

    def download(self, username: str, name: str) -> str:
        """Download existing environment."""
        try:
            return self.__environments[username][name].content
        except KeyError:
            requests_utils.raise_http(status_code=404)

    def list(
            self,
            username: str,
            *,
            limit: int = 100,
            offset: int = 0,
            sort: str = 'name',
    ) -> 'cloud_types.CloudEnvironmentCollection':
        """List environment of a user."""
        reverse: bool = False
        if sort.startswith('-'):
            reverse = True
            sort = sort[1:]

        items: typing.List['cloud_types.CloudEnvironmentRecord'] = sorted(
            map(operator.attrgetter('record'), self.__environments[username].values()),
            key=operator.itemgetter(sort),
            reverse=reverse,
        )

        return {
            'items': items[offset:offset + limit],
            'total': len(items),
        }

    def update(self, username: str, old_name: str, new_name: str, content: str) -> None:
        """Update existing environment with a new data."""
        record: EnvironmentRecord
        try:
            record = self.__environments[username].pop(old_name)
        except KeyError:
            self.create(username=username, name=new_name, content=content)
        else:
            record.record['name'] = new_name
            record.record['yaml_ref'] = f'{new_name}.yml'
            record.record['revision'] += 1
            record.record['updated_at'] = datetime.datetime.utcnow().strftime('%Y-%m-%dT%H:%M:%S.%f+00:00')
            self.__push(username=username, record=EnvironmentRecord(record=record.record, content=content))


@pytest.fixture(scope='function')
def mocked_cloud_api(  # pylint: disable=too-many-statements
        mocker: 'pytest_mock.MockerFixture',
) -> typing.Iterator[Control]:
    """Replace :class:`~anaconda_navigator.api.cloud.api._CloudAPI` with an isolated copy of it."""
    from anaconda_navigator.api.cloud import api as cloud_api
    from anaconda_navigator.utils import workers

    control: Control = Control()
    storage: Storage = Storage(control=control)

    class _CloudAPI(QtCore.QObject):
        """Anaconda Cloud API."""

        sig_token_changed = QtCore.Signal()

        def __init__(self) -> None:
            """Initialize new :class:`~CloudAPI` instance."""
            super().__init__()
            self.__token: typing.Optional[str] = None

        @property
        def username(self):
            """Return username if token is present."""
            return 'username' if self.__token else None

        def __get_username(self) -> str:
            """Retrieve username for current session."""
            if not control.online:
                requests_utils.raise_connection()

            if not control.valid:
                requests_utils.raise_http(status_code=422)

            username: typing.Optional[str] = self.username
            if username is None:
                requests_utils.raise_http(status_code=401)

            return username

        def __update_token(self, token: typing.Optional[str] = None) -> None:
            """Update value of the token."""
            self.__token = token
            self.sig_token_changed.emit()

        @property
        def token(self) -> str:  # noqa: D401
            """Current authorization token."""
            return self.__token or ''

        @workers.Task
        def login(self) -> None:  # pylint: disable=unused-argument
            """Perform a basic authorization request using `username` and `password`."""
            if not control.online:
                requests_utils.raise_connection()
            if not control.valid:
                requests_utils.raise_http(status_code=401)

            self.__update_token('ACCESS_TOKEN')

        @workers.Task
        def ping(self) -> bool:
            """Check if API server is available."""
            return control.online

        @workers.Task
        def logout(self) -> bool:
            """Remove token data and clear session."""
            self.__update_token()
            return True

        @workers.Task
        def list_environments(
                self,
                limit: int = 100,
                offset: int = 0,
                sort: str = cloud_api.EnvironmentSort.NAME_ASC,
        ) -> 'cloud_types.CloudEnvironmentCollection':
            """List available environments for current user."""
            if not control.online:
                requests_utils.raise_connection()

            username: typing.Optional[str] = self.username
            if username is None:
                requests_utils.raise_http(status_code=401)

            return storage.list(username=username, limit=limit, offset=offset, sort=sort)

        @workers.Task
        def create_environment(self, name: str, path: str) -> None:
            """Create a new environment in Cloud."""
            username: str = self.__get_username()

            stream: typing.TextIO
            with open(path, 'rt', encoding='utf-8') as stream:
                content: str = stream.read()

            storage.create(username=username, name=name, content=content)

        @workers.Task
        def update_environment(self, name: str, path: str, rename_to: typing.Optional[str] = None) -> None:
            """Update environment in Cloud."""
            if rename_to is None:
                rename_to = name

            username: str = self.__get_username()

            stream: typing.TextIO
            with open(path, 'rt', encoding='utf-8') as stream:
                content: str = stream.read()

            storage.update(username=username, old_name=name, new_name=rename_to, content=content)

        @workers.Task
        def delete_environment(self, name: str) -> None:
            """Remove environment from Cloud."""
            username: str = self.__get_username()
            storage.delete(username=username, name=name)

        @workers.Task
        def download_environment(self, name: str, path: str) -> None:
            """Download environment description file from the Cloud."""
            username: str = self.__get_username()
            content: str = storage.download(username=username, name=name)

            stream: typing.TextIO
            with open(path, 'wt', encoding='utf-8') as stream:
                stream.write(content)

    mocker.patch('anaconda_navigator.api.cloud.api._CloudAPI', _CloudAPI)

    yield control
