# -*- coding: utf-8 -*-

"""Predicates for checking types of objects."""

from __future__ import annotations

__all__ = ['OfType']

import typing

from . import core


class OfType(core.GenericPredicate[typing.Any]):
    """
    Predicate that checks types of values.

    :param args: Options to filter target values by. Might be exact types, as well as their names (absolute as well as
                 only class names). Examples:

                 - :code:`QPushButton`  # class
                 - :code:`'QPushButton'`  # str
                 - :code:`'qtpy.QtWidgets.QPushButton'`  # str

    :param allow_exact: Control must exactly be of one of provided types. If control will be of a type, inherited from
                        the `args` - it won't pass the check.

    :param allow_parents: Allow any of parents (but not the actual type) be one of the `args`.

    If both `allow_exact` and `allow_parents` are not provided (or provided at the same time) - both actual and parent
    types of the control are checked against `args`.
    """

    __slots__ = ('__allow_exact', '__allow_parents', '__names', '__types')

    def __init__(self, *args: typing.Union[str, type], allow_exact: bool = False, allow_parents: bool = False) -> None:
        """Initialize new :class:`~OfType` instance."""
        names: typing.Set[str] = set()
        types: typing.Set[type] = set()

        arg: typing.Union[str, type]
        for arg in args:
            if isinstance(arg, type):
                types.add(arg)
            else:
                names.add(arg)

        if not any((allow_exact, allow_parents)):
            allow_exact = True
            allow_parents = True

        self.__allow_exact: typing.Final[bool] = allow_exact
        self.__allow_parents: typing.Final[bool] = allow_parents
        self.__names: typing.Final[typing.AbstractSet[str]] = names
        self.__types: typing.Final[typing.AbstractSet[type]] = types

    @property
    def allow_exact(self) -> bool:  # noqa: D401:
        """Search for the exact types."""
        return self.__allow_exact

    @property
    def allow_parents(self) -> bool:  # noqa: D401:
        """Search for items, which parent classes are of applicable types."""
        return self.__allow_parents

    @property
    def names(self) -> typing.AbstractSet[str]:  # noqa: D401:
        """Names of types to search for."""
        return self.__names

    @property
    def types(self) -> typing.AbstractSet[type]:  # noqa: D401:
        """Exact types to search for."""
        return self.__types

    def __call__(self, instance: typing.Any) -> bool:
        """Check if `instance` meets a condition."""
        to_check: typing.Set[type] = set()

        exact_type: type = type(instance)
        if self.allow_parents:
            to_check.update(exact_type.__mro__)
        if self.allow_exact:
            to_check.add(exact_type)
        else:
            to_check.discard(exact_type)

        if not to_check.isdisjoint(self.types):
            return True

        return not self.__names.isdisjoint(
            name
            for current_type in to_check
            for name in (
                current_type.__name__,
                current_type.__qualname__,
                f'{current_type.__module__}.{current_type.__qualname__}',
            )
        )
