#!/bin/bash
set -ex

cd ${SRC_DIR}
# function for facilitate version comparison; cf. https://stackoverflow.com/a/37939589
function version2int { echo "$@" | awk -F. '{ printf("%d%02d\n", $1, $2); }'; }

declare -a EXTRA_CMAKE_ARGS
# Cross builds (linux-aarch64, linux-ppc64le, linux-s390x, osx-arm64) link
# against OpenBLAS instead of MKL. Force FindBLAS to use OpenBLAS so it does
# not attempt to run detection binaries on the build host (which fails when
# cross compiling). Also hand CMake the actual library path so it does not try
# to execute detection binaries under emulation.
if [[ "${target_platform}" == linux-aarch64 || "${target_platform}" == osx-arm64 ]]; then
    EXTRA_CMAKE_ARGS+=(-DBLA_VENDOR=OpenBLAS)
    if [[ "${target_platform}" == osx-arm64 ]]; then
        EXTRA_CMAKE_ARGS+=(
            -DBLAS_LIBRARIES="$PREFIX/lib/libopenblas.dylib"
            -DLAPACK_LIBRARIES="$PREFIX/lib/libopenblas.dylib"
        )
    else
        EXTRA_CMAKE_ARGS+=(
            -DBLAS_LIBRARIES="$PREFIX/lib/libopenblas.so"
            -DLAPACK_LIBRARIES="$PREFIX/lib/libopenblas.so"
        )
    fi
fi

declare -a CUDA_CONFIG_ARGS
if [ ${cuda_compiler_version} != "None" ]; then
    # for documentation see e.g.
    # docs.nvidia.com/cuda/cuda-c-best-practices-guide/index.html#building-for-maximum-compatibility
    # docs.nvidia.com/cuda/cuda-compiler-driver-nvcc/index.html#ptxas-options-gpu-name
    # docs.nvidia.com/cuda/cuda-compiler-driver-nvcc/index.html#gpu-feature-list

    ARCHES=(53 62 72)
    if [ $(version2int $cuda_compiler_version) -ge $(version2int "11.1") ]; then
        # Ampere support for GeForce 30 (sm_86) needs cuda >= 11.1
        LATEST_ARCH=86
        # ARCHES does not contain LATEST_ARCH; see usage below
        ARCHES=( "${ARCHES[@]}" 75 80 )
    elif [ $(version2int $cuda_compiler_version) -ge $(version2int "11.0") ]; then
        # Ampere support for A100 (sm_80) needs cuda >= 11.0
        LATEST_ARCH=80
        ARCHES=( "${ARCHES[@]}" 75 )
    fi
    for arch in "${ARCHES[@]}"; do
        CMAKE_CUDA_ARCHS="${CMAKE_CUDA_ARCHS+${CMAKE_CUDA_ARCHS};}${arch}-real"
    done
    # for -real vs. -virtual, see cmake.org/cmake/help/latest/prop_tgt/CUDA_ARCHITECTURES.html
    # this is to support PTX JIT compilation; see first link above or cf.
    # devblogs.nvidia.com/cuda-pro-tip-understand-fat-binaries-jit-caching
    CMAKE_CUDA_ARCHS="${CMAKE_CUDA_ARCHS+${CMAKE_CUDA_ARCHS};}${LATEST_ARCH}"

    FAISS_ENABLE_GPU="ON"
    CUDA_CONFIG_ARGS+=(
        -DCMAKE_CUDA_ARCHITECTURES="${CMAKE_CUDA_ARCHS}"
    )
    # cmake does not generate output for the call below; echo some info
    echo "Set up extra cmake-args: CUDA_CONFIG_ARGS=${CUDA_CONFIG_ARGS+"${CUDA_CONFIG_ARGS[@]}"}"
else
    FAISS_ENABLE_GPU="OFF"
fi

# Disable BUILD_TESTING to skip perf_tests which require gflags (v1.12.0+)
# Tests are run separately via conda build's test phase
BUILD_TESTING="OFF"

# Build version depending on $CF_FAISS_BUILD (either "generic" or "avx2")
cmake -G Ninja \
    ${CMAKE_ARGS} \
    ${EXTRA_CMAKE_ARGS+"${EXTRA_CMAKE_ARGS[@]}"} \
    -DBUILD_SHARED_LIBS=ON \
    -DBUILD_TESTING=${BUILD_TESTING} \
    -DFAISS_OPT_LEVEL=${CF_FAISS_BUILD} \
    -DFAISS_ENABLE_PYTHON=OFF \
    -DFAISS_ENABLE_GPU=${FAISS_ENABLE_GPU} \
    -DFAISS_ENABLE_EXTRAS=OFF \
    -DCMAKE_BUILD_TYPE=Release \
    -DCMAKE_INSTALL_LIBDIR=lib \
    ${CUDA_CONFIG_ARGS+"${CUDA_CONFIG_ARGS[@]}"} \
    -B _build_${CF_FAISS_BUILD} \
    .

if [[ $CF_FAISS_BUILD == avx2 ]]; then
    TARGET="faiss_avx2"
else
    TARGET="faiss"
fi

cmake --build _build_${CF_FAISS_BUILD} --target ${TARGET} -j $CPU_COUNT
cmake --install _build_${CF_FAISS_BUILD} --prefix $PREFIX
cmake --install _build_${CF_FAISS_BUILD} --prefix _libfaiss_${CF_FAISS_BUILD}_stage/
