#!/usr/bin/env python
"""The command-line interface for installer-attribution-bootstrap."""

import argparse
import logging
import sys
from pathlib import Path

from .installer_attribution_core import (
    read_installer_attribution,
    save_installer_attribution,
    send_install_activation_heartbeat,
)

logger = logging.getLogger(__name__)


def setup_logging(verbose: bool = False) -> None:
    level = logging.DEBUG if verbose else logging.WARNING
    logging.basicConfig(
        level=level, format="%(asctime)s - %(name)s - %(levelname)s - %(message)s"
    )


def main() -> int:
    parser = argparse.ArgumentParser(
        prog="installer-attribution-bootstrap",
        description="Read attribution data from Anaconda installer and save to file",
    )

    parser.add_argument("--version", action="version", version="%(prog)s 1.0.0")
    parser.add_argument(
        "-v", "--verbose", action="store_true", help="enable verbose logging"
    )
    parser.add_argument(
        "--installer-file",
        required=True,
        help="path to the installer file to read attribution data from",
    )
    parser.add_argument(
        "--attribution-out-file",
        help="path where to save the attribution configuration data",
    )
    parser.add_argument(
        "--itoken-out-file",
        help="path where to save the itoken",
    )
    parser.add_argument(
        "--platform",
        choices=["windows", "darwin", "linux"],
        help="override platform detection (e.g., read Windows installer on macOS)",
    )
    parser.add_argument(
        "--dry-run",
        action="store_true",
        help="print where itoken would be saved without writing",
    )

    args = parser.parse_args()

    setup_logging(args.verbose)

    input_path = Path(args.installer_file)
    if not input_path.exists():
        logger.error(f"Error: Input file '{args.installer_file}' does not exist")
        return 1

    try:
        logger.info(f"Reading attribution data from: {args.installer_file}")
        attribution_data = read_installer_attribution(
            args.installer_file, args.platform
        )

        if attribution_data is None:
            logger.error(f"No attribution data found in {args.installer_file}")
            return 1

        success = save_installer_attribution(
            attribution_data, args.itoken_out_file, args.dry_run
        )

        send_install_activation_heartbeat(attribution_data)

        if success:
            logger.info(f"Attribution data successfully written.")
            return 0
        else:
            logger.error(f"Failed to write attribution data.")
            return 1

    except Exception as e:
        logger.error(f"Error processing attribution data: {e}")
        logger.exception("Unexpected error occurred")
        return 1


if __name__ == "__main__":
    sys.exit(main())
