import logging
import os
import platform
import sys
import time
from pathlib import Path
from typing import Optional

from anaconda_anon_usage import tokens
from conda.base.context import context
from conda.gateways.connection.session import CondaSession

from . import installer_attribution_io

logger = logging.getLogger(__name__)

ITOKEN_FILE_NAME = ".installer_token"

# We could call the script post_install like this:
# start /B "%PREFIX%\python.exe" -m installer-attribution-bootstrap --installer-file "%EXEPATH%"
# and start a background task for this bootsrap process.
# How long to attempt the connection. When a connection to our
# activation heartbeat endpoint is blocked or slow, a long timeout would lead to
# a slow activation and a poor user experience. This is a total
# timeout value, inclusive of all retries.
TIMEOUT = 30  # seconds (we can run this as a background task)
ATTEMPTS = 3


def read_installer_attribution(
    filepath: Optional[str] = None, platform_name: Optional[str] = None
) -> Optional[str]:
    """
    Read attribution data from an installer file.

    Args:
        filepath: Path to the installer file.
        platform_name: Override platform detection. Options: 'windows', 'darwin', 'linux'

    Returns:
        Attribution data as string, or None if not found.
    """
    if filepath is None:
        return None
    if not os.path.exists(filepath):
        return None
    try:
        system_platform = platform_name if platform_name else platform.system().lower()
        installer_attribution_reader = (
            installer_attribution_io.INSTALLER_ATTRIBUTION_READERS[system_platform]
        )
        attribution_data = installer_attribution_reader(filepath)
        if attribution_data is None:
            logger.info(f"No attribution data found in {filepath}")
            return None
        return attribution_data
    except Exception as e:
        logger.error(f"Error reading attribution data: {e}")
        return None


def save_installer_attribution(
    attribution_data: str,
    itoken_out_file: Optional[str] = None,
    dry_run: bool = False,
) -> bool:
    """
    Save installer attribution data to local .installer_token file.
    This saves the itoken to a location where anaconda-anon-usage will
    automatically pick it up via its installer_tokens() function, which
    searches directories where .condarc files can be found.
    Args:
        attribution_data: The installer attribution data to save.
        itoken_out_file: The path to save the .installer_token file.
    Returns:
        True if successful, False otherwise
    """
    parsed_attribution_data = installer_attribution_io.parse_installer_attribution(
        attribution_data
    )
    itoken = parsed_attribution_data.get("itoken")
    if not itoken:
        logger.error("No itoken found in attribution data")
        return False

    # Determine output path(s)
    if itoken_out_file is not None:
        # User specified explicit path
        path = Path(itoken_out_file)
    else:
        path = Path(sys.prefix) / ITOKEN_FILE_NAME

    if dry_run:
        logger.info(f"[DRY RUN] Would save installer token to: {path}")
        success = True
    else:
        success = installer_attribution_io.write_installer_attribution_to_file(
            itoken, str(path)
        )

    if success:
        logger.info(f"Installer token saved to: {path}")
        return True
    return False


def send_install_activation_heartbeat(attribution_data: str) -> bool:
    """
    Send installation activation heartbeat to the attribution service.

    Args:
        attribution_data: Raw attribution data string from installer

    Returns:
        True if activation was sent successfully, False otherwise
    """
    try:
        parsed_attribution_data = installer_attribution_io.parse_installer_attribution(
            attribution_data
        )
        itoken = parsed_attribution_data.get("itoken")

        if not itoken:
            logger.info("No itoken found in attribution data")
            return False

    except Exception as e:
        logger.info(f"Failed to parse attribution data: {e}")
        return False

    try:
        anaconda_anon_usage_tokens = tokens.all_tokens()._asdict()
    except Exception as e:
        logger.info(f"Failed to get anaconda-anon-usage tokens: {e}")
        anaconda_anon_usage_tokens = {}

    system_info = collect_system_info()
    payload = {
        "anaconda_anon_usage_tokens": anaconda_anon_usage_tokens,
        "itoken": itoken,
        "system_info": system_info,
        "activated_at": time.strftime("%Y-%m-%dT%H:%M:%SZ", time.gmtime()),
    }

    # Configure session with retries and timeout
    timeout = TIMEOUT / ATTEMPTS
    context.remote_max_retries = ATTEMPTS - 1
    context.remote_backoff_factor = 0  # No backoff between attempts

    url = parsed_attribution_data.get("activation_endpoint")
    session = CondaSession()

    try:
        start_time = time.perf_counter()

        # Send POST request with activation data
        response = session.post(
            url,
            json=payload,
            proxies=session.proxies,
            timeout=timeout,
            headers={"Content-Type": "application/json"},
        )

        delta = time.perf_counter() - start_time

        # Check response status
        if response.status_code == 200:
            logger.info(
                "Activation sent successfully after %.3fs; status: %d",
                delta,
                response.status_code,
            )

            # Log response details if available
            try:
                response_data = response.json()
                if response_data.get("success"):
                    logger.info(
                        "Activation confirmed: %s", response_data.get("message")
                    )
            except Exception:
                pass  # Response might not be JSON

            return True
        else:
            logger.info(
                "Activation request failed; status: %d after %.3fs",
                response.status_code,
                delta,
            )
            return False

    except Exception as exc:
        delta = time.perf_counter() - start_time

        # Handle specific exception types
        if type(exc).__name__ == "ConnectionError":
            if "timeout=" in str(exc):
                logger.info("Activation timed out after %.3fs", delta)
            else:
                logger.info("Connection error during activation after %.3fs", delta)
        else:
            logger.info("Unexpected activation error after %.3fs: %s", delta, exc)

        return False


def collect_system_info() -> dict:
    """
    Collect system information for download activation.
    """
    system_info = {
        "platform": context.platform,
        "arch_name": context.arch_name,
        "platform_system_release": context.platform_system_release,
        "python_implementation_name_version": context.python_implementation_name_version,
        "conda_user_agent": context.user_agent,
    }
    return system_info
