# Copyright (c) "Neo4j"
# Neo4j Sweden AB [https://neo4j.com]
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from contextlib import redirect_stdout
from io import StringIO

import pytest

from neo4j._exceptions import BoltHandshakeError


# isort: off
# tag::driver-introduction-example-import[]
import logging
import sys

from neo4j import GraphDatabase
from neo4j.exceptions import ServiceUnavailable
# end::driver-introduction-example-import[]
# isort: on


# tag::driver-introduction-example[]
class App:
    def __init__(self, uri, user, password):
        self.driver = GraphDatabase.driver(uri, auth=(user, password))

    def close(self):
        # Don't forget to close the driver connection when you are finished
        # with it
        self.driver.close()

    @staticmethod
    def enable_log(level, output_stream):
        handler = logging.StreamHandler(output_stream)
        handler.setLevel(level)
        logging.getLogger("neo4j").addHandler(handler)
        logging.getLogger("neo4j").setLevel(level)

    def create_friendship(self, person1_name, person2_name, knows_from):
        with self.driver.session() as session:
            # Write transactions allow the driver to handle retries and
            # transient errors
            result = session.execute_write(
                self._create_and_return_friendship,
                person1_name,
                person2_name,
                knows_from,
            )
            for row in result:
                print(
                    "Created friendship between: ;"
                    f"{row['p1']}, {row['p2']} from {row['knows_from']}"
                )

    @staticmethod
    def _create_and_return_friendship(
        tx, person1_name, person2_name, knows_from
    ):
        # To learn more about the Cypher syntax,
        # see https://neo4j.com/docs/cypher-manual/current/
        # The Reference Card is also a good resource for keywords
        # https://neo4j.com/docs/cypher-refcard/current/
        query = (
            "CREATE (p1:Person { name: $person1_name }) "
            "CREATE (p2:Person { name: $person2_name }) "
            "CREATE (p1)-[k:KNOWS { from: $knows_from }]->(p2) "
            "RETURN p1, p2, k"
        )
        result = tx.run(
            query,
            person1_name=person1_name,
            person2_name=person2_name,
            knows_from=knows_from,
        )
        try:
            return [
                {
                    "p1": row["p1"]["name"],
                    "p2": row["p2"]["name"],
                    "knows_from": row["k"]["from"],
                }
                for row in result
            ]
        # Capture any errors along with the query and data for traceability
        except ServiceUnavailable:
            logging.exception("%s raised an error", query)
            raise

    def find_person(self, person_name):
        with self.driver.session() as session:
            result = session.execute_read(
                self._find_and_return_person, person_name
            )
            for row in result:
                print(f"Found person: {row}")

    @staticmethod
    def _find_and_return_person(tx, person_name):
        query = (
            "MATCH (p:Person) "
            "WHERE p.name = $person_name "
            "RETURN p.name AS name"
        )
        result = tx.run(query, person_name=person_name)
        return [row["name"] for row in result]


if __name__ == "__main__":
    bolt_url = "%%BOLT_URL_PLACEHOLDER%%"
    user = "<Username for database>"
    password = "<Password for database>"
    App.enable_log(logging.INFO, sys.stdout)
    app = App(bolt_url, user, password)
    app.create_friendship("Alice", "David", "School")
    app.find_person("Alice")
    app.close()
# end::driver-introduction-example[]


def test_driver_introduction_example(uri, auth):
    try:
        s = StringIO()
        with redirect_stdout(s):
            App.enable_log(logging.INFO, sys.stdout)
            app = App(uri, auth[0], auth[1])
            app.create_friendship("Alice", "David", "School")
            app.find_person("Alice")
            app.close()

        # assert s.getvalue().startswith("Found person: Alice")
    except ServiceUnavailable as error:
        if isinstance(error.__cause__, BoltHandshakeError):
            pytest.skip(error.args[0])
