# SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: BSD-2-Clause

import sys
import os
import multiprocessing as mp
import warnings
import pathlib

from numba.cuda.core.config import IS_WIN32
from numba.cuda.core.errors import NumbaWarning
from numba.cuda.cudadrv import nvvm
from numba.cuda.testing import (
    unittest,
    skip_on_cudasim,
    skip_unless_conda_cudatoolkit,
)
from numba.cuda.cuda_paths import (
    _get_libdevice_path_decision,
    _get_cudalib_dir_path_decision,
    get_system_ctk,
    get_system_ctk_libdir,
    _find_cuda_home_from_lib_path,
)


has_cuda = nvvm.is_available()
has_mp_get_context = hasattr(mp, "get_context")


class LibraryLookupBase(unittest.TestCase):
    def setUp(self):
        ctx = mp.get_context("spawn")

        qrecv = ctx.Queue()
        qsend = ctx.Queue()
        self.qsend = qsend
        self.qrecv = qrecv
        self.child_process = ctx.Process(
            target=check_lib_lookup,
            args=(qrecv, qsend),
            daemon=True,
        )
        self.child_process.start()

    def tearDown(self):
        self.qsend.put(self.do_terminate)
        self.child_process.join(3)
        # Ensure the process is terminated
        self.assertIsNotNone(self.child_process)

    def remote_do(self, action):
        self.qsend.put(action)
        out = self.qrecv.get()
        self.assertNotIsInstance(out, BaseException)
        return out

    @staticmethod
    def do_terminate():
        return False, None


def remove_env(name):
    try:
        del os.environ[name]
    except KeyError:
        return False
    else:
        return True


def check_lib_lookup(qout, qin):
    status = True
    while status:
        try:
            action = qin.get()
        except Exception as e:
            qout.put(e)
            status = False
        else:
            try:
                with warnings.catch_warnings(record=True) as w:
                    warnings.simplefilter("always", NumbaWarning)
                    status, result = action()
                qout.put(result + (w,))
            except Exception as e:
                qout.put(e)
                status = False


@skip_on_cudasim("Library detection unsupported in the simulator")
@unittest.skipUnless(has_mp_get_context, "mp.get_context not available")
@skip_unless_conda_cudatoolkit("test assumes conda installed cudatoolkit")
class TestLibDeviceLookUp(LibraryLookupBase):
    def test_libdevice_path_decision(self):
        # Check that the default is using conda environment
        by, info, warns = self.remote_do(self.do_clear_envs)
        if has_cuda:
            self.assertEqual(by, "Conda environment")
        else:
            self.assertEqual(by, "<unknown>")
            self.assertIsNone(info)
        self.assertFalse(warns)
        # Check that CUDA_HOME works by removing conda-env
        by, info, warns = self.remote_do(self.do_set_cuda_home)
        self.assertEqual(by, "CUDA_HOME")
        self.assertTrue(
            info.startswith(os.path.join("mycudahome", "nvvm", "libdevice"))
        )
        self.assertFalse(warns)

        if get_system_ctk("nvvm", "libdevice") is None:
            # Fake remove conda environment so no cudatoolkit is available
            by, info, warns = self.remote_do(self.do_clear_envs)
            self.assertEqual(by, "<unknown>")
            self.assertIsNone(info)
            self.assertFalse(warns)
        else:
            # Use system available cudatoolkit
            by, info, warns = self.remote_do(self.do_clear_envs)
            self.assertEqual(by, "System")
            self.assertFalse(warns)

    @staticmethod
    def do_clear_envs():
        remove_env("CUDA_HOME")
        remove_env("CUDA_PATH")
        return True, _get_libdevice_path_decision()

    @staticmethod
    def do_set_cuda_home():
        os.environ["CUDA_HOME"] = os.path.join("mycudahome")
        _fake_non_conda_env()
        return True, _get_libdevice_path_decision()


@skip_on_cudasim("Library detection unsupported in the simulator")
@unittest.skipUnless(has_mp_get_context, "mp.get_context not available")
@skip_unless_conda_cudatoolkit("test assumes conda installed cudatoolkit")
class TestCudaLibLookUp(LibraryLookupBase):
    def test_cudalib_path_decision(self):
        # Check that the default is using conda environment
        by, info, warns = self.remote_do(self.do_clear_envs)
        if has_cuda:
            self.assertEqual(by, "Conda environment")
        else:
            self.assertEqual(by, "<unknown>")
            self.assertIsNone(info)
        self.assertFalse(warns)

        # Check that CUDA_HOME works by removing conda-env
        self.remote_do(self.do_clear_envs)
        by, info, warns = self.remote_do(self.do_set_cuda_home)
        self.assertEqual(by, "CUDA_HOME")
        self.assertFalse(warns)
        if IS_WIN32:
            # I think only wheels don't have the "Library" directory?
            self.assertTrue(
                info
                in (
                    os.path.join("mycudahome", "bin"),
                    os.path.join("mycudahome", "Library", "bin"),
                )
            )
        else:
            self.assertEqual(info, os.path.join("mycudahome", "lib64"))
        if get_system_ctk_libdir() is None:
            # Fake remove conda environment so no cudatoolkit is available
            by, info, warns = self.remote_do(self.do_clear_envs)
            self.assertEqual(by, "<unknown>")
            self.assertIsNone(info)
            self.assertFalse(warns)
        else:
            # Use system available cudatoolkit
            by, info, warns = self.remote_do(self.do_clear_envs)
            self.assertEqual(by, "System")
            self.assertFalse(warns)

    @staticmethod
    def do_clear_envs():
        remove_env("CUDA_HOME")
        remove_env("CUDA_PATH")
        return True, _get_cudalib_dir_path_decision()

    @staticmethod
    def do_set_cuda_home():
        os.environ["CUDA_HOME"] = os.path.join("mycudahome")
        _fake_non_conda_env()
        return True, _get_cudalib_dir_path_decision()


def _fake_non_conda_env():
    """
    Monkeypatch sys.prefix to hide the fact we are in a conda-env
    """
    sys.prefix = ""


class TestCudaHomeDetection(unittest.TestCase):
    def test_find_cuda_home(self):
        """Test the directory walking logic."""
        import tempfile

        # Create a mock CUDA installation structure
        with tempfile.TemporaryDirectory() as tmpdir:
            cuda_root = pathlib.Path(tmpdir) / "cuda"
            lib64 = cuda_root / "lib64"
            nvvm = cuda_root / "nvvm"
            nvvm_lib64 = nvvm / "lib64"

            lib64.mkdir(parents=True)
            nvvm_lib64.mkdir(parents=True)

            # Create mock library files
            nvrtc_path = lib64 / "libnvrtc.so.12"
            nvrtc_path.touch()

            nvvm_lib = nvvm_lib64 / "libnvvm.so.4"
            nvvm_lib.touch()

            # Test finding CUDA_HOME from nvrtc path
            found_cuda_home = _find_cuda_home_from_lib_path(str(nvrtc_path))

            # Compare resolved paths to handle Windows short path names (e.g., RGROSS~1)
            expected = str(cuda_root.resolve())
            assert found_cuda_home == expected, (
                f"Expected {expected}, got {found_cuda_home}"
            )

            # Test that the nvvm directory exists at the found location
            assert (pathlib.Path(found_cuda_home) / "nvvm").is_dir()


if __name__ == "__main__":
    unittest.main()
