from unittest import mock

import pytest

from snowflake.core import PollingOperation
from snowflake.core.tag import TagValue
from snowflake.core.warehouse import Warehouse, WarehouseCollection, WarehouseResource
from snowflake.core.warehouse._generated import TagAssignment, TagReference

from ...utils import BASE_URL, extra_params, mock_http_response


API_CLIENT_REQUEST = "snowflake.core._generated.api_client.ApiClient.request"


@pytest.fixture
def warehouses(fake_root):
    return WarehouseCollection(fake_root)


@pytest.fixture
def warehouse(warehouses):
    return warehouses["my_wh"]


def test_create_warehouse(fake_root, warehouses):
    args = (fake_root, "POST", BASE_URL + "/warehouses?createMode=errorIfExists")
    kwargs = extra_params(query_params=[("createMode", "errorIfExists")], body={"name": "my_wh"})

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse_res = warehouses.create(Warehouse(name="my_wh"))
        assert isinstance(warehouse_res, WarehouseResource)
        assert warehouse_res.name == "my_wh"
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouses.create_async(Warehouse(name="my_wh"))
        assert isinstance(op, PollingOperation)
        wh_res = op.result()
        assert wh_res.name == "my_wh"
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_iter_warehouse(fake_root, warehouses):
    args = (fake_root, "GET", BASE_URL + "/warehouses")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        mocked_request.return_value = mock_http_response()
        warehouses.iter()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        mocked_request.return_value = mock_http_response()
        op = warehouses.iter_async()
        assert isinstance(op, PollingOperation)
        it = op.result()
        assert list(it) == []
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_create_or_alter_warehouse(fake_root, warehouse):
    args = (fake_root, "PUT", BASE_URL + "/warehouses/my_wh")
    kwargs = extra_params(body={"name": "my_wh"})

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.create_or_alter(Warehouse(name="my_wh"))
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.create_or_alter_async(Warehouse(name="my_wh"))
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_suspend_warehouse(fake_root, warehouse):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:suspend")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.suspend()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.suspend_async()
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_resume_warehouse(fake_root, warehouse):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:resume")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.resume()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.resume_async()
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_fetch_warehouse(fake_root, warehouse):
    from snowflake.core.warehouse._generated.models import Warehouse as WarehouseModel

    model = WarehouseModel(name="my_wh")
    args = (fake_root, "GET", BASE_URL + "/warehouses/my_wh")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        mocked_request.return_value = mock_http_response(model.to_json())
        warehouse.fetch()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        mocked_request.return_value = mock_http_response(model.to_json())
        op = warehouse.fetch_async()
        assert isinstance(op, PollingOperation)
        tab = op.result()
        assert tab.to_dict() == Warehouse(name="my_wh").to_dict()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_drop_warehouse(fake_root, warehouse):
    args = (fake_root, "DELETE", BASE_URL + "/warehouses/my_wh")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.drop()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.drop_async()
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_rename_warehouse(fake_root, warehouse):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:rename")
    kwargs = extra_params(body={"name": "new_wh"})

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.rename("new_wh")
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.rename_async("new_wh")
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_abort_all_queries(fake_root, warehouse):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:abort")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.abort_all_queries()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.abort_all_queries_async()
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_use_warehouse(fake_root, warehouse):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:use")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.use_warehouse()
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.use_warehouse_async()
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_set_tags(fake_root, warehouse, tag):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:set-tags")
    tags = {tag: TagValue(value="value")}
    kwargs = extra_params(
        body=[
            TagAssignment(
                tag_value=v.value, tag_name=k.name, tag_schema=k.schema.name, tag_database=k.database.name
            ).to_dict()
            for k, v in tags.items()
        ]
    )

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.set_tags(tags)
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.set_tags_async(tags)
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_unset_tags(fake_root, warehouse, tag):
    args = (fake_root, "POST", BASE_URL + "/warehouses/my_wh:unset-tags")
    tag_resources = {tag}
    kwargs = extra_params(
        body=[
            TagReference(
                tag_name=tag_res.name, tag_schema=tag_res.schema.name, tag_database=tag_res.database.name
            ).to_dict()
            for tag_res in tag_resources
        ]
    )

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        warehouse.unset_tags(tag_resources)
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        op = warehouse.unset_tags_async(tag_resources)
        assert isinstance(op, PollingOperation)
        op.result()
    mocked_request.assert_called_once_with(*args, **kwargs)


def test_get_tags(fake_root, warehouse):
    args = (fake_root, "GET", BASE_URL + "/warehouses/my_wh:get-tags")
    kwargs = extra_params()

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        mocked_request.return_value = mock_http_response()
        assert warehouse.get_tags() == {}
    mocked_request.assert_called_once_with(*args, **kwargs)

    with mock.patch(API_CLIENT_REQUEST) as mocked_request:
        mocked_request.return_value = mock_http_response()
        op = warehouse.get_tags_async()
        assert isinstance(op, PollingOperation)
        assert op.result() == {}
    mocked_request.assert_called_once_with(*args, **kwargs)
