"""
Manage your Anaconda repository channels.
"""

from __future__ import unicode_literals, print_function

import argparse
from pprint import pformat
from .. import errors
from ..utils.artifacts import SimplePackageSpec
from .base import SubCommandBase
from ..utils.format import MirrorFormatter

class SubCommand(SubCommandBase):
    name = "mirror"

    def main(self):
        self.log.info("")
        args = self.args
        if args.create:
            self.create_mirror(
                args.channel, args.source, args.create, args.mode, args.type, args.licenses,
                args.exclude_spec, args.include_spec, args.cron, args.run_now, args.proxy)
        elif args.delete:
            self.delete(args.channel, args.delete)

        elif args.list:
            self.show_list(args.list)

        elif args.show:
            self.show(args.channel, args.show)

        else:

            raise NotImplementedError()

    def create_mirror(self, channel, source, name, mode, type_, filter_licenses,
                      filter_exclude_specs, filter_include_specs, cron, run_now, proxy):
        filters = {}
        if filter_licenses:
            filters['include_licenses'] = []
            filters['exclude_licenses'] = []
            for license in filter_licenses.split(','):
                group = 'include_licenses'
                if license.startswith('-'):
                    group = 'exclude_licenses'
                    license = license[1:]
                if license.startswith('+'):
                    license = license[1:]
                filters[group].append(license)
        if filter_exclude_specs:
            filters['exclude_specs'] = filter_exclude_specs
        if filter_include_specs:
            filters['include_specs'] = filter_include_specs
        self.api.create_mirror(channel, source, name, mode, filters, type_, cron, run_now, proxy)
        self.log.info('Mirror %s successfully created on channel %s', name, channel)


    def show_list(self, channel):
        data = self.api.get_mirrors(channel)
        self.log.info(MirrorFormatter.format_list(data['items']))
        self.log.info('')

    def show(self, channel, name):
        # TODO: Get mirror api is currently unavailable so we need to use the
        #       GET mirrors api and filter by name...
        # data = self.api.get_mirror(channel, name)

        data = self.api.get_mirrors(channel)
        for mirror in data:
            if mirror['mirror_name'] == name:
                self.log.info(MirrorFormatter.format_detail(mirror))

        self.log.info('')

    def delete(self, channel, name):
        self.api.delete_mirror(channel, name)
        self.log.info('Mirror %s successfully delete on channel %s', name, channel)


    def add_parser(self, subparsers):
        subparser = subparsers.add_parser(
            self.name,
            help='Manage your Anaconda repository {}s'.format(self.name),
            formatter_class=argparse.RawDescriptionHelpFormatter,
            description=__doc__)

        subparser.add_argument('--channel', '-c', help='Channel to mirror to.')
        group = subparser.add_mutually_exclusive_group(required=True)

        # group.add_argument('--create', nargs=2, metavar=self.name.upper())
        group.add_argument(
            '--create',
            metavar=self.name.upper(),
            help="Create a new {}".format(self.name)
        )
        group.add_argument(
            '--delete',
            metavar=self.name.upper(),
            help="Create a new {}".format(self.name)
        )
        group.add_argument(
            '--list',
            '-l',
            metavar=self.name.upper(),
            help="list all {}s for a user".format(self.name)
        )
        group.add_argument(
            '--show',
            metavar=self.name.upper(),
            help="Show all of the files in a {}".format(self.name)
        )

        subparser.add_argument('--source', '-s', help='Path to the source channel to mirror. '
                                                   'I.e.: https://conda.anaconda.org/conda-test')
        subparser.add_argument('--name', '-n', help='Name of the mirror')
        subparser.add_argument(
            '--mode',
            default='active',
            help='Mirror mode. If "active", will download all the files from the source channel '
                 'immediately else, if "passive", download JSON immediately and files on demand '
                 'later'
        )
        subparser.add_argument(
            '--type',
            default='conda',
            help='Mirror type. Possible types: "conda", "python_simple" and "CRAN"'
        )
        subparser.add_argument(
            '--cron',
            default='0 0 * * *',
            help='Cron string to configure the mirror job.'
        )
        subparser.add_argument(
            '--proxy',
            help='Proxy to use for the mirroring in format "http://<PROXYURL>" or "http://<USER>:<PASS>@<PROXYURL>"'
        )

        filters_group = subparser.add_argument_group('mirror filters')
        filters_group.add_argument(
            '--licenses',
            default=None,
            help='List of licenses to filter for. Use comma-separated string, prepent with "-" to exclude, '
                 'or just a license to include. The allowed license values are: agpl, gpl2, gpl3, lgpl, '
                 'bsd, mit, apache, psf, public_domain, proprietary, other, none'
        )
        filters_group.add_argument(
            '--exclude_spec',
            action='append',
            default=None,
            help='MatchSpec to exclude. Use multiple times the option for multiple entries'
        )
        filters_group.add_argument(
            '--include_spec',
            action='append',
            default=None,
            help='MatchSpec to include. Use multiple times the option for multiple entries'
        )

        subparser.add_argument(
            '--run_now',
            action='store_true',
            help='Determines whether the mirror job should run immediately or '
                 'according to the cron schedule'
        )

        subparser.set_defaults(main=self.main)
