# Copyright 2018 The TensorFlow Probability Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ============================================================================
"""Tests for AbsoluteValue Bijector."""

import tensorflow.compat.v2 as tf

from tensorflow_probability.python.bijectors import absolute_value
from tensorflow_probability.python.internal import test_util


@test_util.test_all_tf_execution_regimes
class AbsoluteValueTest(test_util.TestCase):
  """Tests correctness of the absolute value bijector."""

  def testBijectorVersusNumpyRewriteOfBasicFunctionsEventNdims0(self):
    bijector = absolute_value.AbsoluteValue(validate_args=True)
    self.assertStartsWith(bijector.name, "absolute_value")
    x = tf.constant([[0., 1., -1], [0., -5., 3.]])  # Shape [2, 3]
    y = tf.math.abs(x)

    y_ = self.evaluate(y)

    self.assertAllClose(y_, self.evaluate(bijector.forward(x)))
    self.assertAllClose((-y_, y_), self.evaluate(bijector.inverse(y)))
    self.assertAllClose((0., 0.),
                        self.evaluate(bijector.inverse_log_det_jacobian(
                            y, event_ndims=0)))

    # Run things twice to make sure there are no issues in caching the tuples
    # returned by .inverse*
    self.assertAllClose(y_, self.evaluate(bijector.forward(x)))
    self.assertAllClose((-y_, y_), self.evaluate(bijector.inverse(y)))
    self.assertAllClose((0., 0.),
                        self.evaluate(bijector.inverse_log_det_jacobian(
                            y, event_ndims=0)))

  def testNegativeYRaisesForInverseIfValidateArgs(self):
    bijector = absolute_value.AbsoluteValue(validate_args=True)
    with self.assertRaisesOpError("y was negative"):
      self.evaluate(bijector.inverse(-1.))

  def testNegativeYRaisesForILDJIfValidateArgs(self):
    bijector = absolute_value.AbsoluteValue(validate_args=True)
    with self.assertRaisesOpError("y was negative"):
      self.evaluate(bijector.inverse_log_det_jacobian(-1., event_ndims=0))

  def testCompositeTensor(self):
    bijector = absolute_value.AbsoluteValue(validate_args=True)
    flat = tf.nest.flatten(bijector, expand_composites=True)
    unflat = tf.nest.pack_sequence_as(bijector, flat, expand_composites=True)
    x = tf.convert_to_tensor([0., 1., -1.])
    self.assertAllClose(
        bijector.forward(x),
        tf.function(lambda b_: b_.forward(x))(unflat))


if __name__ == "__main__":
  test_util.main()
