import asyncio
import json
import os
import subprocess
import sys
from dataclasses import asdict, dataclass
from datetime import datetime
from multiprocessing.pool import ThreadPool
from typing import Dict, List, Optional, Type
from uuid import uuid4

import httpx
import requests
import tornado
from anaconda_auth.exceptions import TokenNotFoundError
from jupyter_server.base.handlers import APIHandler
from tornado.routing import _RuleList
from tornado.web import HTTPError

from aext_core_server.__about__ import __version__
from aext_core_server.exceptions import UserFeaturesException
from aext_core_server.logger import logger
from aext_core_server.services.auth import auth_service
from aext_core_server.services.user import user_service
from aext_shared.backend_proxy import ProxyResponse
from aext_shared.config import (
    SHARED_CONFIG,
    Environment,
    get_anaconda_org_platform,
    get_environment,
)
from aext_shared.consts import LIST_ORG_USERS_LIMIT, UserAccessCredentials
from aext_shared.errors import BackendError, BadRequest, UnauthorizedError
from aext_shared.handler import BackendHandler, create_rules, should_use_new_oauth

config = SHARED_CONFIG


class ConfigHandler(BackendHandler):
    @tornado.web.authenticated
    async def get(self):
        self.finish(config)


class FeatureFlagInitHandler(BackendHandler):
    @tornado.web.authenticated
    async def get(self):
        feature_flag_context = {}
        try:
            [account, organizations, account_notebooks] = await asyncio.gather(
                self.anaconda_proxy("account"),
                self.anaconda_proxy("organizations/my"),
                self.anaconda_proxy("account/notebooks"),
            )
        except UnauthorizedError as ex:
            logger.debug(f"Unauthorized: Can't get users info {ex}")
            return self.finish(
                ProxyResponse(remote_status_code=httpx.codes.FORBIDDEN, remote_data=feature_flag_context)
            )
        except Exception as ex:
            logger.info(f"Error while fetching users info {ex}")
            return self.finish(
                ProxyResponse(remote_status_code=httpx.codes.BAD_REQUEST, remote_data=feature_flag_context)
            )

        data = account["remote_data"]
        # Keep in sync with passport_to_feature_flag_user in nucleus
        # https://github.com/anaconda/bigbend-platform/blob/main/shared/python/shared/feature_flags.py#L83
        email: str = data["user"].get("email", "")
        is_confirmed: bool = data["profile"]["is_confirmed"]
        is_disabled: bool = data["profile"]["is_disabled"]
        is_anaconda_email = email.lower().endswith("@anaconda.com") and is_confirmed and not is_disabled
        org_ids = [o["id"] for o in organizations["remote_data"]]
        feature_flag_context = {
            "key": f"user:{data['user']['id']}",
            "kind": "user",
            "confirmed": is_confirmed,
            "disabled": is_disabled,
            "is_anaconda_employee": is_anaconda_email,
            "email": data["user"]["email"],
            "username": data["user"]["username"],
            "org_ids": org_ids,
            "notebooks_service_subscription": account_notebooks["remote_data"]["notebooks_service_subscription"],
            "platform": config["platform"],
            "environment": config["environment"],
            "environment_v2": config["environment_v2"],
            "extension_version": config["extension"]["version"],
        }
        self.finish(ProxyResponse(remote_status_code=httpx.codes.OK, remote_data=feature_flag_context))


class ShouldUseNewOAuthHandler(BackendHandler):
    async def post(self, matched_part=None, *args, **kwargs):

        debug_refresh_token = os.environ.get("DEBUG_REFRESH_TOKEN", None)
        # Debug refresh token takes precedence over the refresh token in the request
        refresh_token = debug_refresh_token or self.get_json_body().get("refresh_token")
        if refresh_token is None:
            raise HTTPError(400, "missing refresh_token in request")

        self.finish(json.dumps({"should_use_new_oauth": await should_use_new_oauth(refresh_token)}))


class AnacondaCloudUserOrganizationsHandler(BackendHandler):

    @dataclass
    class UserOrganization:
        id: uuid4
        name: str
        title: str
        created_at: datetime
        members_count: int
        role: str

    @tornado.web.authenticated
    async def get(self):
        try:
            response = await self.anaconda_proxy("organizations/my")
            data = response["remote_data"]
            user_orgs = [
                asdict(
                    self.UserOrganization(
                        id=item["id"],
                        name=item["name"],
                        title=item["title"],
                        created_at=item["created_at"],
                        members_count=item["members_count"],
                        role=item["role"],
                    )
                )
                for item in data
            ]
            self.finish(json.dumps(user_orgs))
        except UnauthorizedError:
            self.set_status(403, "Could not find user credentiansl")
        except BackendError as ex:
            remote_status_code = ex.data.get("remote_status_code")
            if remote_status_code == 403:
                logger.warning("Forbidden: Can't fetch info from Anaconda Cloud")
                status_message = "Forbidden: Check user credentials"
            elif remote_status_code == 404:
                logger.debug("Not found endpoint")
                status_message = "Forbidden: Check user credentials"
            else:
                status_message = "Unknown error"
            self.set_status(remote_status_code, status_message)
        except Exception:
            self.set_status(500, "Unknown error")


class AnacondaCloudOrganizationMembersHandler(BackendHandler):

    @dataclass
    class OrgMember:
        id: uuid4
        email: str
        first_name: str
        last_name: str
        role: str
        subscriptions: List[str]
        groups: Optional[List[str]] = None

    @dataclass
    class OrgMembers:
        members: List["AnacondaCloudOrganizationMembersHandler.OrgMember"]

    @tornado.web.authenticated
    async def get(self, org_name: str):
        try:
            response = await self.anaconda_proxy(f"organizations/{org_name}/users?limit={LIST_ORG_USERS_LIMIT}")
            data = response["remote_data"]
            members = [
                asdict(
                    self.OrgMember(
                        id=item["id"],
                        email=item["email"],
                        first_name=item["first_name"],
                        last_name=item["last_name"],
                        role=item["role"],
                        subscriptions=item["subscriptions"],
                        groups=item["groups"],
                    )
                )
                for item in data["items"]
            ]
            org_members = asdict(self.OrgMembers(members=members))
            self.finish(json.dumps(org_members))
        except UnauthorizedError:
            self.set_status(403, "Could not find user credentiansl")
        except BackendError as ex:
            remote_status_code = ex.data.get("remote_status_code")
            if remote_status_code == 403:
                logger.warning("Forbidden: Can't fetch info from Anaconda Cloud")
                status_message = "Forbidden: Check user credentials"
            elif remote_status_code == 404:
                logger.debug("Not found endpoint")
                status_message = "Forbidden: Check user credentials"
            else:
                status_message = "Unknown error"
            self.set_status(remote_status_code, status_message)
        except Exception:
            self.set_status(500, "Unknown error")


class ExtensionVersionHandler(BackendHandler):
    async def get(self):
        try:
            """
            Returns the installed version plus the latest version published to our main channel
            """
            response = requests.get(f"{config['org']['url']}/package/main/anaconda-toolbox")
            if response.status_code == 200:
                data = response.json()
                try:
                    latest_version = data["platforms"][get_anaconda_org_platform()]
                except KeyError:
                    self.set_status(500, "Could not determine platform/os")
                    self.finish()
                    return

                response = {
                    "anaconda-toolbox": {"latest_published_version": latest_version, "installed_version": __version__}
                }
                self.set_status(200)
                self.finish(json.dumps(response))
            else:
                self.set_status(response.status_code, "Error: Package API responded with an error")
                logger.error(
                    f"Package API responded with error code {response.status_code}. " f"Response {response.text}"
                )
        except Exception as ex:
            logger.exception(ex)
            self.set_status(500, "Error: Could not fetch the extension version")

    def _restart_server(self):
        """
        Restarts the server. This example simply calls sys.exit(0), assuming that
        an external process manager (or the Jupyter server launcher) will restart the server.
        """
        sys.exit(0)

    async def put(self):
        """
        Updates the conda package 'anaconda-toolbox' in the current environment.
        Optionally accepts a query parameter "restart=true" to restart the server
        after updating.
        """
        restart_flag = self.get_argument("restart", "false").lower() == "true"
        allowed_envs = [Environment.local_development, Environment.local_production, Environment.github_actions]
        if config["environment_v2"] not in allowed_envs:
            self.set_status(400, "Environment not allowed")
            return self.finish("Environment not allowed")

        # Build the command to update the package.
        # This command updates 'anaconda-toolbox' non-interactively.
        command = ["conda", "update", "anaconda-toolbox", "-y"]

        try:
            # Run the command in a background thread to avoid blocking.
            await asyncio.to_thread(
                subprocess.run, command, stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True, text=True
            )
        except subprocess.CalledProcessError as err:
            sanitized_err = err.stderr.replace("\r", " ").replace("\n", " ").strip()
            self.set_status(500, f"Error updating anaconda-toolbox {sanitized_err}")
            return self.finish("SUCCESS")

        # Inform the user that the package was updated.
        if restart_flag:
            self.write("anaconda-toolbox updated successfully; restarting server...")
            await self.flush()
            # Schedule a restart after a short delay.
            tornado.ioloop.IOLoop.current().add_callback(self._restart_server)
            self.set_status(200)
            return self.finish("Success")
        else:
            self.write("anaconda-toolbox updated successfully without server restart.")
            self.set_status(200)
            return self.finish("Success")


class LoginHandler(APIHandler):
    pool = None

    @tornado.web.authenticated
    async def get(self):
        """
        Handles login in two different scenarios.
            * When running on-premise/users local machine - handles it using AuthService (backed by anaconda-auth lib)
            * When running on cloud (Anaconda Cloud Notebooks) - handles it by starting the oauth flow between auth.anaconda.com and PythonAnywhere
        """

        if Environment.is_local(config["environment_v2"]):
            # App is running in a local environment, meaning we have to start the oAuth flow
            # from the user's machine (all that is handled by anaconda-auth library)
            """
            This is a workaround for the fact that the login() function in anaconda-cloud-auth
            is not async, and it's not possible to run it in a thread because it starts a tornado
            server. So we run it in a thread pool.

            The next time the user makes a request, if the previous login was still in progress,
            we close the thread pool and start a new one, ensuring that the login will be run again
            without waiting for the previous one and without blocking the main thread.
            """
            if token_info := auth_service.get_token():
                response = json.dumps({"api_key": token_info.api_key})
                self.finish(response)
                return
            try:
                # Try to run a blocking function in a background thread

                # Clean up any existing thread pool
                if hasattr(LoginHandler, "pool") and LoginHandler.pool:
                    LoginHandler.pool.close()
                    LoginHandler.pool = None

                # Create a thread pool for login request
                with ThreadPool(processes=1) as pool:
                    # Store pool reference
                    LoginHandler.pool = pool

                    # Run login in the thread pool
                    future = pool.apply_async(auth_service.login)

                    # Wait to be ready
                    while not future.ready():
                        await tornado.gen.sleep(0.25)

                    # Get the thread response
                    try:
                        api_key = future.get()
                        if not api_key:
                            logger.info("Login failed with unsuccessful. API-KEY not found")
                            self.set_status(401, "Authentication failed")

                        response = json.dumps({"api_key": api_key})
                        self.finish(response)
                    except Exception:
                        logger.info("Login failed with exception")
                        self.set_status(401, "Authentication failed")
                        self.finish()
            except Exception:
                logger.info("Unexpected error during login")
                self.set_status(500, "Internal server error")

            return
        else:
            # The user is running the application on the Cloud (nb.anaconda.com) hosted in
            # PythonAnywhere's infrastructure. In this case the oAuth flow is handled between
            # PythonAnywhere and auth.anaconda.com, therefore the user should be redirected
            # to auth.anaconda.com with return_to set to Anaconda Cloud Notebooks URL
            origin = f"{self.request.protocol}://{self.request.host}"
            auth_url = f"https://auth.{config['anaconda_auth']['domain']}/ui/login?return_to={origin}"
            response = json.dumps({"redirect_auth_url": auth_url})
            self.finish(response)
            return


class ApiKeyHandler(BackendHandler):
    @tornado.web.authenticated
    async def get(self):
        try:
            api_key = auth_service.get_token().api_key
        except (TokenNotFoundError, Exception):
            logger.info("Not found API Key")
            api_key = None

        response = json.dumps({"api_key": api_key})
        self.finish(response)

    @tornado.web.authenticated
    async def put(self):
        try:
            access_credentials: UserAccessCredentials = await self.get_user_access_credentials()
        except UnauthorizedError:
            self.set_status(httpx.codes.UNAUTHORIZED)
            return self.finish()

        # if we reach this part of the code, it is implied that either a valid
        # access token or api key exists and is set in `access_credentials`.

        # api key already exists in keyring
        if access_credentials.api_key:
            self.set_status(httpx.codes.OK)
            return self.finish()

        try:
            auth_service.create_token(access_credentials, save_to_keyring=True)
            self.set_status(httpx.codes.CREATED)
        except UnauthorizedError:
            self.set_status(httpx.codes.UNAUTHORIZED)
        except Exception:
            logger.warning("Unable to retrieve or create API key")
            self.set_status(httpx.codes.SERVICE_UNAVAILABLE)

        self.finish()


class UserFeaturesHandler(BackendHandler):
    @tornado.web.authenticated
    async def get(self):
        try:
            try:
                user_credentials = await self.get_user_access_credentials()
            except UnauthorizedError:
                self.set_status(httpx.codes.FORBIDDEN, "Can't list features - user not authenticated")
                return self.finish()
            status_code, features_data = await user_service.list_features(user_credentials)
            if features_data is not None:
                return self.finish({"features_config": features_data})
            else:
                self.set_status(status_code, "Could not fetch users features")
                return self.finish()
        except UserFeaturesException:
            self.set_status(httpx.codes.INTERNAL_SERVER_ERROR, "Unexpected error while listing users feature")
            return self.finish()


class AccountDetailsHandler(BackendHandler):
    @tornado.web.authenticated
    async def get(self):
        try:
            response = await self.anaconda_proxy("account", method="GET")
            self.finish(response)
        except UnauthorizedError:
            self.set_status(403)
            self.finish(ProxyResponse(remote_status_code=403, remote_data={"error": "User is not authorized"}))
        except BadRequest:
            self.set_status(400)
            self.finish(ProxyResponse(remote_status_code=400, remote_data={"error": "Proxy response is not a JSON"}))
        except Exception:
            self.set_status(500)
            self.finish(ProxyResponse(remote_status_code=500, remote_data={"error": "Unknown error"}))


class LogoutHandler(APIHandler):
    @tornado.web.authenticated
    async def get(self):
        self.clear_all_cookies()
        auth_service.logout()
        app_url = config["app"]["url"]
        return self.redirect(app_url)


def get_routes(base_url: str) -> _RuleList:
    handlers: Dict[str, Type[BackendHandler]] = {
        "login": LoginHandler,
        "api_key": ApiKeyHandler,
        "config": ConfigHandler,
        "feature_flag/init": FeatureFlagInitHandler,
        "should_use_new_oauth": ShouldUseNewOAuthHandler,
        "user/organizations": AnacondaCloudUserOrganizationsHandler,
        "user/features": UserFeaturesHandler,
        r"organization/([^/]+)/members": AnacondaCloudOrganizationMembersHandler,
        "extension/version": ExtensionVersionHandler,
        "account_details": AccountDetailsHandler,
        "logout": LogoutHandler,
    }
    return create_rules(base_url, "aext_core_server", handlers)
