from functools import wraps

from tornado.web import HTTPError

from aext_core_server.exceptions import UserFeaturesException
from aext_core_server.services.user import user_service
from aext_shared.errors import UnauthorizedError


def feature_is_enabled(feature_name):
    """
    This decorator is meant to be used only in classes the inherits from BackendHandler
    since it relies on get_user_access_credentials and also raises HTTPError
    """

    def decorator(method):
        @wraps(method)
        async def wrapper(self, *args, **kwargs):
            if not hasattr(self, "get_user_access_credentials"):
                raise NotImplementedError("Decorated class should implement get_user_access_credentials")
            try:
                credentials = await self.get_user_access_credentials()
            except UnauthorizedError:
                raise HTTPError(403, "Could not fetch user credentials")
            try:
                _, features = await user_service.list_features(credentials)
            except UserFeaturesException:
                raise HTTPError(500, "Could not check user's features")

            try:
                if features is not None and features[feature_name] is False:
                    raise HTTPError(409, f"Feature '{feature_name}' is not enabled")
            except KeyError:
                raise HTTPError(500, f"Unknown '{feature_name}' - check the decorator arguments")

            return await method(self, *args, **kwargs)

        return wrapper

    return decorator
