"""Login reminder dialog."""

from __future__ import annotations

__all__ = ['LoginReminderDialog']

import typing

from qtpy.QtCore import Qt, Signal
from qtpy.QtWidgets import QLabel, QVBoxLayout, QHBoxLayout

from anaconda_navigator.api import cloud
from anaconda_navigator.config import CONF
from anaconda_navigator.utils import attribution
from anaconda_navigator.utils.widgets.signals import DialogOutcome, DialogValue
from anaconda_navigator.widgets import ButtonPrimary, ButtonLink, ButtonSecondaryTextual, CheckBoxBase
from anaconda_navigator.widgets.dialogs.login import AccountOutcome, AccountValue
from anaconda_navigator.widgets.dialogs.login.reminder.styling import BaseCleanDialog
from anaconda_navigator.widgets.web.links import open_link

if typing.TYPE_CHECKING:
    from anaconda_navigator.widgets import main_window


class LoginReminderDialog(BaseCleanDialog):  # pylint: disable=too-many-instance-attributes
    """Login reminder dialog."""
    sig_dialog_request = Signal(DialogOutcome, DialogValue)
    sig_login_request = Signal(AccountOutcome, AccountValue)

    def __init__(self, parent: 'main_window.MainWindow', title: str = 'Sign in for AI Coding Help'):
        """Initialize new :class:`~LoginReminderDialog` instance."""
        super().__init__(parent, title)

        self.__image_label = QLabel()
        self.__image_label.setObjectName('LoginReminderImage')
        self.__image_label.setScaledContents(True)

        self.__message_label = QLabel('Access Anaconda Assistant in Jupyter Notebooks and JupyterLab.')
        self.__message_label.setWordWrap(True)

        self.__do_not_show_again_checkbox = CheckBoxBase('Do not show again')
        self.__do_not_show_again_checkbox.setChecked(CONF.get('main', 'hide_welcoming_sign_up_dialog'))
        self.__do_not_show_again_checkbox.stateChanged.connect(self.__update_do_not_show_again)

        self.__cloud_login_button = ButtonPrimary('Sign In')
        self.__cloud_login_button.setFocusPolicy(Qt.NoFocus)
        self.__repo_login_button = ButtonSecondaryTextual('Connect to a Repository')
        self.__repo_login_button.setFocusPolicy(Qt.NoFocus)

        self.__sign_up_label = QLabel('Don’t have an account? ')
        self.__sign_up_link = ButtonLink('Sign Up')
        self.__sign_up_link.setFocusPolicy(Qt.NoFocus)

        buttons_layout = QHBoxLayout()
        buttons_layout.setAlignment(Qt.AlignRight)
        buttons_layout.addWidget(self.__repo_login_button, alignment=Qt.AlignLeft)
        buttons_layout.addSpacing(16)
        buttons_layout.addWidget(self.__cloud_login_button, alignment=Qt.AlignLeft)

        text_layout = QHBoxLayout()
        text_layout.setAlignment(Qt.AlignRight)
        text_layout.addWidget(self.__sign_up_label, alignment=Qt.AlignLeft)
        text_layout.addSpacing(5)
        text_layout.addWidget(self.__sign_up_link, alignment=Qt.AlignLeft)

        self.__content_layout = QVBoxLayout()
        self.__content_layout.setAlignment(Qt.AlignTop)
        self.__content_layout.addWidget(self.__image_label, alignment=Qt.AlignLeft)
        self.__content_layout.addSpacing(24)
        self.__content_layout.addWidget(self.__message_label)
        self.__content_layout.addSpacing(24)
        self.__content_layout.addWidget(self.__do_not_show_again_checkbox)
        self.__content_layout.addSpacing(16)
        self.__content_layout.addLayout(buttons_layout)
        self.__content_layout.addSpacing(12)
        self.__content_layout.addLayout(text_layout)

        self.__repo_login_button.clicked.connect(lambda: self.sig_dialog_request.emit(
            DialogOutcome.DIALOG_REQUEST, DialogValue.REPO_SELECTOR))

        self.__cloud_login_button.clicked.connect(
            lambda: self.sig_login_request.emit(AccountOutcome.LOGIN_REQUEST, AccountValue.CLOUD),
        )
        cloud.LOGIN_MANAGER.instance.sig_login_done.connect(self.accept)

        self.__sign_up_link.clicked.connect(self.__sign_up)
        self.sig_dialog_request.connect(self.reject)

        self.setLayout(self.__content_layout)

    def accept(self) -> None:
        """Override default `accept` to disconnect global signals."""
        cloud.LOGIN_MANAGER.instance.sig_login_done.disconnect(self.accept)
        return super().accept()

    def __sign_up(self) -> None:
        """Process user-requested "sign up to cloud" action."""
        open_link(
            attribution.POOL.settings.inject_url_parameters(
                'https://auth.anaconda.com/ui/registration/',
                utm_medium='connect-cloud',
                utm_content='signup',
            ),
            origin='welcome-popup',
        )

    def __update_do_not_show_again(self) -> None:
        """Save "do not show again" preference value."""
        CONF.set('main', 'hide_welcoming_sign_up_dialog', self.__do_not_show_again_checkbox.isChecked())
