import scipy
import numpy

try:
	import cupy
except:
	import numpy as cupy

from apricot import FeatureBasedSelection
from apricot.optimizers import *

from sklearn.datasets import load_digits
from sklearn.metrics import pairwise_distances


from nose.tools import assert_less_equal
from numpy.testing import assert_almost_equal
from numpy.testing import assert_array_equal
from numpy.testing import assert_array_almost_equal

#	print("[" + ", ".join(map(str, model.ranking)) + "]")
#	print("[" + ", ".join([str(round(gain, 4)) for gain in model.gains]) + "]")

numpy.random.seed(0)

digits_data = load_digits()
X_digits = digits_data.data[:500]
X_digits_costs = numpy.abs(numpy.random.randn(500))

digits_ranking = [361, 437, 96, 364, 179, 216, 206, 54, 133, 26, 263, 488, 459, 
	384, 192, 316, 219, 79, 407, 187, 374, 318, 225, 150, 69, 495, 439, 163, 
	56, 442, 499, 209, 315, 425, 296, 8, 154, 269, 423, 319, 277, 424, 445, 
	267, 98, 13, 417, 252, 449, 280, 294, 30, 310, 10, 71, 242, 309, 302, 7, 
	365, 64, 226, 87, 117, 347, 35, 195, 413, 389, 27, 326, 366, 142, 408, 458, 
	125, 463, 229, 317, 17, 391, 49, 443, 378, 265, 208, 92, 489, 453, 201, 
	227, 429, 370, 469, 373, 412, 137, 243, 268, 191, 491, 221, 345, 78, 368, 
	394, 253, 15, 114, 39, 322, 472, 433, 244, 171, 363, 211, 284, 61, 435, 405, 
	392, 416, 138, 57, 403, 178, 31, 180, 169, 55, 430, 241, 270, 275, 325, 126, 
	160, 122, 34, 452, 440, 381, 38, 82, 217, 94, 14, 18, 454, 238, 51, 200, 9, 
	298, 65, 149, 74, 346, 314, 456, 349, 479, 305, 52, 44, 141, 1, 59, 357, 
	99, 233, 273, 483, 77, 464, 222, 446, 344, 337, 215, 460, 462, 367, 299, 67, 
	120, 213, 223, 421, 266, 106, 276, 90, 45, 232, 250, 129, 291, 375, 341, 
	436, 131, 159, 143, 251, 487, 261, 231, 471, 313, 173, 376, 228, 203, 135, 
	128, 358, 497, 136, 204, 235, 58, 415, 118, 467, 390, 428, 155, 353, 132, 
	348, 21, 335, 402, 372, 256, 255, 194, 262, 354, 285, 400, 355, 290, 134, 
	332, 254, 451, 177, 239, 320, 184, 323, 395, 282, 496, 447, 297, 303, 396, 
	172, 259, 176, 308, 304, 342, 175, 124, 95]

digits_gains = [88.6281, 49.2776, 49.1627, 33.5536, 30.2392, 21.4102, 21.3163, 
	21.0239, 18.6005, 24.3436, 25.1537, 20.8049, 21.1581, 13.7809, 13.2806, 
	13.6315, 12.9203, 16.6748, 16.568, 14.1052, 12.1243, 13.5557, 13.1612, 
	13.7852, 9.8522, 12.8351, 12.7134, 13.666, 9.6995, 12.1267, 10.3631, 
	12.7313, 11.5199, 10.5392, 10.7362, 10.6637, 9.8994, 9.5494, 11.6034, 
	9.1765, 9.7576, 10.6854, 9.2532, 6.7021, 9.2833, 8.6519, 9.9851, 9.425, 
	7.2098, 7.0312, 8.5304, 9.4939, 8.0097, 8.3233, 7.3764, 7.4018, 8.6303, 
	6.2407, 7.4926, 7.7419, 8.3696, 7.3022, 8.3415, 6.8729, 6.8827, 6.784, 
	7.3556, 9.6413, 6.653, 7.3921, 5.9799, 7.1762, 6.1468, 6.6107, 6.9663, 
	5.1534, 6.105, 7.8232, 9.1366, 6.4755, 6.9012, 6.4959, 5.8511, 6.1167, 
	5.5371, 8.0929, 5.9699, 6.1324, 7.6974, 6.568, 6.0337, 6.7261, 6.8943, 
	5.8799, 5.726, 6.9715, 4.6546, 6.4846, 6.6582, 5.6836, 7.394, 7.2548, 
	5.6177, 6.1787, 5.2397, 5.5067, 5.9741, 6.2165, 4.9515, 5.5464, 4.9831, 
	4.9822, 4.9542, 5.4379, 5.5735, 5.7573, 5.3452, 4.7459, 5.8402, 5.0253, 
	5.6446, 5.5661, 5.7245, 6.032, 4.5713, 5.3202, 6.7798, 5.5925, 5.1634, 
	4.5437, 6.0376, 6.522, 5.488, 5.4749, 5.301, 4.2001, 5.7287, 6.4498, 
	3.8145, 4.4212, 5.8498, 4.8901, 4.7773, 4.7387, 5.6807, 4.5934, 4.2994, 
	5.3386, 3.6605, 3.9401, 4.9063, 4.4232, 4.1919, 4.6068, 4.4669, 4.4454, 
	5.0892, 4.4115, 3.9838, 4.1612, 5.0179, 3.6978, 5.2296, 4.8121, 5.2923, 
	5.1573, 4.4241, 4.003, 3.5921, 4.695, 3.9326, 4.0351, 3.7661, 4.0718, 
	5.529, 4.7826, 4.3825, 4.2937, 4.1185, 4.1308, 3.7346, 4.6208, 3.9064, 
	3.4705, 3.2431, 4.0402, 3.2738, 4.0183, 4.1181, 4.9359, 4.3949, 3.4482, 
	4.0199, 3.2682, 3.5425, 4.3776, 4.18, 3.2546, 3.8277, 3.8881, 3.5466, 
	6.3897, 4.1421, 3.3283, 4.7688, 4.3075, 4.1237, 3.7263, 3.5937, 3.9009, 
	5.2527, 4.6259, 4.4977, 4.1003, 4.4792, 4.5641, 4.5043, 3.185, 3.4011, 
	3.7946, 4.188, 5.0948, 4.1237, 4.4111, 3.9452, 5.084, 3.6193, 3.6234, 
	4.1638, 4.3092, 4.0684, 3.4654, 3.895, 3.597, 4.5198, 4.8692, 4.1396, 
	3.9492, 2.9678, 3.373, 3.6616, 3.4027, 3.3291, 2.8908, 3.4774, 3.5043, 
	4.2464, 3.7875, 4.4772, 3.7516, 3.6698, 3.8446, 3.3235, 4.7009, 4.0049, 
	3.6123, 3.4211, 5.0027, 2.9714, 3.4559, 4.3426, 4.32, 3.0703, 4.0257, 
	3.4569, 3.2732, 3.3623, 3.3122, 3.3131, 3.3538]

digits_greedi_ranking = [361, 437, 96, 364, 179, 216, 206, 54, 133, 26, 263, 
	488, 459, 384, 192, 316, 219, 79, 407, 187, 374, 318, 225, 150, 69, 495, 
	439, 163, 56, 442, 499, 209, 315, 425, 296, 8, 154, 269, 423, 319, 277, 
	424, 445, 267, 98, 13, 417, 252, 449, 280, 294, 30, 310, 10, 71, 242, 
	309, 302, 7, 365, 64, 226, 87, 117, 347, 35, 195, 413, 389, 27, 326, 366, 
	142, 408, 458, 125, 463, 229, 317, 17, 391, 49, 443, 378, 265, 208, 92, 
	489, 453, 201, 227, 429, 370, 469, 373, 412, 137, 243, 268, 191, 491, 
	221, 345, 78, 368, 394, 253, 15, 114, 39, 322, 472, 433, 244, 171, 363, 
	211, 284, 61, 435, 405, 392, 416, 138, 57, 403, 178, 31, 180, 169, 55, 
	430, 241, 270, 275, 325, 126, 160, 122, 34, 452, 440, 381, 38, 82, 217, 
	94, 14, 18, 454, 238, 51, 200, 9, 298, 65, 149, 74, 346, 314, 456, 349, 
	479, 305, 52, 44, 141, 1, 59, 357, 99, 233, 273, 483, 77, 464, 222, 446, 
	344, 337, 215, 460, 462, 367, 299, 67, 120, 213, 223, 421, 266, 106, 276, 
	90, 45, 232, 250, 129, 291, 375, 341, 436, 131, 159, 143, 251, 487, 261, 
	231, 471, 313, 173, 376, 228, 203, 135, 128, 358, 497, 136, 204, 235, 58, 
	415, 118, 467, 390, 428, 155, 353, 132, 348, 21, 335, 402, 372, 256, 255, 
	194, 262, 354, 285, 400, 355, 290, 134, 332, 254, 451, 177, 239, 320, 184, 
	323, 395, 282, 496, 447, 297, 303, 396, 172, 259, 176, 308, 304, 342, 175, 
	124, 95]

digits_greedi_gains = [88.6281, 49.2776, 49.1627, 33.5536, 30.2392, 21.4102, 
	21.3163, 21.0239, 18.6005, 24.3436, 25.1537, 20.8049, 21.1581, 13.7809, 
	13.2806, 13.6315, 12.9203, 16.6748, 16.568, 14.1052, 12.1243, 13.5557,
	13.1612, 13.7852, 9.8522, 12.8351, 12.7134, 13.666, 9.6995, 12.1267, 
	10.3631, 12.7313, 11.5199, 10.5392, 10.7362, 10.6637, 9.8994, 9.5494, 
	11.6034, 9.1765, 9.7576, 10.6854, 9.2532, 6.7021, 9.2833, 8.6519, 
	9.9851, 9.425, 7.2098, 7.0312, 8.5304, 9.4939, 8.0097, 8.3233, 7.3764, 
	7.4018, 8.6303, 6.2407, 7.4926, 7.7419, 8.3696, 7.3022, 8.3415, 6.8729, 
	6.8827, 6.784, 7.3556, 9.6413, 6.653, 7.3921, 5.9799, 7.1762, 6.1468, 
	6.6107, 6.9663, 5.1534, 6.105, 7.8232, 9.1366, 6.4755, 6.9012, 6.4959, 
	5.8511, 6.1167, 5.5371, 8.0929, 5.9699, 6.1324, 7.6974, 6.568, 6.0337, 
	6.7261, 6.8943, 5.8799, 5.726, 6.9715, 4.6546, 6.4846, 6.6582, 5.6836, 
	7.394, 7.2548, 5.6177, 6.1787, 5.2397, 5.5067, 5.9741, 6.2165, 4.9515, 
	5.5464, 4.9831, 4.9822, 4.9542, 5.4379, 5.5735, 5.7573, 5.3452, 4.7459, 
	5.8402, 5.0253, 5.6446, 5.5661, 5.7245, 6.032, 4.5713, 5.3202, 6.7798, 
	5.5925, 5.1634, 4.5437, 6.0376, 6.522, 5.488, 5.4749, 5.301, 4.2001, 
	5.7287, 6.4498, 3.8145, 4.4212, 5.8498, 4.8901, 4.7773, 4.7387, 5.6807, 
	4.5934, 4.2994, 5.3386, 3.6605, 3.9401, 4.9063, 4.4232, 4.1919, 4.6068, 
	4.4669, 4.4454, 5.0892, 4.4115, 3.9838, 4.1612, 5.0179, 3.6978, 5.2296, 
	4.8121, 5.2923, 5.1573, 4.4241, 4.003, 3.5921, 4.695, 3.9326, 4.0351, 
	3.7661, 4.0718, 5.529, 4.7826, 4.3825, 4.2937, 4.1185, 4.1308, 3.7346, 
	4.6208, 3.9064, 3.4705, 3.2431, 4.0402, 3.2738, 4.0183, 4.1181, 4.9359, 
	4.3949, 3.4482, 4.0199, 3.2682, 3.5425, 4.3776, 4.18, 3.2546, 3.8277, 
	3.8881, 3.5466, 6.3897, 4.1421, 3.3283, 4.7688, 4.3075, 4.1237, 3.7263, 
	3.5937, 3.9009, 5.2527, 4.6259, 4.4977, 4.1003, 4.4792, 4.5641, 4.5043, 
	3.185, 3.4011, 3.7946, 4.188, 5.0948, 4.1237, 4.4111, 3.9452, 5.084, 
	3.6193, 3.6234, 4.1638, 4.3092, 4.0684, 3.4654, 3.895, 3.597, 4.5198, 
	4.8692, 4.1396, 3.9492, 2.9678, 3.373, 3.6616, 3.4027, 3.3291, 2.8908, 
	3.4774, 3.5043, 4.2464, 3.7875, 4.4772, 3.7516, 3.6698, 3.8446, 3.3235, 
	4.7009, 4.0049, 3.6123, 3.4211, 5.0027, 2.9714, 3.4559, 4.3426, 4.32, 
	3.0703, 4.0257, 3.4569, 3.2732, 3.3623, 3.3122, 3.3131, 3.3538]

digits_approx_ranking = [361, 437, 96, 364, 179, 216, 206, 54, 133, 26, 263, 
	459, 488, 192, 384, 316, 79, 219, 407, 187, 374, 318, 225, 150, 69, 495, 
	439, 163, 442, 56, 499, 209, 315, 425, 296, 8, 154, 269, 423, 319, 277, 
	424, 445, 267, 98, 13, 417, 449, 252, 294, 280, 30, 310, 10, 71, 242, 
	309, 302, 7, 365, 226, 64, 87, 347, 117, 35, 195, 389, 27, 413, 326, 
	366, 142, 408, 458, 125, 463, 229, 17, 391, 49, 317, 443, 265, 378, 208, 
	92, 489, 453, 201, 227, 429, 370, 469, 373, 412, 137, 243, 268, 191, 491, 
	221, 345, 78, 368, 253, 394, 114, 39, 322, 433, 15, 472, 244, 171, 363, 
	211, 284, 61, 392, 405, 435, 416, 138, 57, 178, 180, 403, 169, 31, 55, 430, 
	270, 241, 275, 325, 126, 160, 34, 122, 82, 452, 381, 440, 14, 38, 94, 217, 
	454, 18, 51, 238, 200, 298, 65, 9, 346, 74, 314, 149, 456, 349, 479, 305, 
	52, 1, 141, 44, 99, 59, 357, 483, 273, 233, 77, 222, 464, 344, 446, 215, 
	337, 462, 367, 67, 299, 460, 120, 106, 213, 266, 421, 276, 90, 250, 232, 
	129, 45, 223, 341, 291, 375, 131, 159, 251, 487, 436, 143, 261, 471, 231, 
	313, 173, 376, 228, 203, 128, 358, 135, 497, 136, 58, 235, 415, 428, 390, 
	204, 118, 353, 155, 467, 132, 348, 335, 21, 402, 256, 372, 194, 262, 255, 
	354, 400, 290, 134, 285, 355, 451, 239, 177, 254, 332, 320, 184, 282, 496, 
	395, 297, 303, 323, 396, 172, 304, 259, 308, 176, 447, 342, 175, 124, 95]

digits_approx_gains = [88.6281, 49.2776, 49.1627, 33.5536, 30.2392, 21.4102, 
	21.3163, 21.0239, 18.6005, 24.3436, 25.1537, 22.0441, 19.9189, 13.7357, 
	13.3258, 13.6315, 17.0328, 12.5623, 16.568, 14.1052, 12.1243, 13.5557, 
	13.1612, 13.7852, 9.8522, 12.8351, 12.7134, 13.666, 12.2839, 9.5424, 
	10.3631, 12.7313, 11.5199, 10.5392, 10.7362, 10.6637, 9.8994, 9.5494, 
	11.6034, 9.1765, 9.7576, 10.6854, 9.2532, 6.7021, 9.2833, 8.6519, 9.9851, 
	7.2957, 9.3391, 8.5895, 6.9722, 9.4939, 8.0097, 8.3233, 7.3764, 7.4018, 
	8.6303, 6.2407, 7.4926, 7.7419, 7.3417, 8.3301, 8.3415, 6.9406, 6.8151, 
	6.784, 7.3556, 6.6915, 7.4707, 9.5242, 5.9799, 7.1762, 6.1468, 6.6107, 
	6.9663, 5.1534, 6.105, 7.8232, 6.5199, 6.9709, 6.5252, 8.9931, 5.8511, 
	5.5649, 6.0888, 8.0929, 5.9699, 6.1324, 7.6974, 6.568, 6.0337, 6.7261, 
	6.8943, 5.8799, 5.726, 6.9715, 4.6546, 6.4846, 6.6582, 5.6836, 7.394, 
	7.2548, 5.6177, 6.1787, 5.2397, 6.0044, 5.4765, 4.9737, 5.5765, 5.0003, 
	4.9929, 6.1239, 4.9667, 5.4379, 5.5735, 5.7573, 5.3452, 4.7459, 5.8402, 
	5.619, 5.6457, 4.9713, 5.7245, 6.032, 4.5713, 6.7975, 5.1957, 5.2931, 
	4.575, 5.5383, 6.0376, 6.522, 5.4915, 5.4713, 5.301, 4.2001, 5.7287, 
	6.4498, 4.4337, 3.802, 5.76, 5.8242, 4.7753, 4.8518, 5.3849, 4.7108, 
	4.2977, 4.5634, 3.9506, 3.6501, 4.4394, 4.89, 4.1919, 4.4805, 4.4599, 
	4.5787, 4.0068, 4.4165, 4.1748, 5.0476, 5.0179, 3.6978, 5.2296, 4.8121, 
	5.2923, 4.0281, 4.4294, 5.1269, 3.9538, 3.5831, 4.6827, 4.0929, 3.7639, 
	4.0163, 5.529, 4.3956, 4.7694, 4.1321, 4.2801, 3.7463, 4.1191, 3.9183, 
	3.4792, 4.0556, 3.2457, 4.5821, 3.2738, 3.4867, 4.0125, 4.4083, 4.924, 
	4.0278, 3.2758, 4.2129, 4.3896, 3.2699, 3.5279, 4.0227, 3.5576, 3.8225, 
	3.8823, 4.1491, 3.3381, 4.3307, 4.1423, 6.3533, 4.7464, 3.7263, 3.9075, 
	3.5871, 5.2527, 4.6259, 4.4977, 4.1003, 4.4792, 4.5142, 3.1924, 4.5468, 
	3.4011, 3.7946, 4.1429, 5.092, 4.4166, 3.6497, 3.6271, 4.1567, 3.9321, 
	4.3316, 4.1626, 5.0512, 4.0684, 3.4654, 3.6033, 3.8887, 4.5198, 4.1464, 
	4.8624, 2.9741, 3.3802, 3.9357, 3.6616, 3.3333, 3.4879, 3.5179, 3.3863, 
	2.8789, 4.4972, 3.6917, 3.7588, 3.7737, 4.211, 3.8446, 3.3235, 3.6289, 
	3.4333, 3.9974, 2.9824, 3.4678, 4.6653, 4.349, 4.3327, 3.2933, 3.0679, 
	3.4689, 4.0204, 4.9506, 3.3623, 3.3122, 3.3131, 3.3538]


digits_stochastic_ranking = [90, 278, 143, 201, 396, 262, 476, 99, 39, 199, 
	321, 163, 381, 152, 342, 478, 45, 78, 331, 224, 26, 253, 349, 439, 446, 
	209, 256, 247, 276, 354, 144, 498, 344, 92, 62, 203, 177, 466, 165, 106, 
	296, 298, 414, 280, 23, 436, 424, 84, 269, 384, 491, 305, 357, 479, 229, 
	109, 401, 139, 337, 24, 493, 218, 323, 110, 487, 410, 486, 50, 313, 220, 
	188, 65, 266, 166, 227, 150, 408, 368, 400, 383, 133, 193, 407, 459, 297, 
	64, 194, 10, 31, 181, 468, 168, 235, 395, 81, 18, 406, 386, 211, 34, 148, 
	333, 241, 22, 376, 119, 35, 248, 154, 405, 137, 160, 89, 122, 382, 146, 
	197, 120, 196, 56, 4, 348, 351, 412, 255, 327, 76, 244, 419, 455]

digits_stochastic_gains = [208.1058, 34.9391, 66.1238, 139.4914, 44.0288, 
	37.259, 14.608, 47.4405, 66.3913, 14.175, 19.795, 240.4258, 45.1475, 
	19.1018, 21.583, 14.3827, 27.3749, 47.8223, 12.8022, 9.4617, 331.1841, 
	41.1507, 28.6518, 174.557, 24.6193, 139.1466, 17.6943, 10.7508, 21.1417, 
	15.3493, 3.9128, 13.725, 21.0808, 46.3346, 11.7431, 16.6547, 14.3263, 
	8.6153, 10.9222, 18.0867, 102.5885, 21.7749, 5.3496, 66.4936, 10.9994, 
	16.9759, 78.2176, 6.8646, 86.2641, 187.3022, 29.4332, 18.5908, 17.4895, 
	18.6013, 38.8778, 5.71, 7.4692, 5.9094, 16.2617, 2.5924, 6.6961, 3.2064, 
	9.0945, 3.2062, 13.2021, 4.8524, 4.2264, 6.9179, 12.2523, 7.5919, 8.8202, 
	15.7501, 13.4587, 7.1858, 28.5414, 106.4279, 34.4298, 23.0747, 9.7728, 
	6.2705, 181.7605, 4.5085, 130.4226, 149.9606, 8.9354, 38.2979, 9.367, 
	44.8715, 15.5617, 3.9888, 2.9066, 6.2502, 10.2346, 8.7379, 6.637, 14.8912, 
	7.4731, 6.6325, 20.6332, 14.6896, 5.0146, 5.5954, 15.8186, 5.4479, 
	10.2851, 5.6058, 35.5256, 7.298, 57.388, 16.4506, 20.9093, 14.4064, 
	4.9722, 14.2542, 2.4066, 5.8037, 6.8069, 11.0472, 5.9731, 71.0232, 
	2.301, 8.439, 3.6548, 19.3062, 7.8866, 2.7835, 6.5157, 16.6291, 3.3089, 
	2.7533]

digits_sample_ranking = [361, 437, 364, 179, 216, 206, 54, 133, 26, 488, 263, 
	459, 384, 192, 219, 79, 407, 187, 318, 225, 150, 69, 495, 439, 163, 56, 
	442, 499, 209, 315, 425, 296, 8, 269, 423, 277, 319, 424, 267, 98, 13, 417, 
	252, 449, 280, 294, 30, 310, 10, 71, 242, 309, 302, 7, 365, 64, 226, 87, 
	347, 117, 35, 195, 389, 413, 27, 326, 142, 408, 458, 125, 463, 229, 17, 
	317, 391, 49, 443, 378, 265, 208, 92, 453, 201, 227, 429, 370, 469, 373, 
	412, 137, 243, 491, 268, 345, 78, 368, 394, 253, 114, 39, 433, 472, 244, 
	363, 211, 284, 61, 392, 435, 405, 416, 57, 138, 178, 180, 31, 169, 55, 
	430, 270, 275, 126, 160, 122, 34, 452, 38, 381, 440, 14, 94, 82, 217, 454, 
	18, 200, 51, 298, 9, 65, 346, 74, 149, 314, 456, 349, 479, 305, 52, 141, 
	44, 99, 59, 357, 483, 273, 77, 464, 222, 446, 344, 337, 215, 462, 460, 299, 
	120, 67, 223, 421, 266, 106, 232, 276, 129, 291, 375, 341, 131, 436, 143, 
	487, 251, 261, 471, 231, 313, 173, 376, 228, 203, 135, 128, 497, 358, 136, 
	235, 204, 58, 415, 467, 155, 132, 348, 335, 402, 194, 372, 256, 255, 262, 
	400, 354, 134, 285, 239, 290, 177, 332, 320, 184, 395, 496, 282, 323, 447, 
	297, 303, 172, 396, 259, 176, 308, 342, 304, 175, 95, 32, 274, 248, 328, 
	360, 338, 188, 406, 212, 197, 411, 359, 12, 379, 484, 399, 420]

digits_sample_gains = [88.6281, 49.2776, 46.1952, 35.1591, 24.0905, 23.6633, 
	22.8776, 20.4509, 25.978, 23.0994, 24.9552, 22.0361, 14.3358, 13.7588, 
	13.9028, 17.7287, 17.5856, 14.9446, 14.6097, 14.1903, 14.6263, 10.5363, 
	13.5932, 13.4639, 14.3232, 10.2515, 12.4887, 10.7796, 13.3691, 12.0276, 
	11.0021, 11.2662, 11.1356, 10.0853, 12.1591, 10.4833, 9.5426, 11.2118, 
	7.1722, 9.8493, 9.14, 10.5354, 9.9864, 7.5875, 7.4246, 9.0387, 10.0267, 
	8.3476, 8.7625, 7.6871, 7.7916, 9.0025, 6.5112, 7.8089, 8.0492, 8.788, 
	7.606, 8.6363, 7.2122, 7.0589, 7.0383, 7.644, 6.9673, 9.8162, 7.6994, 
	6.221, 6.4352, 6.8355, 7.2875, 5.3594, 6.3144, 8.2114, 6.8265, 9.3427, 
	7.1588, 6.7564, 6.0313, 6.3289, 5.7286, 8.4513, 6.1857, 8.0396, 6.7919, 
	6.3059, 6.9368, 7.1671, 6.141, 5.9238, 7.2676, 4.8534, 6.727, 7.7432, 
	6.7939, 5.871, 6.4799, 5.5088, 5.7597, 6.2557, 5.203, 5.8162, 5.2386, 
	5.1904, 5.7111, 6.1262, 5.6227, 4.997, 6.206, 5.9558, 5.2691, 5.9093, 
	6.0531, 4.8762, 6.3456, 7.1599, 5.4449, 5.7894, 4.7688, 6.3723, 6.8906, 
	5.8157, 5.5807, 6.0653, 6.8431, 4.0457, 4.683, 6.2034, 5.0719, 5.0336, 
	5.1226, 5.7008, 4.5525, 5.928, 4.8004, 4.1914, 3.855, 4.4669, 4.6575, 
	4.7497, 4.8366, 4.6947, 4.2414, 4.6538, 5.3152, 4.3872, 5.2803, 3.9046, 
	5.5042, 5.0626, 5.5824, 4.6882, 5.4045, 4.1932, 3.7685, 4.9292, 4.3462, 
	3.9668, 5.8442, 5.0446, 4.614, 4.5163, 4.3327, 4.3527, 3.9379, 4.1269, 
	4.845, 3.419, 3.4615, 4.2472, 4.345, 5.2242, 4.6351, 3.6382, 4.6546, 
	4.2166, 3.4644, 4.0436, 4.1163, 3.7715, 4.3943, 6.6336, 5.0795, 4.3962, 
	4.55, 3.9479, 4.1614, 3.7916, 5.5351, 4.8986, 4.7731, 4.3803, 4.7372, 
	4.833, 4.7654, 3.633, 3.3616, 4.0037, 5.4006, 4.4093, 4.3546, 4.6434, 
	5.3603, 4.4098, 4.3137, 3.6787, 3.834, 4.8287, 3.2203, 5.1323, 4.4007, 
	4.1991, 3.5845, 3.589, 3.8798, 3.7741, 3.5977, 3.99, 3.6831, 4.0231, 
	4.4803, 4.1228, 3.5323, 4.2817, 3.6936, 3.857, 4.8977, 5.2928, 3.1879, 
	3.6964, 4.6235, 4.6104, 3.2628, 4.2733, 3.6797, 3.5954, 3.4702, 3.5199, 
	3.5855, 4.4705, 3.2304, 3.9827, 3.4255, 3.3793, 3.2972, 3.5498, 3.8863, 
	3.7346, 3.8974, 3.778, 2.9142, 3.5567, 4.0718, 3.9667, 3.8971, 4.2189]

digits_modular_ranking = [361, 437, 96, 364, 179, 216, 206, 54, 133, 26, 488, 
	459, 384, 316, 219, 192, 263, 187, 407, 79, 374, 69, 318, 150, 225, 439, 
	495, 56, 499, 315, 209, 163, 296, 425, 269, 8, 154, 423, 319, 442, 424, 
	445, 277, 267, 13, 449, 417, 98, 252, 294, 280, 242, 310, 302, 10, 71, 
	309, 30, 35, 347, 365, 226, 7, 389, 117, 195, 64, 326, 125, 142, 87, 408, 
	27, 458, 463, 366, 17, 265, 378, 92, 443, 49, 391, 229, 489, 208, 201, 
	370, 453, 227, 137, 469, 373, 413, 412, 345, 114, 491, 394, 429, 317, 368, 
	433, 243, 191, 268, 253, 39, 284, 322, 221, 244, 78, 472, 435, 15, 363, 
	405, 61, 138, 392, 171, 416, 180, 211, 122, 169, 325, 57, 18, 55, 178, 
	34, 126, 94, 275, 217, 454, 270, 452, 38, 74, 349, 241, 346, 160, 403, 9, 
	59, 51, 430, 381, 14, 440, 314, 298, 200, 238, 149, 1, 273, 65, 120, 462, 
	31, 305, 233, 446, 299, 99, 483, 337, 479, 357, 456, 82, 52, 367, 90, 129, 
	344, 222, 44, 141, 464, 213, 276, 159, 106, 45, 215, 460, 223, 266, 232, 
	67, 250, 291, 421, 251, 261, 375, 77, 487, 341, 471, 358, 231, 390, 131, 
	497, 203, 335, 428, 118, 415, 143, 228, 173, 348, 136, 128, 58, 355, 255, 
	285, 254, 402, 21, 262, 194, 313, 400, 354, 235, 353, 332, 297, 256, 376, 
	259, 135, 290, 304, 132, 274, 239, 134, 496, 184, 204, 175, 155, 303, 451, 
	359, 282, 328, 177, 395, 342, 320, 124, 248, 95, 308, 396, 360, 240, 60, 
	288, 188]

digits_modular_gains = [88.6281, 49.2776, 49.1627, 33.5536, 30.2392, 21.4102, 
	21.3163, 21.0239, 18.6005, 24.3436, 21.8248, 22.1498, 14.6174, 14.7146, 
	13.8577, 12.8173, 20.7484, 15.0779, 16.6032, 15.667, 12.1243, 10.5438, 
	13.3021, 13.8998, 12.6086, 12.9429, 12.6056, 9.8527, 11.0453, 12.2076, 
	12.8129, 12.7589, 11.0089, 10.5018, 9.9407, 10.6487, 9.8615, 11.7446, 
	9.2623, 10.6284, 10.8207, 9.3886, 9.4869, 6.7021, 8.7993, 7.4797, 9.9837, 
	8.9533, 9.3391, 8.5895, 6.9722, 7.6576, 8.0123, 6.5012, 8.3233, 7.331, 
	8.637, 9.0137, 7.2385, 7.1299, 7.6349, 7.2779, 7.2884, 6.8517, 6.8012, 
	7.4547, 8.0651, 6.1008, 5.3779, 6.229, 7.9713, 6.6562, 7.3511, 7.0436, 
	6.1553, 7.0138, 6.5984, 5.7882, 6.2854, 6.2285, 5.8846, 6.51, 6.9062, 
	7.5401, 6.2211, 8.074, 6.684, 7.0669, 7.6731, 6.0547, 4.787, 5.9046, 
	5.7798, 8.8235, 6.9968, 5.8183, 5.2652, 7.5079, 5.7178, 6.8193, 8.1538, 
	5.3108, 5.1974, 6.3137, 5.5613, 6.4414, 5.986, 5.6107, 4.9517, 5.0234, 
	6.9593, 5.4848, 5.9595, 4.9433, 5.1334, 6.0346, 5.7439, 5.7101, 5.8472, 
	6.1393, 5.5868, 5.3848, 5.7201, 5.2581, 5.1776, 3.997, 4.612, 4.3586, 
	4.5095, 3.9115, 6.062, 6.6625, 4.5401, 5.7957, 4.4819, 5.3081, 4.7344, 
	4.1144, 5.3735, 5.8755, 4.7982, 4.6465, 3.9433, 5.3034, 4.1692, 6.3013, 
	5.03, 4.693, 3.8114, 4.4952, 6.1688, 4.6654, 5.257, 4.7556, 4.2596, 
	4.4525, 4.1426, 4.7913, 5.0721, 4.1121, 3.9113, 4.3963, 3.4838, 4.1385, 
	5.0666, 4.7971, 4.1032, 4.4124, 3.4035, 3.9653, 4.132, 4.2369, 5.1075, 
	4.6578, 4.8352, 5.2105, 5.133, 3.5451, 3.4012, 3.404, 4.1171, 4.3596, 
	4.9272, 4.2309, 4.6939, 4.0407, 4.0911, 3.4635, 3.4982, 3.5903, 3.6595, 
	4.5273, 4.0931, 4.3746, 4.3981, 3.9383, 4.1827, 3.8326, 4.856, 4.3681, 
	3.787, 3.88, 5.1671, 4.1483, 3.522, 3.9363, 3.2727, 3.6151, 3.7704, 4.0696, 
	3.4527, 4.5223, 3.774, 3.7362, 4.0497, 4.5021, 4.651, 4.0528, 4.5525, 
	3.5629, 3.8009, 4.4599, 4.1349, 3.0167, 4.0708, 3.514, 3.9565, 4.5866, 
	3.9307, 3.45, 3.0172, 5.1046, 3.3972, 3.7501, 4.9486, 4.2396, 4.3388, 
	3.1085, 4.1273, 4.3012, 3.2076, 4.3668, 3.4935, 3.4235, 3.9811, 3.1918, 
	3.7005, 3.4823, 3.4726, 3.3583, 3.9762, 3.4183, 3.9798, 3.5049, 4.4256, 
	2.8802, 3.6136, 3.3287, 3.6929, 3.9743, 3.4, 3.7764, 3.353, 3.7937, 3.381, 
	3.4648, 4.2951, 3.2154, 2.8252, 2.9735, 2.9633, 3.3773]

# Test all optimizers

def test_digits_naive():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='naive')
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_ranking)
	assert_array_almost_equal(model.gains, digits_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_lazy():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='lazy')
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_ranking)
	assert_array_almost_equal(model.gains, digits_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_two_stage():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='two-stage')
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_ranking)
	assert_array_almost_equal(model.gains, digits_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_greedi_nn():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='greedi',
		optimizer_kwds={'optimizer1': 'naive', 'optimizer2': 'naive'}, 
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking[:50], digits_greedi_ranking[:50])
	assert_array_almost_equal(model.gains[:50], digits_greedi_gains[:50], 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_greedi_ll():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='greedi',
		optimizer_kwds={'optimizer1': 'lazy', 'optimizer2': 'lazy'}, 
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking[:30], digits_greedi_ranking[:30])
	assert_array_almost_equal(model.gains[:30], digits_greedi_gains[:30], 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_greedi_ln():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='greedi',
		optimizer_kwds={'optimizer1': 'lazy', 'optimizer2': 'naive'}, 
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_greedi_ranking)
	assert_array_almost_equal(model.gains, digits_greedi_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_greedi_nl():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='greedi',
		optimizer_kwds={'optimizer1': 'naive', 'optimizer2': 'lazy'}, 
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking[:30], digits_greedi_ranking[:30])
	assert_array_almost_equal(model.gains[:30], digits_greedi_gains[:30], 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_approximate():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='approximate-lazy')
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_approx_ranking)
	assert_array_almost_equal(model.gains, digits_approx_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_stochastic():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='stochastic',
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_stochastic_ranking)
	assert_array_almost_equal(model.gains, digits_stochastic_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_sample():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='sample',
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_sample_ranking)
	assert_array_almost_equal(model.gains, digits_sample_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)

def test_digits_modular():
	model = FeatureBasedSelection(100, 'sqrt', optimizer='modular',
		random_state=0)
	model.fit(X_digits, sample_cost=X_digits_costs)
	assert_array_equal(model.ranking, digits_modular_ranking)
	assert_array_almost_equal(model.gains, digits_modular_gains, 4)
	assert_less_equal(sum(X_digits_costs[model.ranking]), 100)
