import os
from typing import Optional

import alembic
from aext_project_filebrowser_server import is_github_actions_runner
from aext_project_filebrowser_server.consts import ALEMBIC_DIR
from aext_project_filebrowser_server.exceptions import AlembicConfigDBPropertyNotFound
from aext_project_filebrowser_server.migrations import alembic_config
from aext_project_filebrowser_server.services.database import SQLite, db

from aext_shared import logger as custom_logger

logger = custom_logger.logger


def migrate():
    try:
        config = alembic.config.Config()
        if is_github_actions_runner():
            config.set_main_option("script_location", ALEMBIC_DIR)
            config.set_main_option("sqlalchemy.url", alembic_config.get_db_connection_gh_runner_url())
            alembic.command.upgrade(config, "head")
        else:
            config.set_main_option("script_location", ALEMBIC_DIR)
            config.set_main_option("sqlalchemy.url", alembic_config.get_db_connection_unittest_url())
            alembic.command.upgrade(config, "head")
    except (alembic.util.exc.CommandError, AlembicConfigDBPropertyNotFound):
        logger.error("Failed to migrate database while preparing tests")


def _get_database_path() -> str:
    if is_github_actions_runner():
        return alembic_config.get_gh_runner_db_path()
    return alembic_config.get_unittest_db_path()


def setup_database(test_db_file_path: Optional[str] = None):
    if not test_db_file_path:
        test_db_file_path = _get_database_path()
    migrate()
    db.driver = SQLite(test_db_file_path)
    return db


def tear_down_database(test_db_file_path: Optional[str] = None):
    if not test_db_file_path:
        test_db_file_path = _get_database_path()
    try:
        os.remove(test_db_file_path)
    except FileNotFoundError:
        pass
