import os
import re
import sqlite3
import sys
from dataclasses import asdict
from unittest import mock
from uuid import uuid4

import httpx
import pytest
import tornado
from aext_project_filebrowser_server.consts import (
    FileOperations,
    LocalStorageSyncState,
    RequestMethods,
)
from aext_project_filebrowser_server.exceptions import (
    AddFileErrorOnDatabase,
    AddFileErrorOnLocalStorage,
)
from aext_project_filebrowser_server.handlers import project_service
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.services.local_storage import (
    DiffFileOperations,
    FileOperationsDict,
    local_storage,
)
from aext_project_filebrowser_server.services.sync import (
    SyncCoroutine,
    SyncFileResult,
    put_file,
    sync_pull_delete_file,
)
from aext_project_filebrowser_server.utils import run_coroutine
from tests.test_assets.project_data import (
    ALL_SYNC_STATE_VARIATIONS_CONTENTS,
    EMPTY_LOCAL_PROJECT,
    LOCAL_PROJECT,
    ONE_FOLDER_ONE_FILE_LOCAL_PROJECT,
)

from .setup_tests import BasicDBAppHTTPTests
from .test_assets.responses import EXPECTED_RESPONSE_1, PAYLOAD_1, PAYLOAD_2

# Skip all tests if the platform is Windows
pytestmark = pytest.mark.skipif(sys.platform.startswith("win"), reason="Skipping tests on Windows")


@pytest.fixture
def test_file():
    file_path = "/tmp/file.txt"
    with open(file_path, "w") as f:
        f.write("test")
    yield file_path
    os.remove(file_path)


@pytest.fixture()
def mock_cloud_request_status_code():
    return 201


@pytest.fixture
def mock_cloud_request(mock_cloud_request_status_code):
    with mock.patch("aext_project_filebrowser_server.services.sync.cloud_request") as m:
        m.return_value = httpx.Response(
            status_code=mock_cloud_request_status_code,
            json={
                "name": "autompg-3-step.csv",
                "description": None,
                "writtenby": "7ef9f202-7659-4f0d-b28e-5c5c841949a2",
                "type": "file",
                "size": 0,
                "etag": "",
                "last_modified": "2024-06-11T14:58:01.871032",
                "file_version_id": "4d525af6-f8d7-4021-b22a-58979d90ec26",
                "url": "https://nucleus-latest.anacondaconnect.com/api/projects/04a3f6ac-9407-440c-a39b-f27b3eb53d28/files/autompg-3-step.csv?version=4d525af6-f8d7-4021-b22a-58979d90ec26",
                "signed_url": "https://anaconda-cloud-dev-us-east-1-projects.s3.amazonaws.com/04a3f6ac-9407-440c-a39b-f27b3eb53d28/66304ced-132e-4bd1-9d79-367f8a883939/files/autompg-by-redirect.csv?AWSAccessKeyId=ASIA3ZWAJ6QQVV7FNUH6&Signature=vvZGQkijZwC2RR6OvwAJr3j7aXk%3D&x-amz-security-token=IQoJb3JpZ2luX2VjEMz%2F%2F%2F%2F%2F%2F%2F%2F%2F%2FwEaCXVzLWVhc3QtMSJIMEYCIQDUchZszY31PP4CeHp%2BkYeYNYbetqGPWZeDQyNFtxFE9QIhAK0AhGU0cCZoQe%2BQ7rC225jPzAifRDdDQ9ptsjHRuMYAKv4ECGQQBBoMODExMDc4MzgyNjI1IgyHj6vkiMyeGMUPUUgq2wTl1VosTXXB3DTHTrLLzZD4qkdRpn0l4TRcDr0%2F326XTRc1S4A25HKspb2eURlJ1loO6lV2R2aXrJs1WlOqyGTmYwOZofA%2FWzSyT9GjGwd4dTtbYpUcHyEtOY2rnp08y5kijYke0vZn0n71M2cJK4b1HCxh3ycKY5lJFoWV8g%2FJbJl0QQ4TRNseQj%2FAhKZ4W5epC4TdHDR03QACCESzxlrIxSX%2BfaUu3gK7bItxPPWXEao0EAfjQtjEd9ygdGmViA1rU6HQA4FZvKWFWD3ygXt25lmYQ5ZRTfVwjHhPSJTLErIUUIxuAjc085Ni8cjC0sjptrgVSnDH7xSj5mPIE%2B8iOf4dlpugiFTXqNXOJlLi8KcyhH7X%2BPJK84RCH8s8AqoQ4mHUKZCnGyNaZwECVWE6WQjl0uTBawmYb3Vzhun%2BOibJ0useYq6Po2AaGIqDPYmKfKaPuI4e%2B9zmC4nYZGcQsUGpQaqW9VwUHwr3evp8Ua0WTtSuExDEcylu9K8XhCTY8tqidHaR78%2F8cRNRoNN3RVFTmLQ%2BaKGpBKrlkmVnQEeYbv4etPR4MpypZUmNKLOQka5ZZni56W%2FET2DkJzW7r2hcYt2quP%2Bb78ENePKhqTprWr4ZmrPEiEf90hc%2FIj4cG0qI8Sc5xwSO8u97YwPUenClEMwvxcwgiLejyP0SYsCBsAYQCiL1xQ7uTX9pxxte%2BwjkzXwHsL2E6M6VrRhNZt%2BNg%2FSnDmwitbP5TXiq4LeKdf8UsZz0fzjzH8UIBzwG%2BQioSFnYiAdC2jJeNpYMPkmunWkAAEDRehMwzYWLpgY6mQH6TOBaxQqhix30s0cbgCr8PrQS4eNicBxg105mtOtYGOxb1v%2F7ek7ukJmXh8WQDgWdj26QMsT11Vv0fcb%2BKiYEbQTqFyZke3LNB7bwG4JWPEPNw7wAUcmhH6gU3mbPTTuDPm%2FfammPQpT%2Fmzn9o8l6JPzBm4XZrxQkg7%2FK0bz0NvISqgvWBEDGkPmJvGeg89veA1smJmVd1zE%3D&Expires=1690487073",
                "metadata": {},
                "project_id": "04a3f6ac-9407-440c-a39b-f27b3eb53d28",
                "deleted": False,
            },
        )
        yield m


@pytest.fixture()
def mock_make_request_status_code():
    return 200


@pytest.fixture
def mock_make_request(mock_make_request_status_code):
    with mock.patch("aext_project_filebrowser_server.services.sync.make_request") as m:
        m.return_value = httpx.Response(status_code=mock_make_request_status_code)
        yield m


def _test_process(msg: str):
    print(msg)


async def _test_coroutine(msg: str):
    return SyncFileResult(True, "projec-id", "my/file", "Succeeded")


class FileSyncTest(BasicDBAppHTTPTests):
    def test_set_content_sync_state(self):
        project = self.create_project(LOCAL_PROJECT)
        data_json = project.data.to_json()

        there_is_no_synced_items = not re.search(r"\bsynced\b", data_json)
        assert there_is_no_synced_items
        assert project.set_content_sync_state("tmp/project/bar.txt", LocalStorageSyncState.SYNCED)
        data_json = project.data.to_json()
        assert LocalStorageSyncState.SYNCED in data_json

    def test_set_content_sync_should_return_false_if_key_does_not_match(self):
        project = self.create_project(LOCAL_PROJECT)
        assert not project.set_content_sync_state("tmp/project/does-not-exist.txt", LocalStorageSyncState.SYNCED)

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.list_cloud_project_files")
    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.diff")
    @mock.patch("aext_project_filebrowser_server.services.sync.get_headers")
    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.create_directory")
    @tornado.testing.gen_test
    async def test_sync_pull_project_files(self, mock_create_dir, mock_get_headers, mock_diff, *args, **kwargs):
        mock_get_headers.return_value = {}
        pull_ops = FileOperationsDict(
            {
                "tmp/project/foo.txt": FileOperations.LOCAL_WRITE,
                "tmp/project/bar.txt": FileOperations.LOCAL_WRITE,
                "tmp/project/images/image1.jpeg": FileOperations.LOCAL_WRITE,
            }
        )
        push_ops = {}
        mock_diff.return_value = DiffFileOperations(pull_ops, push_ops)
        project = self.create_project(LOCAL_PROJECT)
        await project_service.sync_pull_project_files(project.id, db=self.db)
        assert mock_create_dir.await_count == 5
        assert mock_diff.call_count == 1

    @tornado.testing.gen_test
    async def test_run_coroutine(self):
        futures = [
            SyncCoroutine(_test_coroutine, ("coro-test-1",), "coro-test-1"),
            SyncCoroutine(_test_coroutine, ("coro-test-2",), "coro-test-2"),
        ]
        results = await run_coroutine(futures)
        assert len(results) == 2

    @tornado.testing.gen_test
    async def test_create_directory(self):
        project = self.create_project(LOCAL_PROJECT)
        path = await local_storage.create_project_root_directory(project.id)
        assert os.path.exists(path)
        path = await local_storage.delete_project_directory(project.id)
        assert not os.path.exists(path)

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.list_cloud_project_files")
    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.diff")
    @tornado.testing.gen_test
    async def test_sync_push_project_files(self, mock_diff, mock_list_cloud_project_files):
        push_ops = DiffFileOperations(
            push_operations=FileOperationsDict(
                {"tmp/project/foo.txt": FileOperations.CLOUD_DELETE, "tmp/project/bar.txt": FileOperations.CLOUD_WRITE}
            ),
            pull_operations={},
        )
        mock_diff.return_value = push_ops
        project = self.create_project(LOCAL_PROJECT)
        await local_storage.create_project_root_directory(project.id)
        self.create_project_files(project)
        await project_service.sync_push_project_files(project.id, db=self.db)
        await local_storage.delete_project_directory(project.id)
        assert mock_diff.call_count == 1
        assert mock_list_cloud_project_files.call_count == 1

    @tornado.testing.gen_test
    async def test_add_project_files(self):
        # Setup: Create a new empty project and its root directory
        new_project = self.create_project(EMPTY_LOCAL_PROJECT)
        self.assertIsNotNone(new_project, "Failed to create a new project")

        await local_storage.create_project_root_directory(new_project.id)
        self.create_project_files(new_project)

        # Action: Add files to the project in the specified subfolder
        payload_1 = PAYLOAD_1.get("filenames", [])
        await project_service.upload_project_files(new_project.id, "./subfolder", payload_1)

        # Verification: Retrieve the updated project and check the contents
        updated_project = LocalProject.get(new_project.id)
        self.assertIsNotNone(updated_project, "Failed to retrieve the updated project")

        # Assertions: Check if the files were added correctly
        self.assertEqual(updated_project.data.contents[0].key, "subfolder")
        self.assertEqual(asdict(updated_project.data.contents[0]), EXPECTED_RESPONSE_1)

        # Action: Update files to a project - same files
        payload_2 = PAYLOAD_2.get("filenames", [])
        await project_service.upload_project_files(new_project.id, "./subfolder", payload_2)
        updated_project_2 = LocalProject.get(new_project.id)

        # Assertions: The file having just been overwritten, the structure remains the same.
        self.assertEqual(asdict(updated_project_2.data.contents[0]), EXPECTED_RESPONSE_1)

    @tornado.testing.gen_test
    async def test_add_project_files_path_traversal_attack(self):
        # Setup: Create a new empty project and its root directory
        new_project = self.create_project(EMPTY_LOCAL_PROJECT)
        self.assertIsNotNone(new_project, "Failed to create a new project")

        await local_storage.create_project_root_directory(new_project.id)
        self.create_project_files(new_project)

        # Action: Add files to the project in the specified subfolder
        payload_1 = PAYLOAD_1.get("filenames", [])
        await project_service.upload_project_files(new_project.id, "../../../var/www", payload_1)
        #
        # Verification: Retrieve the updated project and check the contents
        updated_project = LocalProject.get(new_project.id)
        self.assertIsNotNone(updated_project, "Failed to retrieve the updated project")

        # Assertions: Check if the files were added correctly
        self.assertEqual(updated_project.data.contents[0].key, "var", "subfolder key does not match")

        subfolder_contents = updated_project.data.contents[0].contents
        self.assertEqual(len(subfolder_contents), 1, "subfolder does not contain the expected number of files")

        # Create a dictionary to hold the expected file keys
        expected_files = {
            "var/www/conda-project.yaml": "File 'conda-project.yaml' is missing or its key is incorrect",
            "var/www/file.ipynb": "File 'file.ipynb' is missing or its key is incorrect",
        }

        for content in subfolder_contents[0].contents:
            self.assertIn(content.key, expected_files, f"Unexpected file key: {content.key}")
            del expected_files[content.key]

        # Ensure all expected files were found
        self.assertEqual(len(expected_files), 0, f"Missing files: {list(expected_files.keys())}")

    @tornado.testing.gen_test
    async def test_add_files_wrong_projects(self):

        # Action: Not project id
        project = LocalProject.get(None)

        # Assertion: it has to be None
        self.assertIsNone(project)

        # Action: wrong project id
        project = LocalProject.get("123")

        # Action: Add files to the wrong project
        response = await project_service.upload_project_files(123, "subfolder", PAYLOAD_1)
        self.assertIsNone(response)

    @mock.patch("aext_project_filebrowser_server.schemas.local_project.LocalProject.get")
    @tornado.testing.gen_test
    async def test_add_project_files_on_get_project_db_exception(self, mock_add_project_files):
        mock_add_project_files.side_effect = sqlite3.IntegrityError("Mock Integrity Error")

        # Action: Create a empty project
        project = self.create_project(EMPTY_LOCAL_PROJECT)

        # Action: Try to add files to the wrong project id
        with self.assertRaises(AddFileErrorOnDatabase):
            await project_service.upload_project_files(project.id, "subfolder", PAYLOAD_1)

    @mock.patch("aext_project_filebrowser_server.schemas.local_project.LocalProject.update")
    @tornado.testing.gen_test
    async def test_add_project_files_on_update_project_db_exception(self, mock_update_project_files):
        mock_update_project_files.side_effect = sqlite3.IntegrityError("Mock Integrity Error")

        # Action: Create a empty project
        project = self.create_project(EMPTY_LOCAL_PROJECT)

        # Action: Try to add files to the wrong project id
        with self.assertRaises(sqlite3.IntegrityError):
            await project_service.upload_project_files(project.id, "subfolder", PAYLOAD_1)

        get_project = LocalProject.get(project.id)

        # Assertion: Expect DB rollback and content show be empty for the project
        self.assertEqual(get_project.data.contents, [])

    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.add_files")
    @tornado.testing.gen_test
    async def test_add_project_files_on_save_files_exception(self, mock_local_storage_add_files):
        mock_local_storage_add_files.side_effect = AddFileErrorOnLocalStorage("Mock Permission denied")

        # Action: Create a empty project
        project = self.create_project(EMPTY_LOCAL_PROJECT)

        # Action: Try to add files to the wrong project id
        payload_1 = PAYLOAD_1.get("filenames", [])
        with self.assertRaises(AddFileErrorOnLocalStorage):
            await project_service.upload_project_files(project.id, "subfolder", payload_1)

        get_project = LocalProject.get(project.id)

        # Assertion: Expect DB rollback and content show be empty for the project
        self.assertEqual(get_project.data.contents, [])

    def test_get_status(self):
        # all statuses present - expect status = `sync_error`
        contents = ALL_SYNC_STATE_VARIATIONS_CONTENTS
        new_project = self.create_project_with_sync_data(str(uuid4()), f"project-{uuid4()}", contents)
        assert new_project.set_project_status() == LocalStorageSyncState.SYNC_ERROR

        # all statuses present except `sync_error` - expect status = `syncing`
        contents = ALL_SYNC_STATE_VARIATIONS_CONTENTS[1:]
        new_project = self.create_project_with_sync_data(str(uuid4()), f"project-{uuid4()}", contents)
        assert new_project.set_project_status() == LocalStorageSyncState.SYNCING

        # all statuses present except `sync_error` and `syncing` - expect status = `has_conflict`
        contents = ALL_SYNC_STATE_VARIATIONS_CONTENTS[2:]
        new_project = self.create_project_with_sync_data(str(uuid4()), f"project-{uuid4()}", contents)
        assert new_project.set_project_status() == LocalStorageSyncState.HAS_CONFLICT

        # only `unstaged` and `synced` present - expect status = `unstaged`
        contents = ALL_SYNC_STATE_VARIATIONS_CONTENTS[3:]
        new_project = self.create_project_with_sync_data(str(uuid4()), f"project-{uuid4()}", contents)
        assert new_project.set_project_status() == LocalStorageSyncState.UNSTAGED

        # only `synced` present - expect status = `synced`
        contents = ALL_SYNC_STATE_VARIATIONS_CONTENTS[4:]
        new_project = self.create_project_with_sync_data(str(uuid4()), f"project-{uuid4()}", contents)
        assert new_project.set_project_status() == LocalStorageSyncState.SYNCED

    @tornado.testing.gen_test
    async def test_sync_pull_delete_file(self):
        project = self.create_project(LOCAL_PROJECT)
        assert project.key_exists("tmp/project/foo.txt")
        await sync_pull_delete_file(project.id, "tmp/project/foo.txt")

        # refresh instance from DB
        project = LocalProject.get(project.id)
        assert project.key_exists("tmp/project/foo.txt") is False

    @tornado.testing.gen_test()
    async def test_sync_pull_delete_file_with_empty_contents(self):
        project = self.create_project(EMPTY_LOCAL_PROJECT)
        assert project.key_exists("tmp/project/foo.txt") is False

        await sync_pull_delete_file(project.id, "tmp/project/foo.txt")
        assert project.key_exists("tmp/project/foo.txt") is False

    @tornado.testing.gen_test()
    async def test_sync_pull_delete_file_with_one_folder_and_one_file(self):
        project = self.create_project(ONE_FOLDER_ONE_FILE_LOCAL_PROJECT)
        assert project.key_exists("tmp/bar.txt") is True

        await sync_pull_delete_file(project.id, "tmp/bar.txt")
        project = LocalProject.get(project.id)
        assert project.key_exists("tmp/bar.txt") is False
        assert project.key_exists("tmp/") is False


async def test_put_file_straight_to_s3(test_file, mock_cloud_request, mock_make_request):
    project_id = str(uuid4())
    user_creds = {
        "access_token": "123",
        "CF-Access-Client-Id": "CF-123",
        "CF-Access-Client-Secret": "CFS-123",
    }
    assert await put_file(project_id, "/tmp/file.txt", "file.txt", user_creds)
    assert mock_make_request.awaited_once
    mock_cloud_request.assert_has_awaits(
        [
            mock.call(
                f"projects/{project_id}/file-preload/file.txt",
                RequestMethods.PUT,
                user_creds,
                json_payload={"metadata": {}},
            ),
            mock.call(
                f"projects/{project_id}/file-versions/4d525af6-f8d7-4021-b22a-58979d90ec26",
                RequestMethods.PATCH,
                user_creds,
            ),
        ]
    )


@pytest.mark.parametrize("mock_cloud_request_status_code", [204, 400, 404, 403, 500])
async def test_put_file_straight_to_s3_should_return_none_if_request_to_projects_api_fails(
    test_file, mock_cloud_request, mock_make_request
):
    project_id = str(uuid4())
    user_creds = {
        "access_token": "123",
        "CF-Access-Client-Id": "CF-123",
        "CF-Access-Client-Secret": "CFS-123",
    }
    assert await put_file(project_id, "/tmp/file.txt", "file.txt", user_creds) is None
    assert mock_cloud_request.awaited_once
    mock_make_request.assert_not_awaited


@pytest.mark.parametrize("mock_make_request_status_code", [204, 400, 404, 403, 500])
async def test_put_file_straight_to_s3_should_return_none_if_request_to_s3_fails(
    test_file, mock_cloud_request, mock_make_request
):
    project_id = str(uuid4())
    user_creds = {
        "access_token": "123",
        "CF-Access-Client-Id": "CF-123",
        "CF-Access-Client-Secret": "CFS-123",
    }
    assert await put_file(project_id, "/tmp/file.txt", "file.txt", user_creds) is None
    assert mock_cloud_request.awaited_once
    mock_make_request.awaited_once
