import json
import os
import shutil
import sqlite3
import sys
from pathlib import Path
from unittest import mock
from uuid import uuid4

import pytest
import respx
import tornado
from aext_project_filebrowser_server.consts import PROJECTS_FILES_DIR, InternalOrigins
from aext_project_filebrowser_server.exceptions import (
    CloudProjectNameAlreadyExistsError,
)
from aext_project_filebrowser_server.metrics import metrics
from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProject,
    CloudProjectFileTree,
    CloudProjectPermissions,
    ObjectPermissionInfo,
    ProjectPermission,
)
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.schemas.projects import ProjectMetadata
from aext_project_filebrowser_server.utils import (
    create_empty_file,
    generate_normalized_name,
    get_latest_kernel_spec_name,
    is_file,
    sanitize_filename_path,
)
from httpx import Response
from tornado import httpclient

from aext_shared.backend_proxy import ProxyResponse
from aext_shared.config import get_config
from aext_shared.errors import BackendError

from .setup_tests import BasicAppHTTPTests, BasicDBAppHTTPTests
from .test_assets.project_data import (
    CLOUD_PROJECT,
    LIST_CLOUD_PROJECTS_RESPONSE,
    LOCAL_PROJECT,
    LOCAL_PROJECT_WITH_PYTHON_SNIPPET,
    TREE_4b6aa8af,
    TREE_593a2c7d,
)

config = get_config()

# Skip all tests if the platform is Windows
pytestmark = pytest.mark.skipif(sys.platform.startswith("win"), reason="Skipping tests on Windows")


class TestHealthzRouteHandler(BasicAppHTTPTests):
    def test_healthz_handler(self):
        headers = {"Authorization": "token abc"}
        response = self.fetch("/aext_project_filebrowser_server/healthz", headers=headers)
        assert response.code == 200
        expected = {"live": True}
        assert json.loads(response.body) == expected


class TestPullAllRouteHandler(BasicDBAppHTTPTests):
    @respx.mock
    def test_pull_all_handler_success(self):
        projects = [LocalProject.create(LOCAL_PROJECT)]
        json_response = [project.to_frontend() for project in projects]
        with mock.patch("aext_project_filebrowser_server.services.sync.ProjectSync.pull_all") as m:
            m.return_value = projects
            respx.get().respond(status_code=200, json=json_response)

            headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
            response = self.fetch("/aext_project_filebrowser_server/project/pull", headers=headers)

            assert response.code == 200
            assert json.loads(response.body) == json_response

    @mock.patch("aext_project_filebrowser_server.services.sync.get_username")
    def test_pull_all_projects_handler_should_remove_local_projects_if_cloud_does_not_have_any_projects(
        self, get_username
    ):
        owner_id = LOCAL_PROJECT["owner"]["id"]
        get_username.return_value = owner_id
        with mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.get_cloud_projects") as m:
            m.return_value = []

            project = self.create_project(LOCAL_PROJECT)
            self.create_project_files(project)
            headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
            response = self.fetch("/aext_project_filebrowser_server/project/pull", headers=headers)
            assert response.code == 200
            assert json.loads(response.body) == []
            assert LocalProject.list(self.db) == []

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.sync_pull_project_files")
    @mock.patch("aext_project_filebrowser_server.services.sync.get_username")
    def test_pull_all_projects_handler_apply_changes_when_the_project_exists_on_cloud(
        self, get_username, sync_pull_project_files
    ):
        """
        The test starts with a local project created but that is not stored in the cloud.
        The cloud does have two other projects.
        Therefor the test asserts that the local project is deleted and the other two present in
        the are created locally
        """
        owner_id = LOCAL_PROJECT["owner"]["id"]
        get_username.return_value = owner_id
        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.get_projects") as m:
            cloud_projects = LIST_CLOUD_PROJECTS_RESPONSE["items"]
            m.return_value = [CloudProject.create(project) for project in cloud_projects]
            project = self.create_project(LOCAL_PROJECT)
            self.create_project_files(project)
            cloud_project_1_id = "593a2c7d-1233-4649-af78-c224ad69f695"
            cloud_project_2_id = "4b6aa8af-b4af-459d-ae9d-19a969cdbb4c"
            with mock.patch(
                "aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.get_project_filetree"
            ) as m_filetree:

                def filetree_response(proj_id, **kwargs):
                    if proj_id == cloud_project_1_id:
                        return CloudProjectFileTree(**TREE_593a2c7d)
                    elif proj_id == cloud_project_2_id:
                        return CloudProjectFileTree(**TREE_4b6aa8af)

                m_filetree.side_effect = filetree_response

                headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
                response = self.fetch("/aext_project_filebrowser_server/project/pull", headers=headers)
                assert response.code == 200
                assert LocalProject.get(LOCAL_PROJECT["id"]) is None
                assert LocalProject.get(cloud_project_1_id)
                assert LocalProject.get(cloud_project_2_id)
                assert json.loads(response.body) == [
                    {
                        "id": "593a2c7d-1233-4649-af78-c224ad69f695",
                        "name": "99mb",
                        "title": "99MB",
                        "created_at": "2024-07-03T14:43:25.448694",
                        "updated_at": "2024-07-03T14:43:25.448694",
                        "metadata": {"user_client": "unknown", "hidden": False, "tags": ["data", "notebook"]},
                        "owner": {"id": "e761da58-732e-4943-af40-5d4caba41bfe", "type": "user", "name": None},
                        "contents": [
                            {
                                "key": "Screenshot 2024-07-08 at 10.36.53.png",
                                "name": "Screenshot 2024-07-08 at 10.36.53.png",
                                "type": "file",
                                "project_id": "593a2c7d-1233-4649-af78-c224ad69f695",
                                "contents": [],
                                "details": {
                                    "file_version_id": "7ca07abb-00e3-4af5-abe3-d0bef187bdf4",
                                    "metadata": {},
                                },
                                "status": "syncing",
                                "metadata": {},
                            },
                            {
                                "key": "tmp_99MB_file",
                                "name": "tmp_99MB_file",
                                "type": "file",
                                "project_id": "593a2c7d-1233-4649-af78-c224ad69f695",
                                "contents": [],
                                "details": {
                                    "file_version_id": "1d7c4b98-2ad4-49ae-adb9-86fab9190921",
                                    "metadata": {},
                                },
                                "status": "syncing",
                                "metadata": {},
                            },
                        ],
                        "default_environment": "undefined_environment",
                        "status": "syncing",
                    },
                    {
                        "id": "4b6aa8af-b4af-459d-ae9d-19a969cdbb4c",
                        "name": "150mb",
                        "title": "150MB",
                        "created_at": "2024-07-03T14:47:11.415080",
                        "updated_at": "2024-07-03T14:47:11.415080",
                        "metadata": {"user_client": "unknown", "hidden": False, "tags": ["data", "notebook"]},
                        "owner": {"id": "e761da58-732e-4943-af40-5d4caba41bfe", "type": "user", "name": None},
                        "contents": [
                            {
                                "key": "10mb.txt",
                                "name": "10mb.txt",
                                "type": "file",
                                "project_id": "4b6aa8af-b4af-459d-ae9d-19a969cdbb4c",
                                "contents": [],
                                "details": {
                                    "file_version_id": "e6b24960-69df-4c47-ad2c-7859602638dc",
                                    "metadata": {},
                                },
                                "status": "syncing",
                                "metadata": {},
                            },
                            {
                                "key": "tmp_10MB_file",
                                "name": "tmp_10MB_file",
                                "type": "file",
                                "project_id": "4b6aa8af-b4af-459d-ae9d-19a969cdbb4c",
                                "contents": [],
                                "details": {
                                    "file_version_id": "5a46e799-4b5b-4402-8c06-a222c3c30689",
                                    "metadata": {},
                                },
                                "status": "syncing",
                                "metadata": {},
                            },
                            {
                                "key": "tmp_10MB_file2",
                                "name": "tmp_10MB_file2",
                                "type": "file",
                                "project_id": "4b6aa8af-b4af-459d-ae9d-19a969cdbb4c",
                                "contents": [],
                                "details": {
                                    "file_version_id": "643607d3-f09c-484c-b004-a54050623bfd",
                                    "metadata": {},
                                },
                                "status": "syncing",
                                "metadata": {},
                            },
                        ],
                        "default_environment": "undefined_environment",
                        "status": "syncing",
                    },
                ]


class TestPullOneRouteHandler(BasicDBAppHTTPTests):
    @respx.mock
    def test_pull_one_handler_success(self):
        local_project = LocalProject.create(LOCAL_PROJECT)
        with mock.patch("aext_project_filebrowser_server.services.sync.ProjectSync.pull") as m:
            m.return_value = local_project
            respx.get().respond(status_code=200, json=local_project.to_frontend())

            headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
            response = self.fetch(f"/aext_project_filebrowser_server/project/pull/{uuid4()}", headers=headers)

            assert response.code == 200
            assert json.loads(response.body) == local_project.to_frontend()

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.sync_pull_project_files")
    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.get_cloud_project_filetree")
    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.get_cloud_project")
    def test_pull_one_handler_update_project_data(
        self, mock_get_cloud_project, mock_get_cloud_project_filetree, mock_sync_pull_project_files
    ):
        local_project = self.create_default_project()
        cloud_project = CloudProject.create(CLOUD_PROJECT)
        cloud_project.id = local_project.id

        mock_get_cloud_project.return_value = cloud_project
        mock_get_cloud_project_filetree.return_value = CloudProjectFileTree(cloud_project.id, stored_files=[])
        mock_sync_pull_project_files.return_value = []

        test_project = LocalProject.get(local_project.id)
        assert test_project.name == local_project.name
        assert test_project.data.title == local_project.data.title

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(f"/aext_project_filebrowser_server/project/pull/{local_project.id}", headers=headers)

        assert response.code == 200
        body = json.loads(response.body)
        cloud_to_local_project = cloud_project.to_local_project(
            mock_get_cloud_project_filetree.stored_files, default_environment=local_project.default_environment
        )
        self.assertEqual(body, cloud_to_local_project.to_frontend())

        test_project = LocalProject.get(local_project.id)
        self.assertEqual(test_project.name, cloud_project.name)
        self.assertEqual(test_project.data.title, cloud_project.title)
        self.assertEqual(test_project.default_environment, local_project.default_environment)


class TestDownloadProjectFileHandler(BasicAppHTTPTests):
    def test_download_file(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.download_file") as m:
            m.return_value = b"my file content"
            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{uuid4()}/download/file/test.txt", headers=headers
            )

            assert response.code == 200
            assert response.body == b"my file content"

    @respx.mock
    def test_download_should_return_500_if_download_fails(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        respx.get().respond(status_code=500, text="An error occurred")

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{uuid4()}/download/file/test.txt", headers=headers
        )

        assert response.code == 500


class TestProjectHandler(BasicDBAppHTTPTests):
    random_project_id = "84ece307-5fc1-4082-b35b-9f288e386596"

    def _request_data(self):
        headers = {"Authorization": "token abc"}
        json_response = {
            "id": "f843a721-a7b6-4630-a395-9292b5a0f93b",
            "name": "mytitles",
            "title": "MyTitles",
            "description": "foo bar",
            "created_at": "2024-04-17T10:53:46.263434",
            "updated_at": "2024-04-17T10:53:46.263434",
            "metadata": {"user_client": "unknown", "hidden": False, "tags": []},
            "owner": {"id": "e761da58-732e-4943-af40-5d4caba41bfe", "type": "user", "name": None},
            "default_environment": get_latest_kernel_spec_name(),
        }
        return headers, json_response

    def test_create_project(self):
        headers, json_response = self._request_data()
        environment = json_response.pop("default_environment")

        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.create_project") as m:
            m.return_value = CloudProject.create(json_response)
            response = self.fetch(
                "/aext_project_filebrowser_server/project",
                headers=headers,
                method="POST",
                body=json.dumps({"title": "MyTitle", "default_environment": environment}),
            )
            project = LocalProject.get(json_response["id"])
            response_data = json.loads(response.body)
            assert response.code == 201
            assert response_data["id"] == json_response["id"]
            assert project.default_environment == environment

    def test_create_project_db_uniqueness_owner_project_name(self):
        headers, json_response = self._request_data()
        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.create_project") as m:
            m.return_value = CloudProject.create(json_response)
            response = self.fetch(
                "/aext_project_filebrowser_server/project",
                headers=headers,
                method="POST",
                body='{"title": "MyTitle"}',
            )
            assert response.code == 201

            response = self.fetch(
                "/aext_project_filebrowser_server/project",
                headers=headers,
                method="POST",
                body='{"title": "MyTitle"}',
            )
            assert response.code == 409
            assert response.reason == "Project name already taken"

    def test_create_project_cloud_uniqueness_owner_project_name(self):
        headers, json_response = self._request_data()
        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.fetch") as m:
            m.side_effect = BackendError(
                status_code=200,
                data={
                    "reason": "proxy_bad_response",
                    "remote_status_code": 409,
                    "remote_data": {
                        "error": {
                            "code": "conflict",
                            "message": "Conflict",
                            "human_readable_id": "illinois-cup-fox-venus",
                        }
                    },
                },
            )
            response = self.fetch(
                "/aext_project_filebrowser_server/project",
                headers=headers,
                method="POST",
                body='{"title": "MyTitle"}',
            )
            assert response.code == 409
            assert response.reason == "Project name already taken"

    def test_create_project_title_is_mandatory(self):
        headers, json_response = self._request_data()
        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.create_project") as m:
            m.return_value = ProxyResponse(remote_status_code=201, remote_data=json_response)
            response = self.fetch(
                "/aext_project_filebrowser_server/project",
                headers=headers,
                method="POST",
                body='{"title": ""}',
            )
            assert response.code == 400
            assert json.loads(response.buffer.getvalue())["reason"] == "A project title must be provided"

    @mock.patch("aext_project_filebrowser_server.schemas.local_project.LocalProject.delete")
    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.delete_project_directory")
    def test_delete_project_success(self, mock_delete_proj_dir, mock_delete_project_from_db):
        headers, _ = self._request_data()
        with mock.patch(
            "aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.delete_project"
        ) as mock_delete_project:
            mock_delete_project.return_value = True
            response = self.fetch(
                "/aext_project_filebrowser_server/project/ae499a7b-afe6-4ace-b9e9-393a8d8bdd21",
                headers=headers,
                method="DELETE",
                body=None,
            )
            mock_delete_project.assert_awaited_once()
            mock_delete_proj_dir.assert_awaited_once()
            mock_delete_project_from_db.assert_called_once()
            assert response.code == 200

    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.delete_project_directory")
    @mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.delete_project")
    def test_delete_project_should_remove_from_db(self, mock_delete_project, *args):
        headers, _ = self._request_data()
        mock_delete_project.return_value = ProxyResponse(remote_status_code=204, remote_data=None)
        project = self.create_project(LOCAL_PROJECT)
        assert LocalProject.get(project.id, self.db)

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}",
            headers=headers,
            method="DELETE",
            body=None,
        )
        assert response.code == 200
        assert LocalProject.get(project.id, self.db) is None

    def test_delete_project_fail(self):
        headers, _ = self._request_data()
        with mock.patch("aext_project_filebrowser_server.clients.http.ProjectAPIClientHTTP.delete_project") as m:
            m.return_value = ProxyResponse(remote_status_code=400, remote_data=None)
            response = self.fetch(
                "/aext_project_filebrowser_server/project/f61fe192-wrong-uuid-f1e1ac175bc123",
                headers=headers,
                method="DELETE",
                body=None,
            )
            assert response.code == 400
            assert json.loads(response.buffer.getvalue())["reason"] == "Invalid project id"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_sync_push_should_return_404_if_project_does_not_exist(self, mock_authenticate):
        headers, _ = self._request_data()
        project_id = uuid4()
        with mock.patch("aext_project_filebrowser_server.schemas.local_project.LocalProject.get") as m:
            m.return_value = None
            response = self.fetch(
                f"/aext_project_filebrowser_server/project/push/{project_id}",
                headers=headers,
                method="GET",
            )
            print(response)
            print(response.body)
        assert response.code == 404
        assert response.reason == f"Project {project_id} not found"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_sync_push_should_return_404_in_case_there_are_no_projects(self, mock_authenticate):
        headers, _ = self._request_data()
        with mock.patch("aext_project_filebrowser_server.schemas.local_project.LocalProject.list") as m:
            m.return_value = []
            response = self.fetch(
                "/aext_project_filebrowser_server/project/push",
                headers=headers,
                method="GET",
            )
        assert response.code == 404
        assert response.reason == "There are no projects to be pushed"

    def test_update_project_fail_invalid_project_id(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            "/aext_project_filebrowser_server/project/invalid_id",
            headers=headers,
            method="PATCH",
            body=json.dumps({"invalid_key": "invalid_value"}),
        )

        assert response.code == 400

        body = json.loads(response.body)
        assert body["remote_status_code"] == 400
        assert body["reason"] == "Invalid project id"

    def test_update_project_fail_invalid_payload(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{self.random_project_id}",
            headers=headers,
            method="PATCH",
            body=json.dumps({"invalid_key": "invalid_value"}),
        )

        assert response.code == 400

        body = json.loads(response.body)
        assert body["remote_status_code"] == 400
        assert body["reason"] == "A project title or default environment must be provided"

    def test_update_project_fail_inexistent_project(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{self.random_project_id}",
            headers=headers,
            method="PATCH",
            body=json.dumps({"title": "new-title", "default_environment": "new-env"}),
        )

        assert response.code == 404

        body = json.loads(response.body)
        assert body["remote_status_code"] == 404
        assert body["reason"] == f"Unable to locate project {self.random_project_id} locally"

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.update_cloud_project")
    def test_update_project_fail_conflicting_title(self, mock_update_cloud_project):
        local_project_1 = self.create_default_project()

        local_project_2_title = "Project 2"
        local_project_2_data = {
            **LOCAL_PROJECT,
            "id": self.random_project_id,
            "title": local_project_2_title,
            "name": generate_normalized_name(local_project_2_title),
        }
        local_project_2 = self.create_project(local_project_2_data)
        mock_update_cloud_project.side_effect = CloudProjectNameAlreadyExistsError

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{local_project_2.id}",
            headers=headers,
            method="PATCH",
            body=json.dumps({"title": local_project_1.data.title}),
        )

        assert response.code == 409
        assert response.error.message == f"A project with title={local_project_1.data.title} already exists"

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.update_cloud_project")
    def test_update_project_title(self, mock_update_cloud_project):
        local_project = self.create_default_project()
        mock_update_cloud_project.return_value = local_project
        new_title = "new_title"

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{local_project.id}",
            headers=headers,
            method="PATCH",
            body=json.dumps({"title": new_title}),
        )

        body = json.loads(response.body)
        assert response.code == 200
        assert body["title"] == new_title

        local_project = LocalProject.get(local_project.id)
        assert local_project.name == generate_normalized_name(new_title)
        assert local_project.data.title == new_title

    @mock.patch("aext_project_filebrowser_server.handlers.list_kernel_specs_names")
    def test_update_project_default_environment(self, mock_list_kernel_specs_names):
        local_project = self.create_default_project()
        mock_list_kernel_specs_names.return_value = ["old_env", "new_env"]
        new_default_environment = "new_env"

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{local_project.id}",
            headers=headers,
            method="PATCH",
            body=json.dumps({"default_environment": new_default_environment}),
        )

        body = json.loads(response.body)
        assert response.code == 200
        assert body["default_environment"] == new_default_environment

        local_project = LocalProject.get(local_project.id)
        assert local_project.default_environment == new_default_environment

    @mock.patch("aext_project_filebrowser_server.handlers.list_kernel_specs_names")
    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.update_cloud_project")
    def test_update_project_title_and_default_environment(
        self, mock_update_cloud_project, mock_list_kernel_specs_names
    ):
        local_project = self.create_default_project()

        mock_update_cloud_project.return_value = local_project
        mock_list_kernel_specs_names.return_value = ["old_env", "new_env"]

        new_title = "new_title"
        new_default_environment = "new_env"

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{local_project.id}",
            headers=headers,
            method="PATCH",
            body=json.dumps({"title": new_title, "default_environment": new_default_environment}),
        )

        body = json.loads(response.body)
        assert response.code == 200
        assert body["title"] == new_title
        assert body["default_environment"] == new_default_environment

        local_project = LocalProject.get(local_project.id)
        assert local_project.name == generate_normalized_name(new_title)
        assert local_project.data.title == new_title
        assert local_project.default_environment == new_default_environment


class TestDeleteProjectFileHandler(BasicDBAppHTTPTests):

    @classmethod
    def _search_key(cls, contents, target_key):
        """
        Recursively search for a key in the contents.

        Args:
            contents (list): A list of items to search.
            target_key: The key to search for.

        Returns:
            bool: True if the target key is found, False otherwise.

        Notes:
            This function uses a recursive approach to traverse the contents
            and check each item's key. If the target key is found in any
            of the items, the function returns True.
        """

        def recurse(contents):
            for item in contents:
                if item.key == target_key:
                    return True
                if recurse(item.contents):
                    return True
            return False

        return recurse(contents)

    def setUp(self):
        super().setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        project_local_content = self.read_file(
            os.path.join(
                self.current_path,
                self.test_assets,
                "project-175bb011-b88b-4e04-b812-f0c2893aca05-local-content.json",
            )
        )
        insert_statement = (
            """INSERT INTO projects (id, data, owner, name, default_environment)
        VALUES ('175bb011-b88b-4e04-b812-f0c2893aca05', '%s', '3c01c0e2-1825-49c0-a371-ce9395c25f03', 'my-project', 'anaconda-kernel');
        """
            % project_local_content
        )
        self.db.execute(insert_statement)

        self.project_id = "175bb011-b88b-4e04-b812-f0c2893aca05"
        self.destination_project_dir = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")
        self.source_project_dir = Path(f"{self.current_path}/{self.test_assets}/projects/{self.project_id}")
        if self.destination_project_dir.exists() and self.destination_project_dir.is_dir():
            shutil.rmtree(self.destination_project_dir)
        shutil.copytree(self.source_project_dir, self.destination_project_dir)

    def test_delete_file_project_success(self):
        project = LocalProject.get(self.project_id)
        project_dir = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")

        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/level1/level2/level3/file-x.jpeg"]}),
        )

        # Make sure the File and parent directory is not there anymore
        project_modified = LocalProject.get(self.project_id)

        self.assertFalse(self._search_key(project_modified.data.contents, "file-x.jpeg"))
        self.assertFalse(self._search_key(project_modified.data.contents, "level1"))

        # file should not exist anymore
        removed_file_path = Path(f"{project_dir}/level1/level2/level3/file-x.jpeg")
        self.assertFalse(os.path.exists(removed_file_path))

        assert response.code == 201

    def test_delete_a_subfolder_with_files_success(self):
        project = LocalProject.get(self.project_id)
        project_dir = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")

        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/images/jpeg"]}),
        )

        # Make sure the File and parent directory is not there anymore
        project_modified = LocalProject.get(self.project_id)
        self.assertFalse(self._search_key(project_modified.data.contents, "image-a.jpeg"))
        self.assertFalse(self._search_key(project_modified.data.contents, "jpeg"))

        # directory should not exist anymore
        removed_folder_path = Path(f"{project_dir}/images/jpeg")
        self.assertFalse(os.path.exists(removed_folder_path))

        assert response.code == 201

    def test_delete_a_subsubfolder_with_files_success(self):
        project = LocalProject.get(self.project_id)
        project_dir = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")

        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/level1/level2/level3"]}),
        )

        # assert the original payload
        self.assertTrue(self._search_key(project.data.contents, "level1"))
        self.assertTrue(self._search_key(project.data.contents, "level1/level2"))
        self.assertTrue(self._search_key(project.data.contents, "level1/level2/level3"))
        self.assertTrue(self._search_key(project.data.contents, "level1/level2/level3/file-x.jpeg"))

        # assert the response
        project_modified = LocalProject.get(self.project_id)
        self.assertTrue(self._search_key(project_modified.data.contents, "level1"))
        self.assertTrue(self._search_key(project_modified.data.contents, "level1/level2"))
        self.assertFalse(self._search_key(project_modified.data.contents, "level1/level2/level3"))
        self.assertFalse(self._search_key(project_modified.data.contents, "level1/level2/level3/file-x.jpeg"))

        # folder should not exist anymore
        removed_file_path = Path(f"{project_dir}/level1/level2/level3")
        self.assertFalse(os.path.exists(removed_file_path))

        assert response.code == 201

    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.remove_object")
    def test_delete_a_folder_with_files_success(self, mock_local_storage_remove_object):
        project = LocalProject.get(self.project_id)

        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/images"]}),
        )

        # Make sure the File and parent directory is not there anymore
        project_modified = LocalProject.get(self.project_id)

        self.assertFalse(self._search_key(project_modified.data.contents, "image-a.jpeg"))
        self.assertFalse(self._search_key(project_modified.data.contents, "images"))

        assert response.code == 201
        assert mock_local_storage_remove_object.call_count == 1

    def test_delete_file_project_not_existed_success(self):
        project = LocalProject.get(self.project_id)

        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/images/does-not-exist.jpeg"]}),
        )
        # also should return a 201 case the file doesnt exist
        assert response.code == 201

    def test_delete_file_from_an_non_valid_project_success(self):

        headers = self._request_data()
        response = self.fetch(
            "/aext_project_filebrowser_server/project/123-does-not-exist/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/images/does-not-exist.jpeg"]}),
        )
        # also should return a 201 case the file doesnt exist
        assert response.code == 201

    @mock.patch("aext_project_filebrowser_server.handlers.metrics.send", new_callable=mock.AsyncMock)
    @mock.patch(
        "aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service",
        new_callable=mock.AsyncMock,
    )
    def test_delete_file_from_internal_origin(self, mock_authenticate, mock_send_metrics):
        user_credentials = {"client": InternalOrigins.WATCHDOG.value}
        mock_authenticate.return_value = user_credentials

        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        headers["X-Origin-Internal"] = InternalOrigins.WATCHDOG.value

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/delete",
            headers=headers,
            method="POST",
            body=json.dumps({"filenames": ["/images"]}),
        )

        assert response.code == 201
        mock_authenticate.assert_awaited_once_with(accept_internal_origin=True)
        mock_send_metrics.assert_awaited_once_with(user_credentials=user_credentials, metric_data=mock.ANY)

        # Make sure the File and parent directory is not there anymore
        project = LocalProject.get(self.project_id)
        self.assertFalse(self._search_key(project.data.contents, "image-a.jpeg"))
        self.assertFalse(self._search_key(project.data.contents, "images"))


class TestUpdateProjectFileHandler(BasicDBAppHTTPTests):
    def setUp(self) -> None:
        super().setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        project_local_content = self.read_file(
            os.path.join(
                self.current_path,
                self.test_assets,
                "project-175bb011-b88b-4e04-b812-f0c2893aca05-local-content.json",
            )
        )
        insert_statement = (
            """INSERT INTO projects (id, data, owner, name, default_environment)
            VALUES ('175bb011-b88b-4e04-b812-f0c2893aca05', '%s', '3c01c0e2-1825-49c0-a371-ce9395c25f03', 'my-project', 'anaconda-kernel');
            """
            % project_local_content
        )
        try:
            self.db.execute(insert_statement)
        except sqlite3.IntegrityError:
            pass

        self.project_id = "175bb011-b88b-4e04-b812-f0c2893aca05"
        self.project_root = f"{PROJECTS_FILES_DIR}/{self.project_id}"
        self.existing_file = f"{self.project_root}/images/jpeg/image-a.jpeg"
        self.existing_file_path, _ = os.path.split(self.existing_file)
        os.makedirs(self.existing_file_path, exist_ok=True)
        create_empty_file(self.existing_file)

    def tearDown(self):
        super().tearDown()
        try:
            shutil.rmtree(self.project_root)
        except Exception:
            pass

    def test_update_file_for_unknown_project_should_return_404(self):
        headers = self._request_data()
        response = self.fetch(
            "/aext_project_filebrowser_server/project/123-does-not-exist/file/update",
            headers=headers,
            method="POST",
            body=json.dumps({"file_relative_path": "images/does-not-exist.jpeg", "new_filename": "exists.jpeg"}),
        )
        assert response.code == 404
        assert str(response.error) == "HTTP 404: Project not found"

    def test_update_file_for_unknown_file_should_return_404(self):
        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{self.project_id}/file/update",
            headers=headers,
            method="POST",
            body=json.dumps({"file_relative_path": "images/does-not-exist.jpeg", "new_filename": "exists.jpeg"}),
        )
        assert response.code == 404
        assert str(response.error) == "HTTP 404: Local file not found"

    def test_update_file_with_same_name_should_return_file_already_exists_and_409(self):
        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{self.project_id}/file/update",
            headers=headers,
            method="POST",
            body=json.dumps({"file_relative_path": "images/jpeg/image-a.jpeg", "new_filename": "image-a.jpeg"}),
        )
        assert response.code == 409
        assert str(response.error) == "HTTP 409: The file image-a.jpeg already exists"

    def test_update_file_name(self):
        headers = self._request_data()
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{self.project_id}/file/update",
            headers=headers,
            method="POST",
            body=json.dumps({"file_relative_path": "images/jpeg/image-a.jpeg", "new_filename": "image-b.jpeg"}),
        )
        assert response.code == 204
        assert is_file(os.path.join(self.existing_file_path, "image-b.jpeg"))
        project = LocalProject.get(self.project_id)
        assert project.key_exists("images/jpeg/image-b.jpeg")
        assert project.key_exists("images/jpeg/image-a.jpeg") is False

    def test_update_folder_name(self):
        headers = self._request_data()
        new_folder_name = "jpg"
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{self.project_id}/file/update",
            headers=headers,
            method="POST",
            body=json.dumps({"file_relative_path": "images/jpeg", "new_filename": new_folder_name}),
        )
        assert response.code == 204
        assert is_file(f"{self.project_root}/images/{new_folder_name}/image-a.jpeg")
        project = LocalProject.get(self.project_id)
        assert project.key_exists(f"images/{new_folder_name}/image-b.jpeg")
        assert project.key_exists(f"images/{new_folder_name}/image-a.jpeg")
        assert project.key_exists(f"images/{new_folder_name}/misc/image-c.jpeg")


class TestProjectAddFilesHandler(BasicDBAppHTTPTests):

    def setUp(self):
        super().setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        project_local_content = self.read_file(
            os.path.join(
                self.current_path,
                self.test_assets,
                "project-175bb011-b88b-4e04-b812-f0c2893aca05-local-content.json",
            )
        )
        insert_statement = (
            """INSERT INTO projects (id, data, owner, name, default_environment)
        VALUES ('175bb011-b88b-4e04-b812-f0c2893aca05', '%s', '3c01c0e2-1825-49c0-a371-ce9395c25f03', 'my-project', 'anaconda-kernel');
        """
            % project_local_content
        )
        self.db.execute(insert_statement)

        self.project_id = "175bb011-b88b-4e04-b812-f0c2893aca05"
        self.destination_project_dir = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")
        self.source_project_dir = Path(f"{self.current_path}/{self.test_assets}/projects/{self.project_id}")
        self.assets_folder = Path(f"{self.current_path}/{self.test_assets}")

        if self.destination_project_dir.exists() and self.destination_project_dir.is_dir():
            shutil.rmtree(self.destination_project_dir)
        shutil.copytree(self.source_project_dir, self.destination_project_dir)

    def test_project_add_files_success(self):
        with mock.patch(
            "aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files"
        ) as mock_sync_push_project_files:
            project = LocalProject.get(self.project_id)
            headers = self._request_data()

            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{project.id}/file/add",
                headers=headers,
                method="POST",
                body=json.dumps(
                    {
                        "filename_paths": [
                            {"src_path": f"{self.assets_folder}/file_diff_1.txt", "dest_path": "./file_diff_1.txt"},
                            {"src_path": f"{self.assets_folder}/file_diff_2.txt", "dest_path": "./file_diff_2.txt"},
                        ]
                    }
                ),
            )

            self.assertEqual(response.code, 201)
            mock_sync_push_project_files.assert_not_awaited()

            for filename in ["file_diff_1.txt", "file_diff_2.txt"]:
                filename_path = os.path.join(self.destination_project_dir, filename)
                self.assertTrue(filename_path)

    @mock.patch("aext_project_filebrowser_server.utils.get_file_size")
    def test_project_add_too_big_files_should_return_413(self, mock_get_file_size):
        mock_get_file_size.return_value = 999999999999999
        project = LocalProject.get(self.project_id)
        headers = self._request_data()

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/add",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "filename_paths": [
                        {"src_path": f"{self.assets_folder}/file_diff_1.txt", "dest_path": "./file_diff_1.txt"},
                        {"src_path": f"{self.assets_folder}/file_diff_2.txt", "dest_path": "./file_diff_2.txt"},
                    ]
                }
            ),
        )
        self.assertEqual(response.code, 413)
        self.assertEqual(str(response.error), "HTTP 413: File file_diff_1.txt is too big")

    def test_add_project_files_with_non_existent_file(self):
        project = LocalProject.get(self.project_id)
        headers = self._request_data()

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/add",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "filename_paths": [
                        {"src_path": f"{self.assets_folder}/file_diff_1.txt", "dest_path": "./file_diff_1.txt"},
                        {"src_path": f"{self.assets_folder}/file_doesnt_exit.txt", "dest_path": "./file_diff_2.txt"},
                    ]
                }
            ),
        )

        self.assertEqual(response.code, 400)
        self.assertEqual(response.reason, "File file_doesnt_exit.txt does not exist")

    def test_add_project_files_with_project_not_found(self):
        headers = self._request_data()

        response = self.fetch(
            "/aext_project_filebrowser_server/project/11111111-1111-1111-1111-111111111111/file/add",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "filename_paths": [
                        {"src_path": f"{self.assets_folder}/file_diff_1.txt", "dest_path": "./file_diff_1.txt"},
                        {"src_path": f"{self.assets_folder}/file_diff_2.txt", "dest_path": "./file_diff_2.txt"},
                    ]
                }
            ),
        )

        self.assertEqual(response.code, 404)
        self.assertEqual(response.reason, "Project not found")

    def test_add_file_with_auto_push(self):
        with mock.patch(
            "aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files"
        ) as mock_sync_push_project_files:
            project = LocalProject.get(self.project_id)
            headers = self._request_data()

            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{project.id}/file/add?auto_push=true",
                headers=headers,
                method="POST",
                body=json.dumps(
                    {
                        "filename_paths": [
                            {"src_path": f"{self.assets_folder}/file_diff_1.txt", "dest_path": "./file_diff_1.txt"},
                            {"src_path": f"{self.assets_folder}/file_diff_2.txt", "dest_path": "./file_diff_2.txt"},
                        ]
                    }
                ),
            )

            assert response.code == 201
            mock_sync_push_project_files.assert_awaited_once_with(project.id)

            for filename in ["file_diff_1.txt", "file_diff_2.txt"]:
                filename_path = os.path.join(self.destination_project_dir, filename)
                self.assertTrue(filename_path)


class TestProjectCreateFilesHandler(BasicDBAppHTTPTests):

    def setUp(self):
        super().setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        project_local_content = self.read_file(
            os.path.join(
                self.current_path,
                self.test_assets,
                "project-175bb011-b88b-4e04-b812-f0c2893aca05-local-content.json",
            )
        )
        insert_statement = (
            """INSERT INTO projects (id, data, owner, name, default_environment)
            VALUES ('175bb011-b88b-4e04-b812-f0c2893aca05', '%s', '3c01c0e2-1825-49c0-a371-ce9395c25f03', 'my-project', 'anaconda-kernel');
            """
            % project_local_content
        )
        self.db.execute(insert_statement)

        self.project_id = "175bb011-b88b-4e04-b812-f0c2893aca05"

    @mock.patch(
        "aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.project_service.update_cloud_project"
    )
    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_create_notebook_file(self, mock_authenticate, mock_update_cloud_project):
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": "username",
        }
        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        filename = "test.ipynb"
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/create",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": f"./{filename}"}),
        )
        assert response.code == 200
        project = LocalProject.get(self.project_id)
        file_content = project.get_content_by_key(filename)
        assert file_content.metadata == {}

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.update_file_metadata")
    def test_create_file_should_return_500_and_delete_file_if_metadata_fails(
        self, mock_update_file_metadata, mock_authenticate
    ):
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": "username",
        }
        mock_update_file_metadata.return_value = False
        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        filename = "test.ipynb"
        metadata = {"metadata": {"test": 1}}

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/create",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": f"./{filename}", **metadata}),
        )
        assert response.code == 500
        project = LocalProject.get(self.project_id)
        assert project.get_content_by_key(filename) is None

    @mock.patch(
        "aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.project_service.update_cloud_project"
    )
    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_create_file_with_valid_code_snippets_metadata(self, mock_authenticate, mock_update_cloud_project):
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": "username",
        }
        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        filename = "snippet.py"
        metadata = {"metadata": {"test": 1, "code_snippet": {"language": "Python"}}}

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/create",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": f"./{filename}", **metadata}),
        )
        assert response.code == 200
        mock_update_cloud_project.assert_called_with(
            project.id, project_metadata=ProjectMetadata(user_client="foo", hidden=False, tags=["code_snippet"])
        )
        project = LocalProject.get(self.project_id)
        file_content = project.get_content_by_key(filename)
        assert file_content.metadata == metadata.get("metadata")

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_create_file_with_invalid_code_snippets_metadata_should_return_400(self, mock_authenticate):
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": "username",
        }
        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        filename = "test.ipynb"
        metadata = {"metadata": {"test": 1, "code_snippet": {"xpto": "Python"}}}

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/create",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": f"./{filename}", **metadata}),
        )
        assert response.code == 400
        assert response.reason == "Code snippets schema is invalid"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_create_file_with_auto_push(self, mock_authenticate):
        with mock.patch(
            "aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files"
        ) as mock_sync_push_project_files:
            mock_authenticate.return_value = {
                "access_token": "user-access-token",
                "new_access_token": "user-new-access-token",
                "api_key": "user-api-key",
                "username": "username",
            }
            project = LocalProject.get(self.project_id)
            headers = self._request_data()
            filename = "test.ipynb"

            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{project.id}/file/create?auto_push=true",
                headers=headers,
                method="POST",
                body=json.dumps({"filename": f"./{filename}"}),
            )
            assert response.code == 200
            mock_sync_push_project_files.assert_awaited_once_with(project.id)

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_create_file_without_auto_push(self, mock_authenticate):
        with mock.patch(
            "aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files"
        ) as mock_sync_push_project_files:
            mock_authenticate.return_value = {
                "access_token": "user-access-token",
                "new_access_token": "user-new-access-token",
                "api_key": "user-api-key",
                "username": "username",
            }
            project = LocalProject.get(self.project_id)
            headers = self._request_data()
            filename = "test.ipynb"

            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{project.id}/file/create?auto_push=false",
                headers=headers,
                method="POST",
                body=json.dumps({"filename": f"./{filename}"}),
            )
            assert response.code == 200
            mock_sync_push_project_files.assert_not_awaited()

    @mock.patch(
        "aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.project_service.update_cloud_project"
    )
    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_create_snippet_file(self, mock_authenticate, mock_update_cloud_project):
        with mock.patch(
            "aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files"
        ) as mock_sync_push_project_files:
            mock_authenticate.return_value = {
                "access_token": "user-access-token",
                "new_access_token": "user-new-access-token",
                "api_key": "user-api-key",
                "username": "username",
            }
            project = LocalProject.get(self.project_id)
            headers = self._request_data()
            filename = "snippet.py"
            metadata = {"metadata": {"code_snippet": {"language": "Python"}}}

            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{project.id}/file/create?auto_push=true",
                headers=headers,
                method="POST",
                body=json.dumps({"filename": f"./{filename}", **metadata}),
            )
            assert response.code == 200
            mock_update_cloud_project.assert_called_with(
                project.id, project_metadata=ProjectMetadata(user_client="foo", hidden=False, tags=["code_snippet"])
            )
            mock_sync_push_project_files.assert_awaited_once_with(project.id, [filename])

    @mock.patch("aext_project_filebrowser_server.handlers.metrics.send", new_callable=mock.AsyncMock)
    @mock.patch(
        "aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service",
        new_callable=mock.AsyncMock,
    )
    def test_create_file_from_internal_origin(self, mock_authenticate, mock_send_metrics):
        user_credentials = {"client": InternalOrigins.WATCHDOG.value}
        mock_authenticate.return_value = user_credentials

        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        headers["X-Origin-Internal"] = InternalOrigins.WATCHDOG.value
        filename = "test.ipynb"

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/create",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": f"./{filename}"}),
        )

        assert response.code == 200
        mock_authenticate.assert_awaited_once_with(accept_internal_origin=True)
        mock_send_metrics.assert_awaited_once_with(user_credentials=user_credentials, metric_data=mock.ANY)


class TestProjectPushFileHandler(BasicDBAppHTTPTests):
    def setUp(self):
        super().setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        project_local_content = self.read_file(
            os.path.join(
                self.current_path,
                self.test_assets,
                "project-175bb011-b88b-4e04-b812-f0c2893aca05-local-content.json",
            )
        )
        insert_statement = (
            """INSERT INTO projects (id, data, owner, name, default_environment)
            VALUES ('175bb011-b88b-4e04-b812-f0c2893aca05', '%s', '3c01c0e2-1825-49c0-a371-ce9395c25f03', 'my-project', 'anaconda-kernel');
            """
            % project_local_content
        )
        self.db.execute(insert_statement)

        self.project_id = "175bb011-b88b-4e04-b812-f0c2893aca05"

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files")
    def test_push_file_success(self, mock_sync_push_project_files):
        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        filename = "test.ipynb"

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/push",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": filename}),
        )

        assert response.code == 200
        mock_sync_push_project_files.assert_awaited_once_with(project.id, [sanitize_filename_path(filename)])

    def test_push_file_without_filename_should_return_400(self):

        project = LocalProject.get(self.project_id)
        headers = self._request_data()

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/push",
            headers=headers,
            method="POST",
            body=json.dumps({}),
        )

        assert response.code == 400
        assert response.reason == "Filename not provided"

    def test_push_file_with_invalid_project_should_return_404(self):

        invalid_project_id = "invalid-project-id"
        headers = self._request_data()
        filename = "test.ipynb"

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{invalid_project_id}/file/push",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": filename}),
        )

        assert response.code == 404
        assert response.reason == f"Local project (id={invalid_project_id}) not found"

    @mock.patch("aext_project_filebrowser_server.services.projects.ProjectService.sync_push_project_files")
    def test_push_file_from_internal_origin(self, mock_sync_push_project_files):

        project = LocalProject.get(self.project_id)
        headers = self._request_data()
        headers["X-Origin-Internal"] = "internal-service"
        filename = "test.ipynb"

        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/file/push",
            headers=headers,
            method="POST",
            body=json.dumps({"filename": filename}),
        )

        assert response.code == 200
        mock_sync_push_project_files.assert_awaited_once_with(project.id, [sanitize_filename_path(filename)])


class TestProjectListUserFilesHandler(BasicDBAppHTTPTests):
    def setUp(self):
        super().setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"

    @property
    def get_test_fake_user_path(self):
        # A property that returns a fake user home path
        return Path(f"{self.current_path}/{self.test_assets}/user_home")

    def test_list_user_files_success(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        user_home_path = str(self.get_test_fake_user_path)

        with mock.patch(
            "aext_project_filebrowser_server.services.filesystem_operations.USER_JLAB_HOME_PATH", new=user_home_path
        ):
            response = self.fetch(
                "/aext_project_filebrowser_server/jlab/files/list",
                headers=headers,
                method="GET",
            )

            # Parse the response body to JSON format
            response_json = json.loads(response.body)

            # Define a list of expected file paths
            expected_files = [
                "fakefile-100kb.bin",
                "my_notebook.ipynb",
                "test.csv",
                "test_folder/test2.csv",
                "folder_with_hidden_file/simple.ipynb",
            ]

            # Iterate through each file in the response JSON
            for each_file in response_json["files"]:
                # Assert that the file path in the response is in the list of expected files
                self.assertIn(each_file["path"], expected_files)

                # Construct the actual file path by combining the user home path with the file path from the response
                actual_file_path = Path(f"{user_home_path}/{each_file['path']}")

                # Assert that the constructed file path exists in the filesystem
                self.assertTrue(os.path.exists(actual_file_path))

            self.assertEqual(response.code, 201)


class TestProjectSharingPermissionsRouteHandler(BasicDBAppHTTPTests):
    def setUp(self):
        self.mocked_api = respx.mock(base_url=config["api"]["url"], assert_all_called=False)
        super().setUp()

    def test_list_project_permissions_should_return_404_if_project_does_not_exist(self):
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            "/aext_project_filebrowser_server/project/31511ce6-e65e-4e64-878b-2fc57e271b90/permissions?organizations=anacondainc",
            headers=headers,
            method="GET",
        )
        assert response.code == 404

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_list_project_permissions(self, mock_authenticate):

        project = self.create_project(LOCAL_PROJECT)
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": f"{project.owner}",
        }
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        with mock.patch(
            "aext_project_filebrowser_server.services.projects.ProjectService.list_project_permission",
        ) as m:
            m.return_value = CloudProjectPermissions(
                permissions=[
                    ProjectPermission(
                        type="user_id",
                        object=ObjectPermissionInfo(
                            id="e761da58-732e-4943-af40-5d4caba41bfe",
                            email="user@anaconda.com",
                            first_name="User",
                            last_name="Anaconda",
                        ),
                        roles=["owner"],
                    )
                ]
            )
            response = self.fetch(
                f"/aext_project_filebrowser_server/project/{project.id}/permissions?organizations=anacondainc",
                headers=headers,
                method="GET",
            )
            assert response.code == 200
            assert json.loads(response.body) == {
                "permissions": [
                    {
                        "type": "user_id",
                        "object": {
                            "id": "e761da58-732e-4943-af40-5d4caba41bfe",
                            "email": "user@anaconda.com",
                            "first_name": "User",
                            "last_name": "Anaconda",
                        },
                        "roles": ["owner"],
                    }
                ]
            }

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_list_project_permissions_should_return_403_for_non_owners(self, mock_authenticate):

        project = self.create_project(LOCAL_PROJECT)
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": "username",
        }
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/permissions?organizations=anacondainc",
            headers=headers,
            method="GET",
        )
        assert response.code == 403

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_update_project_permission_should_return_400_if_action_is_invalid(self, mock_authenticate):
        project = self.create_project(LOCAL_PROJECT)
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": f"{project.owner}",
        }
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/permissions",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "permissions": [
                        {
                            "permission": "writer",
                            "user_id": "e761da58-732e-4943-af40-5d4caba41bfe",
                            "user_email": "rrosa@anaconda.com",
                            "action": "addd",
                        }
                    ]
                }
            ),
        )
        assert response.code == 400
        assert response.reason == "Invalid action addd"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_update_project_permission_should_return_400_if_permission_is_invalid(self, mock_authenticate):
        project = self.create_project(LOCAL_PROJECT)
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": f"{project.owner}",
        }
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/permissions",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "permissions": [
                        {
                            "permission": "wwriter",
                            "user_id": "e761da58-732e-4943-af40-5d4caba41bfe",
                            "user_email": "rrosa@anaconda.com",
                            "action": "add",
                        }
                    ]
                }
            ),
        )
        assert response.code == 400
        assert response.reason == "Invalid permission wwriter"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_update_project_permissions_should_return_403_for_non_owners(self, mock_authenticate):
        project = self.create_project(LOCAL_PROJECT)
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": "username",
        }
        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/permissions",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "permissions": [
                        {
                            "permission": "writer",
                            "user_id": "e761da58-732e-4943-af40-5d4caba41bfe",
                            "user_email": "rrosa@anaconda.com",
                            "action": "add",
                        }
                    ]
                }
            ),
        )
        assert response.code == 403
        assert response.reason == "Only owners can manage permissions"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    def test_update_project_permissions(self, mock_authenticate):
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": LOCAL_PROJECT["owner"]["id"],
        }
        project = self.create_project(LOCAL_PROJECT)
        update_permission_response = Response(
            status_code=200,
            json={
                "items": [
                    {"type": "user_id", "relation": "finder", "id": "e761da58-732e-4943-af40-5d4caba41bfe"},
                    {"type": "user_id", "relation": "owner", "id": "e761da58-732e-4943-af40-5d4caba41bfe"},
                    {"type": "user_id", "relation": "reader", "id": "e761da58-732e-4943-af40-5d4caba41bfe"},
                ],
                "num_items": 3,
            },
        )
        organization_members_response = Response(
            status_code=200,
            json={
                "items": [
                    {
                        "id": "e761da58-732e-4943-af40-5d4caba41bfe",
                        "email": "user@anaconda.com",
                        "first_name": "User",
                        "last_name": "Anaconda",
                        "role": "admin",
                        "subscriptions": ["security_subscription"],
                        "groups": "null",
                    },
                    {
                        "id": "b51ba6a3-95c5-4895-b725-467d9f2f6013",
                        "email": "user+1@anaconda.com",
                        "first_name": "User+1",
                        "last_name": "Anaconda",
                        "role": "member",
                        "subscriptions": [],
                        "groups": "null",
                    },
                ],
                "total_count": 2,
                "filtered_count": 2,
            },
        )
        self.mocked_api.get("organizations/anacondainc/users").mock(side_effect=lambda r: organization_members_response)
        self.mocked_api.post(f"projects/{LOCAL_PROJECT['id']}/permissions").mock(
            side_effect=lambda r: update_permission_response
        )
        self.mocked_api.start()

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{project.id}/permissions",
            headers=headers,
            method="POST",
            body=json.dumps(
                {
                    "permissions": [
                        {
                            "permission": "reader",
                            "user_id": "e761da58-732e-4943-af40-5d4caba41bfe",
                            "user_email": "user+1@anaconda.com",
                            "action": "add",
                        }
                    ]
                }
            ),
        )
        assert response.code == 200
        assert response.reason == "Permissions updated"

    @mock.patch("aext_project_filebrowser_server.handlers.ProjectFileBrowserHandler.authenticate_project_service")
    @mock.patch("aext_project_filebrowser_server.metrics.FileBrowserSnakeEyes.send")
    def test_list_users_permissions_in_project(self, mock_metrics, mock_authenticate):
        mock_authenticate.return_value = {
            "access_token": "user-access-token",
            "new_access_token": "user-new-access-token",
            "api_key": "user-api-key",
            "username": LOCAL_PROJECT["owner"]["id"],
        }

        get_users_permission_response = Response(
            status_code=200,
            json={"own": True, "modify": True, "read": True, "share": True, "find": True, "delete": True},
        )
        self.mocked_api.get(f"projects/{LOCAL_PROJECT['id']}/permissions/my").mock(
            side_effect=lambda r: get_users_permission_response
        )
        self.mocked_api.start()

        headers = {"Authorization": "token abc", "Cookie": "access_token=xyz"}
        response = self.fetch(
            f"/aext_project_filebrowser_server/project/{LOCAL_PROJECT['id']}/permissions/my",
            headers=headers,
            method="GET",
        )
        assert response.code == 200
        assert response.reason == "User's permission"
        assert json.loads(response.body) == get_users_permission_response.json()
