import json
import os
import sqlite3.dbapi2
from dataclasses import asdict, dataclass, field, fields, is_dataclass
from datetime import datetime
from typing import Dict, List, Optional, Union

from aext_project_filebrowser_server.consts import (
    CODE_SNIPPET_TAG,
    PROJECTS_FILES_DIR,
    UNDEFINED_ENVIRONMENT,
    FileType,
    LocalStorageSyncState,
)
from aext_project_filebrowser_server.schemas.projects import (
    ProjectMetadata,
    ProjectOwner,
)
from aext_project_filebrowser_server.services.database import Database
from aext_project_filebrowser_server.services.database import db as database
from aext_project_filebrowser_server.utils import (
    get_max_priority,
    get_username,
    is_file,
)

from aext_shared import logger as custom_logger

logger = custom_logger.logger


@dataclass
class FileInfo:
    path: str
    size: str
    last_modified: str


@dataclass
class LocalProjectFileDetails:
    file_version_id: str
    metadata: dict = field(default_factory=dict)

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class LocalProjectContent:
    key: str
    name: str
    type: FileType
    project_id: str
    contents: List["LocalProjectContent"]
    details: Optional[LocalProjectFileDetails] = None
    status: LocalStorageSyncState = LocalStorageSyncState.UNSTAGED
    metadata: Dict = field(default_factory=dict)

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class LocalProjectData:
    name: str
    title: str
    created_at: datetime
    updated_at: datetime
    metadata: ProjectMetadata
    owner: ProjectOwner
    contents: List[LocalProjectContent]

    def to_json(self):
        """
        Converts a LocalProjectData instance into a json
        Returns: the json representation of LocalProjectData

        """
        return json.dumps(asdict(self))


@dataclass
class LocalProject:
    id: str
    name: str
    owner: str
    tags: List[str]
    default_environment: str = UNDEFINED_ENVIRONMENT
    data: Union[LocalProjectData, dict] = field(default_factory=dict)
    status: LocalStorageSyncState = LocalStorageSyncState.UNSTAGED

    def __post_init__(self):
        # calculate project status after instantiation
        self.status = self.set_project_status()

    @staticmethod
    def get_file_absolute_path(project_id: str, file_relative_path: str) -> str:
        """
        Returns file absolute path for a given project
        Args:
            project_id: id of the project
            file_relative_path: file relative path

        Returns: string containing the absolute path

        """
        return os.path.join(PROJECTS_FILES_DIR, project_id, file_relative_path)

    def data_to_json(self):
        if isinstance(self.data, dict):
            return json.dumps(self.data)
        return self.data.to_json()

    def to_json(self):
        """
        Converts a LocalProject dataclass object into json
        Returns: json representation of the project

        """
        return json.dumps(asdict(self))

    def calculate_project_folder_statuses(self) -> LocalStorageSyncState:
        """
        Calculate the status of all folders and subfolders based on the project files.

        This method recursively traverses the project's folder structure, calculates the status of each subfolder
        based on its contents (files and subfolders), and updates the parent folder status accordingly.
        It also calculates the project status based on the files and subfolders status.

        Args:
            self: The object instance. e.g.: LocalProject.data.contents

        Returns:
            A LocalStorageSyncState representing the overall project status, or None if the project is empty.
        """

        def set_folder_statuses(
            contents: list[LocalProjectContent],
            file_statuses: set[LocalStorageSyncState] = set(),
            folder_statuses: Dict = dict(),
            parent_dir: LocalProjectContent = None,
        ) -> LocalStorageSyncState:
            """
            Recursively calculate the status of a folder and its contents, considering files and subfolders.

            Args:
                contents (list[LocalProjectContent]): The list of child elements (files and folders).
                file_statuses (set[LocalStorageSyncState], optional): A set of file statuses to consider.
                folder_statuses (dict, optional): A dictionary tracking all folder statuses for debugging purposes.
                parent_dir (LocalProjectContent, optional): The parent directory in the recursive traversal.

            Returns:
                A LocalStorageSyncState representing the status of the folder.
            """
            for child in contents:
                if child.type == FileType.DIRECTORY:

                    # calculate the folder status based on the files inside this folder
                    folder_status = set_folder_statuses(child.contents, file_statuses, folder_statuses, child)

                    # compare the old folder status with the new status and set the highest one
                    # this is a courner case where we might have multiple folders on the same level
                    new_folder_status = folder_status
                    new_folder_status.add(child.status)
                    child.status = get_max_priority(new_folder_status)

                    # now update the parent folder statua
                    if parent_dir:
                        # compare the status of the old parent folder with the new one
                        new_parent_folder_status = set([parent_dir.status, child.status])
                        # Calculate the highest priority status
                        parent_dir.status = get_max_priority(new_parent_folder_status)

                        # track all folder status - just for debugging purpose
                        if parent_dir.key not in folder_statuses:
                            folder_statuses.update({parent_dir.key: [child.status]})
                        else:
                            folder_statuses[parent_dir.key].append(child.status)

                    # all files in a subfolder were scanned, now, just clean it for the next subfolder
                    file_statuses = set()

                else:
                    # create a set list of the status of all files in a subfolder
                    file_statuses.add(child.status)

            # check if the loop reached the end, if so, return project status, otherwise, keep the loop
            if not parent_dir:
                # match cases where there is not folders or end of the loop
                if child.key == contents[0].key or not folder_statuses:
                    return contents[0].status
            else:
                return file_statuses

        # if there is not content, then just return
        if not self.data.contents:
            return None

        project_status = set_folder_statuses(self.data.contents)
        return project_status

    def to_frontend(self):
        """
        Converts a LocalProject dataclass object into the schema expected by the frontend.

        Args:
            (None)

        Returns:
            The frontend schema for LocalProject
        """

        # calculate the status of all project folders on the fly
        project_status = self.calculate_project_folder_statuses()

        return {
            "id": self.id,
            "name": self.name,
            "title": self.data.title,
            "created_at": self.data.created_at,
            "updated_at": self.data.updated_at,
            "metadata": asdict(self.data.metadata),
            "owner": asdict(self.data.owner),
            "contents": [asdict(content) for content in self.data.contents],
            "default_environment": self.default_environment,
            "status": project_status,
        }

    def save(self, db: Optional[Database] = None) -> sqlite3.dbapi2.Cursor:
        """
        Save a LocalProject object into the database
        Args:
            db: an optional database driver

        Returns: the database cursor

        """
        if db is None:
            db = database

        query = """
        INSERT INTO projects (id, data, owner, name, default_environment, tags)
        VALUES (?, ?, ?, ?, ?, ?)
        """
        args = (self.id, self.data_to_json(), self.owner, self.name, self.default_environment, json.dumps(self.tags))
        return db.execute(query, args)

    def get_tags(self, db: Optional[Database] = None) -> List[str]:
        """
        Retrieve a project's tags

        Args: (None)

        Returns: a list of strings representing the project's tags
        """
        if db is None:
            db = database

        query = """
        SELECT tags FROM projects
        WHERE id = ?
        """
        args = (self.id,)
        cursor = db.execute(query, args)

        return json.loads(cursor.fetchall()[0][0])

    @classmethod
    def delete(cls, project_id: str, db: Optional[Database] = None):
        query = """
        DELETE FROM projects
        WHERE id = ?
        """
        if db is None:
            db = database
        args = (project_id,)
        return db.execute(query, args)

    @staticmethod
    def _create_content(
        project_id: str, project_contents: list, fallback_status: LocalStorageSyncState = LocalStorageSyncState.UNSTAGED
    ):
        """
        Given a project's contents list this function makes sure to instantiate ProjectContent objects
        and return a list containing the ProjectContent.
        Args:
            project_id: the id of the project
            project_contents: project's contents that is returned by the Projects API

        Returns: A list containing instances of ProjectContent

        """

        def get_metadata(item: Dict):
            # Metadata when comes from cloud is found in details but when this function is used
            # to create objects coming from the DB there is an especific metadata field
            if metadata := item.get("metadata"):
                return metadata
            elif details := item.get("details"):
                return details.get("metadata", {}) or {}
            return {}

        def create_content_recursively(_content: list):
            contents = []
            for item in _content:
                content_details = None
                if item["type"] == FileType.FILE and item["details"]:
                    details = item["details"]
                    content_details = LocalProjectFileDetails(
                        details.get("file_version_id"), details.get("metadata", {})
                    )
                project_content = LocalProjectContent(
                    name=item["name"],
                    key=item["key"],
                    type=item["type"],
                    project_id=project_id,
                    status=item.get("status", fallback_status),
                    details=content_details,
                    contents=[],
                    metadata=get_metadata(item),
                )
                contents.append(project_content)
                if project_content.type == FileType.DIRECTORY:
                    project_content.contents = create_content_recursively(item["contents"])

            return contents

        return create_content_recursively(project_contents)

    @classmethod
    def get(cls, project_id: str, db: Optional[Database] = None) -> Optional["LocalProject"]:
        """
        Retrieves a project from the database give it's id
        Args:
            project_id: id of the project
            db: an option database driver

        Returns:

        """
        if db is None:
            db = database

        query = """
            SELECT * FROM projects WHERE id=?
        """
        args = (project_id,)
        db.execute(query, args)
        row = db.fetchone()

        if not row:
            return None

        project_data = json.loads(row["data"])
        data_content = project_data.get("contents", [])
        project_content = cls._create_content(project_id, data_content)

        owner = ProjectOwner(**project_data["owner"])
        metadata = ProjectMetadata(**project_data["metadata"])

        return cls(
            id=row["id"],
            name=row["name"],
            owner=owner.id,
            default_environment=row["default_environment"],
            tags=json.loads(row["tags"]),
            data=LocalProjectData(
                name=project_data["name"],
                title=project_data["title"],
                created_at=project_data["created_at"],
                updated_at=project_data["updated_at"],
                metadata=metadata,
                owner=owner,
                contents=project_content,
            ),
        )

    @classmethod
    def create(cls, project_data: Dict) -> "LocalProject":
        """
        Given the project data returned by the Projects service this function creates a new LocalProject
        object and saves it into the database.
        Args:
            project_data: LocalProject that returned by the projects api

        Returns: a LocalProject instance

        """
        owner = ProjectOwner(**project_data["owner"])
        metadata = ProjectMetadata(**project_data["metadata"])
        contents = cls._create_content(project_data["id"], project_data.get("contents", []))

        data = LocalProjectData(
            contents=contents,
            created_at=project_data["created_at"],
            updated_at=project_data["updated_at"],
            metadata=metadata,
            owner=owner,
            title=project_data["title"],
            name=project_data["name"],
        )

        return LocalProject(
            id=project_data["id"],
            name=project_data["name"],
            owner=owner.id,
            default_environment=project_data["default_environment"],
            tags=metadata.tags,
            data=data,
        )

    def update(self, db: Optional[Database] = None, auto_commit: bool = True):
        """
        Updates a project information in the database
        Args:
            db: a database driver
            auto_commit: auto commit the updated

        Returns:

        """
        if db is None:
            db = database
        sql = """
            UPDATE projects
            SET data = ?, owner = ?, name = ?, default_environment = ?
            WHERE id = ?;
        """
        args = (json.dumps(asdict(self.data)), self.owner, self.name, self.default_environment, self.id)
        return db.execute(sql, args, auto_commit)

    def replace_data_value(self, search_key, new_value) -> bool:
        """
        Search for a key in the data dictionary and replace its value with a new value.

        Args:
            search_key (str): The key to search for.
            new_value: The new value to replace the existing value with.

        Returns:
            bool: True if the key was found and its value was replaced, False otherwise.
        """

        def replace_value_recursively(data, search_key, new_value) -> bool:
            """
            Helper function to recursively search for a key and replace its value.
            """
            for key, value in data.items():
                if key == search_key:
                    data[key] = new_value
                    return True
                elif isinstance(value, dict):
                    if replace_value_recursively(value, search_key, new_value):
                        return True
            return False

        # Call the helper function to search for and replace the value
        return replace_value_recursively(self.data, search_key, new_value)

    def get_content_by_key(
        self, search_key: str, data_content: Optional[LocalProjectContent] = None
    ) -> Optional[LocalProjectContent]:
        """
        Returns the data related to the search key
        Args:
            search_key: key that will be searched in the project
            data_content: the data related to the key

        Returns: LocalProjectContent

        """
        if not data_content:
            data_content = self.data.contents

        def get_content_by_key_recursively(_search_key, _data_content):
            for item in _data_content:
                if item.key == _search_key:
                    return item
                elif item.type == FileType.DIRECTORY:
                    if content := get_content_by_key_recursively(search_key, item.contents):
                        return content

        return get_content_by_key_recursively(search_key, data_content)

    def key_exists(self, search_key: str) -> bool:
        """
        Search for a key in the data dictionary

        Args:
            search_key (str): The key to search for.

        Returns:
            bool: True if the key was found, False otherwise.
        """
        # Call the helper function to search for the value key
        return bool(self.get_content_by_key(search_key, self.data.contents))

    def update_data_value_in_a_folder(self, search_key: str, new_obj: LocalProjectContent):
        """
        Search for a key in the data dictionary and update a data value

        Args:
            search_key (str): The key to search for.
            new_obj (LocalProject): object to be updated

        Returns:
            bool: True if the key was found, False otherwise.
        """

        def update_object_by_key_recursively(
            data: LocalProjectData, search_key: str, new_obj: LocalProjectContent
        ) -> bool:
            for item in data:
                if item.key == search_key:
                    item = new_obj
                    return True
                elif isinstance(item.contents, list):
                    if update_object_by_key_recursively(item.contents, search_key, new_obj):
                        return True
            return False

        # Call the helper function to search for the value key
        return update_object_by_key_recursively(self.data.contents, search_key, new_obj)

    def add_data_value_in_a_folder(self, search_key: str, new_obj: LocalProjectContent):
        """
        Search for a directory (key) in the data field and add a new object value

        Args:
            search_key (str): The key to search for.
            new_obj (LocalProjectContent): object to be added

        Returns:
            bool: True if the key was found, False otherwise.
        """

        def add_object_recursively(data: LocalProjectData, search_key: str, new_obj: LocalProjectContent) -> bool:
            # if means the new object will be added on the top of the list
            # e.g.: root project folder
            if not search_key:
                data.append(new_obj)
                return True

            for item in data:
                if item.key == search_key:
                    item.contents.append(new_obj)
                    return True
                elif isinstance(item.contents, list):
                    if add_object_recursively(item.contents, search_key, new_obj):
                        return True
            return False

        # Call the helper function to search for the value key
        return add_object_recursively(self.data.contents, search_key, new_obj)

    def set_content_sync_state(self, content_key: str, new_state: LocalStorageSyncState) -> bool:
        """
        Given a content key this function looks for it and set a new state to the content (file/dir)
        Args:
            content_key: A key identifying the content within the project
            new_state: the new state of the content

        Returns: boolean reporting if the operation succeeded or not.

        """

        contents = self.data.contents or []

        def set_state_recursively(contents, content_key, new_state) -> bool:
            for content in contents:
                if content.key == content_key:
                    content.status = new_state
                    return True
                if content.type == FileType.DIRECTORY:
                    if set_state_recursively(content.contents, content_key, new_state):
                        return True
            return False

        return set_state_recursively(contents, content_key, new_state)

    def update_file_version_id(self, file_key: str, file_version_id: str) -> bool:
        """
        Given a content key this function looks for it and set a new state to the content (file/dir)
        Args:
            file_key: A key identifying the file(content) within the project
            file_version_id: new file version id

        Returns: boolean reporting if the operation succeeded or not.

        """

        contents = self.data.contents or []

        def set_file_version_id_recursively(contents: List[LocalProjectContent], content_key, file_version_id) -> bool:
            for content in contents:
                if content.key == content_key:
                    if content.details is None:
                        content.details = LocalProjectFileDetails(file_version_id=file_version_id)
                    else:
                        content.details.file_version_id = file_version_id
                    return True
                if content.type == FileType.DIRECTORY:
                    if set_file_version_id_recursively(content.contents, content_key, file_version_id):
                        return True
            return False

        return set_file_version_id_recursively(contents, file_key, file_version_id)

    # NOTE: It might be deprecated in favor of update_project_folder_statuses()
    def set_project_status(self) -> LocalStorageSyncState:
        """
        Calculates the project's status based on its contents.

        Checks to assess the status are done in the following priority:
            1. SYNC_ERROR
            2. SYNCING
            3. HAS_CONFLICT
            4. UNSTAGED
            5. SYNCED

        Args:
            contents: an instance of schemas.local_project.LocalProjectContent.

        Returns:
            LocalStorageSyncState: the state of the project.
        """

        # TODO: set states for directories recursively
        # TODO: calculate project status based on directory
        def get_states_recursively(
            contents: list[LocalProjectContent], file_states: set[LocalStorageSyncState]
        ) -> set[LocalStorageSyncState]:
            for content in contents:
                if content.type == FileType.DIRECTORY:
                    get_states_recursively(content.contents, file_states)
                file_states.add(LocalStorageSyncState(content.status))
            return file_states

        file_states = get_states_recursively(self.data.contents, set())

        # check for errors first
        # then check if syncing is done
        # if so, check if conflict exists
        # if syncing is done and no conflict, check if there are any unstaged files
        # if none of the above are valid, the project is synced with cloud
        # TODO: improve algorithm to determine project status
        status_priority = LocalStorageSyncState.list_values()

        status_counts = {status: 0 for status in status_priority}
        for status in file_states:
            status_counts[status.value] += 1

        for status in status_priority:
            if status_counts[status] > 0:
                return status

        return LocalStorageSyncState.SYNCED

    @classmethod
    def create_from_db_data(cls, db_data: dict) -> "LocalProject":
        project_data = json.loads(db_data["data"])

        owner = ProjectOwner(**project_data["owner"])
        metadata = ProjectMetadata(**project_data["metadata"])

        data = LocalProjectData(
            name=project_data["name"],
            title=project_data["title"],
            created_at=project_data["created_at"],
            updated_at=project_data["updated_at"],
            metadata=metadata,
            owner=owner,
            contents=cls._create_content(db_data["id"], project_data.get("contents", [])),
        )

        return LocalProject(
            id=db_data["id"],
            name=db_data["name"],
            owner=owner.id,
            default_environment=db_data["default_environment"],
            tags=json.loads(db_data["tags"]),
            data=data,
        )

    @classmethod
    def list(cls, db: Optional[Database] = None, owner_id: Optional[str] = None) -> List["LocalProject"]:
        if db is None:
            db = database

        args = ()
        query = """
                SELECT * from projects
            """
        if owner_id:
            args = (owner_id,)
            query += " where owner = ?"

        db.execute(query, args)
        rows = db.fetchall()
        return cls._instantiate_projects(rows)

    @classmethod
    def _instantiate_projects(cls, rows: List):
        projects = []
        if not rows:
            return projects
        for row in rows:
            project = cls.create_from_db_data(row)
            projects.append(project)
        return projects

    @classmethod
    def list_owned_projects(cls, db: Optional[Database] = None) -> List["LocalProject"]:
        user_id = get_username()
        return cls.list(db=db, owner_id=user_id)

    def list_files(self) -> List[LocalProjectContent]:
        """
        Returns a flat list of files
        Returns: a list of files

        """
        file_list = []

        def list_recursively(data: List[LocalProjectContent], _file_list: list) -> List[LocalProjectContent]:
            item: LocalProjectContent
            for item in data:
                if item.type == FileType.FILE:
                    _file_list.append(item)
                elif item.type == FileType.DIRECTORY:
                    list_recursively(item.contents, _file_list)
            return _file_list

        # Call the helper function to search for the value key
        return list_recursively(self.data.contents, file_list)

    def add_file(self, file_relative_path: str, db_save=True) -> bool:
        """
        Adds a single file from the local database.

        Args:
            file_relative_path: relative path to the file that needs to be added
            db_save: save the changes also to the database

        Returns: boolean representing success/failure of operation

        """
        # file does not exist on disk
        file_absolute_path = LocalProject.get_file_absolute_path(self.id, file_relative_path)
        if not is_file(file_absolute_path):
            return False

        # file already exists in db
        filekey_list = [item.key for item in self.list_files()]
        if file_relative_path in filekey_list:
            return False

        parent_dir, filename = os.path.split(file_relative_path)
        file_content = LocalProjectContent(
            name=filename,
            key=file_relative_path,
            type=FileType.FILE,
            status=LocalStorageSyncState.UNSTAGED,
            contents=[],
            project_id=self.id,
        )

        self.add_data_value_in_a_folder(parent_dir, file_content)

        if db_save:
            self.update()

        return True

    def remove_file_or_folder(self, relative_path: str, db_save=False) -> bool:
        """
        Removes a file or a whole folder from the local database.

        Args:
            relative_path: relative path to the file/folder that needs to be deleted
            db_save: save the changes also to the database

        Returns: boolean representing success/failure of operation

        """

        def _remove_file_recursively(project_contents: List[LocalProjectContent], leaf_dir: LocalProjectContent = None):
            content: LocalProjectContent
            for content in project_contents:
                if content.type == FileType.FILE and content.key == relative_path:
                    if leaf_dir:
                        leaf_dir.contents.remove(content)
                    else:
                        # remove file from the root project folder
                        project_contents.remove(content)
                    return True
                elif content.type == FileType.DIRECTORY:

                    # it will remove the whole directory and files
                    if content.key == relative_path:

                        # case leaf is not empty, then remove the whole content
                        if leaf_dir is not None:
                            leaf_dir.contents.remove(content)

                        # case leaf if empty, this means we are on the root folder,
                        # then just remove the content
                        else:
                            self.data.contents.remove(content)

                    # it will remove a file or directory in a subdirectory
                    if _remove_file_recursively(content.contents, content):
                        if is_directory_is_now_empty := not bool(len(content.contents)):
                            if leaf_dir is not None:
                                leaf_dir.contents.remove(content)
                            else:
                                # Remove the parent folder if the parent_dir is empty
                                self.data.contents.remove(content)
                            return is_directory_is_now_empty

        if not self.key_exists(relative_path):
            # file does not exist in the project
            return False

        _remove_file_recursively(self.data.contents)
        if db_save:
            self.update()

        return True

    def update_files_status(self, status: LocalStorageSyncState, db_save=False):
        def set_state_recursively(_contents, status):
            for _content in _contents:
                _content.status = status
                if _content.type == FileType.DIRECTORY:
                    set_state_recursively(_content.contents, status)

        contents = self.data.contents
        set_state_recursively(contents, status)

        if db_save:
            self.update()

    @classmethod
    def get_default_environment(cls, project_id: str) -> str:
        """
        Retrieves a project's default environment.

        Args:
            project_id: the id of the project.

        Returns:
            a string representing the project's default environment if it exists;
            `UNDEFINED_ENVIRONMENT` otherwise.

        """
        local_project = cls.get(project_id)

        default_environment = UNDEFINED_ENVIRONMENT
        if local_project:
            default_environment = local_project.default_environment

        return default_environment

    def is_owned_by(self, user_id: str) -> bool:
        return user_id == self.owner

    @classmethod
    def filter_metadata(
        cls, metadata_field: str, field_value: str, db: Optional[Database] = None
    ) -> List["LocalProject"]:
        """
        Filters projects based on a metadata field

        Args:
            metadata_field: the field that will be used to filter the metadata
            field_value: the value that should be matched
            db: the database driver
        Returns: A list of LocalProjec
        """
        if db is None:
            db = database

        args = ()
        query = f"""
            WITH RECURSIVE content_tree AS (
            -- Base case: extract all top-level contents from each row
            SELECT
                projects.rowid AS project_rowid,
                json_each.value AS content,
                json_each.value->>'$.metadata.{metadata_field}' AS _field
            FROM
                projects,
                json_each(projects.data, '$.contents')

            -- Recursive case: find nested contents within the current content
            UNION ALL

            SELECT
                content_tree.project_rowid,
                json_each.value AS content,
                json_each.value->>'$.metadata.{metadata_field}' AS _field
            FROM
                content_tree,
                json_each(content_tree.content, '$.contents')
            )
            SELECT DISTINCT projects.*
            FROM projects
            JOIN content_tree ON projects.rowid = content_tree.project_rowid
            WHERE LOWER(content_tree._field) = '{field_value}';

        """
        logger.info(query)
        db.execute(query, args)
        rows = db.fetchall()
        logger.info(rows)
        return cls._instantiate_projects(rows)

    def contains_snippet_files(self) -> bool:
        """
        Checks whether a local project contains any code snippets

        Args: (None)

        Returns: bool
        """

        def check_for_code_snippets_recursively(contents: LocalProjectContent):
            for item in contents:
                if item.type == FileType.DIRECTORY:
                    return check_for_code_snippets_recursively(item.contents)
                elif CODE_SNIPPET_TAG in item.metadata:
                    return True
            return False

        return check_for_code_snippets_recursively(self.data.contents)


@dataclass
class CodeSnippetMetadata:
    language: str

    def validate(self) -> bool:
        data = asdict(self)
        # Get all the field names of the dataclass
        dataclass_fields = {f.name for f in fields(self)}

        # check for extra or missing keys
        if dataclass_fields != set(data.keys()):
            return False

        # validate nested dataclass fields
        for _field in fields(self):
            field_value = data.get(_field.name)
            if is_dataclass(_field.type):
                try:
                    nested_instance = _field.type(**field_value)
                except TypeError:
                    return False
                if not nested_instance.validate(field_value):
                    return False
        return True
