import os
import shutil
import sqlite3
from typing import Dict, List, Optional, Tuple

from aext_project_filebrowser_server.consts import (
    PROJECTS_FILES_DIR,
    FileOperations,
    FileType,
    LocalStorageSyncState,
)
from aext_project_filebrowser_server.exceptions import (
    AddFileErrorOnLocalStorage,
    FailedToCopyProjectFiles,
    LocalFileAlreadyExists,
    LocalFileNotFound,
    LocalFilePermissionError,
    LocalProjectNotFoundError,
    ProjectFailList,
    ProjectFolderNotFound,
)
from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProjectFile,
    CloudProjectFileList,
)
from aext_project_filebrowser_server.schemas.local_project import (
    LocalProject,
    LocalProjectContent,
    LocalProjectData,
)
from aext_project_filebrowser_server.services.conflict_resolution import FileDiff
from aext_project_filebrowser_server.services.database import db
from aext_project_filebrowser_server.utils import (
    file_exists,
    get_intermediate_directories,
    is_dir,
    sanitize_directory_path,
    sanitize_filename_path,
)

from aext_shared import logger as custom_logger

logger = custom_logger.logger


class FileOperationsDict(dict):
    def filter_by_operation_type(self, operation_type: FileOperations):
        return {k: v for k, v in self.items() if v == operation_type}

    def _contains(self, target: str):
        return {k: v for k, v in self.items() if target in v.value}

    def filter_local_operations(self):
        return self._contains("local_")

    def filter_cloud_operations(self):
        return self._contains("cloud_")


class DiffFileOperations:
    def __init__(self, pull_operations: FileOperationsDict, push_operations: FileOperationsDict):
        self.pull_operations = pull_operations
        self.push_operations = push_operations


class LocalStorage:
    async def diff(self, project_id: str, cloud_file_list: CloudProjectFileList) -> DiffFileOperations:
        """
        Compares the local project with the cloud project and returns a list of actions that should
        be executed for the PUSH and PULL operations.
        Args:
            project_id: id of the project
            cloud_file_list: project's list of file on the cloud

        Returns: a tuple where the first element (push_operations) contains the list of
        actions the PUSH operation must run. The second element (pull_operations) contain the list
        of actions that the PULL operation must execute.

        """
        push_operations = FileOperationsDict()
        pull_operations = FileOperationsDict()
        cloud_file: CloudProjectFile
        for cloud_file in cloud_file_list.items:
            # iterates over the list of files present on the cloud - this means an iteration over
            # local files should also be performed to see if there are files present locally
            # but not on the cloud
            file_relative_path = cloud_file.name
            file_absolute_path = LocalProject.get_file_absolute_path(project_id, file_relative_path)
            try:
                are_equal = await FileDiff.are_files_equal(file_absolute_path, cloud_file.etag)
                if not are_equal:
                    # Exists on both CLOUD and LOCAL but the content is different
                    # if PUSH -> WRITE file on the CLOUD with the new content
                    # if PULL -> WRITE file LOCALLY with the new content
                    push_operations[file_relative_path] = FileOperations.CLOUD_WRITE
                    pull_operations[file_relative_path] = FileOperations.LOCAL_WRITE
            except LocalFileNotFound:
                # Exists on the CLOUD but not LOCAL
                # if PUSH -> DELETE CLOUD
                # if PULL -> CREATE LOCAL
                push_operations[file_relative_path] = FileOperations.CLOUD_DELETE
                pull_operations[file_relative_path] = FileOperations.LOCAL_WRITE

        # So far we already covered files that belong to both and files that are present
        # on the CLOUD but not LOCALLY
        # The goal now is to find file that are present LOCALLY but not on CLOUD

        local_project = LocalProject.get(project_id)
        local_file_list = local_project.list_files()
        local_file_set = {file.key for file in local_file_list}
        cloud_file_set = {file.name for file in cloud_file_list.items}
        files_present_only_locally = local_file_set - cloud_file_set

        for local_file in files_present_only_locally:
            # Exists LOCAL but not on the CLOUD
            # if PUSH -> CREATE CLOUD
            # if PULL -> DELETE LOCAL
            push_operations[local_file] = FileOperations.CLOUD_WRITE
            pull_operations[local_file] = FileOperations.LOCAL_DELETE

        return DiffFileOperations(pull_operations, push_operations)

    def _append_status_to_filetree(self, content: LocalProjectContent, status: LocalStorageSyncState) -> Dict:
        if not content:
            return

        for item in content:
            item["status"] = status
            self._append_status_to_filetree(item["contents"], status)

        return content

    async def list_projects(self) -> List["LocalProject"]:
        try:
            return LocalProject.list()
        except sqlite3.IntegrityError as ex:
            logger.error("Not possible to list the projects due DB integrity error")
            raise ProjectFailList from ex

    async def create_directory(self, project_id: str, dir_name: str, base_dir: str = PROJECTS_FILES_DIR) -> str:
        """
        Create a directory within the project's folder

        Returns:
        """
        root_dir = os.path.join(base_dir, project_id)
        path = os.path.join(root_dir, dir_name)
        os.makedirs(path, exist_ok=True)
        return path

    async def delete_project_directory(self, project_id: str, base_dir: str = PROJECTS_FILES_DIR) -> str:
        """
        Deletes project's and it's contents

        Returns: folder deleted
        """
        path = os.path.join(base_dir, project_id)
        shutil.rmtree(path, ignore_errors=True)
        return os.path.join(base_dir, project_id)

    async def create_project_root_directory(self, project_id: str, base_dir: str = PROJECTS_FILES_DIR):
        """
        Creates the project root directory
        """
        path = os.path.join(base_dir, project_id)
        os.makedirs(path, exist_ok=True)
        return path

    def copy_files(self, project_id: str, filenames: list = []) -> list:
        """
        Copy files for a given project.

        This function copies files for a specified project. It takes a `project_id` and an optional list of `filenames`. If no filenames are provided, it will copy all files for the project.

        Parameters:
            self: The object that this function is a part of.
            project_id (str): The ID of the project to copy files from.
            filenames (list, optional): A list of file names to copy. Defaults to an empty list.

        Returns:
            list: A list of file paths that were copied.
        """
        successful_files = []

        # check if project folder exists
        project_path = os.path.join(PROJECTS_FILES_DIR, project_id)
        if not os.path.exists(project_path):
            logger.exception("Project folder does not exist")
            raise ProjectFolderNotFound

        for file_info in filenames:
            src_path_filename = sanitize_filename_path(file_info["src_path"], keep_leading_slash=True)
            dest_path_filename = sanitize_filename_path(file_info["dest_path"])
            final_destination = os.path.join(project_path, sanitize_filename_path(file_info["dest_path"]))

            # Try to copy the files
            try:
                # Check if file exists
                if not os.path.isfile(src_path_filename):
                    raise FailedToCopyProjectFiles

                # Ensure the destination directory exists
                os.makedirs(os.path.dirname(final_destination), exist_ok=True)

                shutil.copy(src_path_filename, final_destination)
                successful_files.append(dest_path_filename)
            except (FailedToCopyProjectFiles, Exception):
                if successful_files:
                    try:
                        os.remove(dest_path_filename)
                        logger.exception(f"Not possible to copy the file {dest_path_filename}.")
                    except Exception:
                        logger.exception(
                            f"Not possible to revert remove operation while adding files {dest_path_filename}."
                        )
                        raise
                raise

        return successful_files

    async def add_files(
        self, project_id: str, dir_name: str = None, filenames: list = [], base_dir: str = PROJECTS_FILES_DIR
    ) -> bool:
        """
        Save files within the project's folder

        Parameters:
            project_id: The ID of the project to add files to.
            dir_name: The directory name to store the files in. Defaults to None.
            filenames: A Dict of file names to add. Defaults to None.
            base_dir: The base directory for the project. Defaults to PROJECTS_FILES_DIR.

        Returns:
            str: The path to the added files.
        """
        # build path_dir for the files
        root_dir = os.path.join(base_dir, project_id)
        dir_name = sanitize_directory_path(dir_name)
        path_dir = os.path.join(root_dir, dir_name)

        # create folder is doesn't exist
        os.makedirs(path_dir, exist_ok=True)

        try:
            for filename in filenames:
                file_path = os.path.join(path_dir, filename.filename)

                with open(file_path, "wb") as output_file:
                    output_file.write(filename.body)

                logger.info(f"File {filename.filename} is saved successfully.\n")
        except PermissionError:
            logger.error(f"Failed to save file {filename.filename}: Permission denied.\n")
            raise AddFileErrorOnLocalStorage
        except IOError as ex:
            logger.error(f"Failed to save file {filename.filename}: I/O error occurred - {str(ex)}.\n")
            raise AddFileErrorOnLocalStorage
        except Exception as ex:
            logger.error(f"Failed to save file {filename.filename}.: An unexpected error occurred- {str(ex)}.\n")
            raise AddFileErrorOnLocalStorage

        return True

    async def remove_object(self, project_id: str, relative_path: str) -> bool:
        """
        Remove a file or directory from the project, deleting its content in the FileSystem
        Args:
            project_id: id of the project
            file_relative_path: relative path to the file
        Returns: boolean

        """
        absolute_path = os.path.join(PROJECTS_FILES_DIR, project_id, relative_path)
        logger.info(f"Deleting {absolute_path}")
        try:
            try:
                if os.path.isdir(absolute_path):
                    shutil.rmtree(absolute_path, ignore_errors=True)
                    return True
                elif os.path.isfile(absolute_path):
                    os.remove(absolute_path)
                    return True
                return False
            except FileNotFoundError:
                logger.info("Delete file - File not found")
                return True
            except PermissionError:
                # it's a directory
                shutil.rmtree(absolute_path, ignore_errors=True)
                return True
        except Exception as ex:
            logger.error(f"Failed to delete file from local storage {ex}")
            return False

    @staticmethod
    def add_content(
        project: LocalProject,
        file_relative_path: str,
        content_status: Optional[LocalStorageSyncState] = LocalStorageSyncState.SYNCING,
        db_save: Optional[bool] = True,
    ) -> Tuple[LocalProject, bool]:
        """
        Given a path to a file, it adds all the contents, including intermediate folders
        that might have been missing.
        Args:
            project: local project instance
            file_relative_path: file relative path

        Returns: boolean

        """
        intermediate_dirs = get_intermediate_directories(file_relative_path)
        parent_dir = None
        added_at_least_one = False
        for directory, file_type in intermediate_dirs:
            if not project.key_exists(directory):
                file_name = os.path.basename(directory)
                file_content = LocalProjectContent(
                    name=file_name,
                    key=directory,
                    type=file_type,
                    status=content_status,
                    contents=[],
                    project_id=project.id,
                )

                if parent_dir:
                    project.add_data_value_in_a_folder(parent_dir, file_content)
                else:
                    project.data.contents.append(file_content)

                added_at_least_one = True
                if db_save:
                    project.update(auto_commit=True)
            parent_dir = directory

        return project, added_at_least_one

    def update_file_metadata(self, project_id: str, filename: str, metadata: Dict) -> bool:
        """
        Updates file's metadata within a project
        Args:
            project_id: id of the project that the file belongs to
            filename: file's filename (key)
            metadata: the new metadata

        Returns: boolean
        """

        if project := LocalProject.get(project_id):
            if file := project.get_content_by_key(filename):
                file.metadata = metadata
                project.update(db, auto_commit=True)
                return True
        return False

    def update_project_metadata(self, project_id: str, metadata: Dict) -> bool:
        """
        Updates a local project's metadata

        Args:
            project_id: id of the project
            metadata: the new metadata

        Returns: boolean
        """
        if project := LocalProject.get(project_id):
            project.data.metadata = metadata
            project.update(db, auto_commit=True)
            return True
        return False

    def rename_object(self, project_id: str, current_file_relative_path: str, new_file_name: str) -> bool:
        """
        Given a file or folder relative path rename it to the new desired name
        Args:
            project_id: id of the project
            current_file_relative_path: object current relative path
            new_file_name: object new name

        Returns:

        """

        def _update_file_keys(contents: List[LocalProjectContent], current_name: str, new_name: str):
            content: LocalProjectContent
            for content in contents:
                content.key = content.key.replace(current_name, new_name, 1)
                if content.type == FileType.DIRECTORY:
                    _update_file_keys(content.contents, current_name, new_name)

        if project := LocalProject.get(project_id):
            if file := project.get_content_by_key(current_file_relative_path):
                path, current_filename = os.path.split(current_file_relative_path)
                new_file_relative_path = os.path.join(path, new_file_name)
                new_file_full_path = os.path.join(PROJECTS_FILES_DIR, project_id, new_file_relative_path)
                current_file_full_path = os.path.join(PROJECTS_FILES_DIR, project_id, current_file_relative_path)
                file.name = new_file_name
                file.key = os.path.join(path, new_file_name)
                file.status = LocalStorageSyncState.UNSTAGED
                logger.info(f"Renaming {current_file_full_path} to {new_file_full_path}")
                try:
                    if file_exists(new_file_full_path):
                        raise LocalFileAlreadyExists
                    os.rename(current_file_full_path, new_file_full_path)
                    if is_dir(new_file_full_path):
                        _update_file_keys(file.contents, current_file_relative_path, new_file_relative_path)
                    project.update()
                    return True
                except FileNotFoundError as ex:
                    logger.error(f"Could not rename file - original file does not exist {ex}")
                    raise LocalFileNotFound
                except PermissionError:
                    logger.error("Can't rename file - permission error")
                    raise LocalFilePermissionError
                except FileExistsError:
                    logger.error("File already exists")
                    raise LocalFileAlreadyExists
                except Exception as ex:
                    logger.error(f"Unexpected error while renaming file {ex}")
                    raise
            else:
                raise LocalFileNotFound
        raise LocalProjectNotFoundError

    async def filter_snippets_by_language(self, snippet_language: str) -> List["LocalProject"]:
        """
        List projects containing snippets that is filtered by snippet type
        """
        return LocalProject.filter_metadata("code_snippet.language", snippet_language)


local_storage = LocalStorage()
