import copy
import os
import sqlite3
from dataclasses import asdict
from typing import Callable, Coroutine, Dict, List, Optional, Tuple

from aext_project_filebrowser_server.clients.base import ProjectAPIClientInterface
from aext_project_filebrowser_server.consts import (
    CODE_SNIPPET_TAG,
    PROJECTS_FILES_DIR,
    FileOperations,
    FileType,
    LocalStorageSyncState,
)
from aext_project_filebrowser_server.exceptions import (
    AddFileErrorOnDatabase,
    AddFileErrorOnLocalStorage,
    AddProjectFilesError,
    CloudProjectFileListError,
    CloudProjectFileTreeError,
    CloudProjectNotFoundError,
    CodeSnippetsSchemaError,
    FailedToCopyProjectFiles,
    FailedToStartSyncPull,
    LocalProjectNotFoundError,
    ProjectFailCreate,
    ProjectFailDelete,
    ProjectFolderNotFound,
    ProjectNameConflict,
)
from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProject,
    CloudProjectFileList,
    CloudProjectFileTree,
    CloudProjectPermissions,
    MyProjectPermissions,
    ProjectPermission,
    ProjectPermissionUpdate,
    UserProjectPermissions,
)
from aext_project_filebrowser_server.schemas.local_project import (
    CodeSnippetMetadata,
    LocalProject,
    LocalProjectContent,
)
from aext_project_filebrowser_server.schemas.organizations import (
    OrganizationMember,
    OrganizationMembers,
)
from aext_project_filebrowser_server.schemas.projects import (
    ProjectMetadata,
    SyncFileResult,
    UploadFileResult,
)
from aext_project_filebrowser_server.services.database import db
from aext_project_filebrowser_server.services.local_storage import (
    DiffFileOperations,
    local_storage,
)
from aext_project_filebrowser_server.services.sync import (
    SyncCoroutine,
    sync_pull_delete_file,
    sync_pull_write_file,
    sync_push_delete_file,
    sync_push_write_file,
    sync_result_processor,
)
from aext_project_filebrowser_server.utils import (
    is_dir,
    is_file,
    run_concurrently,
    run_coroutine,
    sanitize_directory_path,
    sanitize_filename_path,
)

from aext_shared import logger as custom_logger
from aext_shared.config import get_config
from aext_shared.errors import BackendError, UnauthorizedError

logger = custom_logger.logger
config = get_config()


class ProjectService:
    _client = None
    _user_credentials = None

    def set_user_credentials(self, user_credentials: Dict):
        self._user_credentials = user_credentials

    @property
    def client(self) -> ProjectAPIClientInterface:
        return self._client

    @client.setter
    def client(self, client: ProjectAPIClientInterface) -> None:
        self._client = client

    async def get_cloud_project(self, project_id: str, **kwargs) -> CloudProject:
        """
        Uses the project service client to retrieve a specific project from the cloud.

        Args:
            project_id: a string representing the project id.

        Returns: an instance of schemas.cloud_project.CloudProject in case it exists on the cloud.
        """
        project = await self.client.get_project(project_id, **kwargs)
        if not project:
            raise CloudProjectNotFoundError

        return project

    async def get_cloud_projects(self, **kwargs) -> List[CloudProject]:
        """
        Uses the project service client to retrieve all existing projects from the cloud.

        Args: (None)

        Returns: a list with schemas.cloud_project.CloudProject instances.
        """
        kwargs["timeout"] = 20
        return await self.client.get_projects(**kwargs)

    async def get_cloud_project_filetree(self, project_id: str, **kwargs) -> CloudProjectFileTree:
        """
        Uses the project service client to retrieve a specific project's filetree from the cloud.

        Args:
            project_id: a string representing the project id.
            project_name: a string representing the project name.

        Returns: an instance of schemas.cloud_project.CloudProjectFileTree in case the project exists on the cloud.
        """
        filetree = await self.client.get_project_filetree(project_id, **kwargs)
        if not filetree:
            raise CloudProjectFileTreeError

        return CloudProjectFileTree(project_id=project_id, stored_files=filetree.stored_files)

    async def list_cloud_project_files(self, project_id: str, **kwargs) -> CloudProjectFileList:
        """
        Uses the project service client to retrieve the list of files of a specific project.
        The list is a flat list, different from
        Args:
            project_id:
            **kwargs:

        Returns:

        """
        files_list = await self.client.list_project_files(project_id, **kwargs)
        if not files_list:
            raise CloudProjectFileListError

        return files_list

    async def _sync_pull_dir_files(
        self, project: LocalProject, file_tree: Dict, pull_operations: Dict
    ) -> List[Coroutine]:
        """
        Iterates over all project file tree items and perform the necessary operations to
        pull the files from cloud.
        Args:
            project: instance of the local project
            file_tree: the project file tree structure
            pull_operations: all the operations that needs to be performed per file

        Returns: None

        """
        coroutines = []
        # checking for file that needs to be written since they won't be
        # found in the file tree
        for file_path, operation in copy.deepcopy(pull_operations).items():
            if operation == FileOperations.LOCAL_WRITE:
                download_args = (
                    project.id,
                    file_path,
                    self._user_credentials,
                )
                await local_storage.create_directory(project.id, os.path.dirname(file_path))
                local_storage.add_content(project, file_path)
                coroutines.append(SyncCoroutine(sync_pull_write_file, download_args, file_path))
                pull_operations.pop(file_path)

        for item in file_tree:
            file_path = item["key"]
            file_type = item["type"]
            if file_type == FileType.FILE:
                if file_path in pull_operations:
                    operation = pull_operations[file_path]
                    coro = None
                    if operation == FileOperations.LOCAL_DELETE:
                        delete_args = (project.id, file_path)
                        coro = SyncCoroutine(sync_pull_delete_file, delete_args, file_path)
                    if coro:
                        coroutines.append(coro)
            elif file_type == FileType.DIRECTORY:
                await local_storage.create_directory(project.id, file_path)
                await self._sync_pull_dir_files(project, item["contents"], pull_operations)

        if coroutines:
            logger.debug(f"Running {len(coroutines)} COROUTINES {coroutines}")
            await run_coroutine(coroutines, lambda sync_results: sync_result_processor(sync_results, SyncFileResult))
        return coroutines

    async def sync_pull_project_files(self, project_id: str, **kwargs):
        """
        Fetches files from Project service and saves it to the local storage
        Args:
            project_id: Project's id to be synced

        Returns: None

        """
        await local_storage.create_project_root_directory(project_id)

        if not (project := LocalProject.get(project_id, kwargs.pop("db", None))):
            raise LocalProjectNotFoundError

        contents = asdict(project.data)["contents"]
        cloud_file_list = await self.list_cloud_project_files(project_id)
        diff: DiffFileOperations = await local_storage.diff(project_id, cloud_file_list)
        local_ops = diff.pull_operations.filter_local_operations()  # noqa
        try:
            if local_ops:
                return await self._sync_pull_dir_files(project, contents, local_ops)
            else:
                project.update_files_status(LocalStorageSyncState.SYNCED, db_save=True)
        except Exception as ex:
            logger.error(f"Failed to start sync pull process {ex}")
            raise FailedToStartSyncPull

    async def download_file(self, project_id: str, filename: str, **kwargs) -> Optional[bytes]:
        """
        Downloads the file content from Projects Service
        Args:
            project_id: project's id
            filename: name of the file that will be downloaded

        Returns: File content in bytes

        """
        return await self.client.download_file(project_id, filename, **kwargs)

    async def sync_push_project_files(self, project_id: str, filenames: Optional[List[str]] = None, **kwargs) -> bool:
        """
        Uploads project's files stored in the local storage to the Projects Service
        Args:
            project_id: Id of the project that will have its files uploaded
            filenames: (optional) list of relative file paths to push; pushes all project files if None.

        Returns: boolean

        """

        async def sync_push_recursively(
            contents: List[LocalProjectContent],
            push_operations: Dict,
            coroutines: Optional[List[Tuple[Callable, Tuple, str]]] = None,
        ):
            if coroutines is None:
                coroutines = []

            for item in contents:
                # Get the full path of the item
                file_relative_path = item.key
                file_full_path = os.path.join(PROJECTS_FILES_DIR, project_id, file_relative_path)
                if is_file(file_full_path):
                    if file_relative_path in push_operations:
                        operation = push_operations[file_relative_path]
                        if operation == FileOperations.CLOUD_WRITE:
                            write_args = (project_id, file_full_path, file_relative_path, self._user_credentials)
                            coroutines.append(SyncCoroutine(sync_push_write_file, write_args, file_relative_path))
                elif is_dir(file_full_path):
                    # If it's a directory, recursively call the function
                    await sync_push_recursively(item.contents, push_operations, coroutines)

        project = LocalProject.get(project_id, db=kwargs.get("db"))
        project.data.contents.reverse()
        try:
            coroutines = []
            if filenames:
                cloud_ops = {}
                for filename in filenames:
                    cloud_ops[filename] = FileOperations.CLOUD_WRITE
            else:
                cloud_file_list = await self.list_cloud_project_files(project_id)
                diff: DiffFileOperations = await local_storage.diff(project_id, cloud_file_list)
                cloud_ops = diff.push_operations.filter_cloud_operations()
                # checking for cloud delete operations since the files don't exist locally
                # and therefore won't be found in sync_push_recursively

            for file_relative_path, operation in copy.deepcopy(cloud_ops).items():
                if operation == FileOperations.CLOUD_DELETE:
                    delete_args = (project_id, file_relative_path, self._user_credentials)
                    coroutines.append(SyncCoroutine(sync_push_delete_file, delete_args, file_relative_path))
                    cloud_ops.pop(file_relative_path)
                    logger.info(coroutines)

            if cloud_ops:
                await sync_push_recursively(project.data.contents, cloud_ops, coroutines)
            if coroutines:
                await run_coroutine(
                    coroutines, lambda sync_results: sync_result_processor(sync_results, UploadFileResult)
                )
            return True
        except (BackendError, UnauthorizedError) as ex:
            logger.debug(f"User does not have permission to change project {ex}")
            raise
        except Exception as ex:
            logger.exception(f"Could not push files {ex}")
            return False

    async def create_project(
        self, project_title: str, project_default_environment: str, tags: List[str], **kwargs
    ) -> LocalProject:
        """
        Creates a project in Projects Service and also saves it to the local storage
        Args:
            project_title: title is the displayable project reference
            project_default_environment: a default environment to be used by conda-project
            tags: a list of tags to filter projects

        Returns: If successfully created returns a LocalProject instance, otherwise it returns None
        """
        cloud_metadata = {"tags": tags}

        cloud_project = await self.client.create_project(project_title, metadata=cloud_metadata, **kwargs)
        if not cloud_project:
            raise ProjectFailCreate

        project = cloud_project.to_local_project([], project_default_environment)

        try:
            project.save(db)
            await local_storage.create_project_root_directory(project.id)
            return project

        except sqlite3.IntegrityError as ex:
            raise ProjectNameConflict from ex

    async def delete_project(self, project_id: str, **kwargs) -> bool:
        """
        Deletes a project from Projects Service and also from local storage
        Args:
            project_id: Project's id

        Returns: boolean reporting whether or not the project was successfully deleted

        """
        try:
            logger.info(f"Deleting the project: {project_id}")
            project_deleted_on_cloud = await self.client.delete_project(project_id, **kwargs)
            if project_deleted_on_cloud:
                LocalProject.delete(project_id)
                await local_storage.delete_project_directory(project_id)
                return True

        except ProjectFailDelete as ex:
            logger.error(f"Not possible to delete the project due to a BackendError. {ex.message}")
            raise

        except sqlite3.IntegrityError as ex:
            logger.error("Not possible to delete the project due to a DB integrity error")
            raise ProjectFailDelete from ex

    async def create_project_file(self, project_id: str, filename: str) -> bool:
        """
        This method creates a new file in a project. It first retrieves the project, then
        attempts to create the file using the local storage system. If an error occurs,
        it rolls back the changes and raises an exception.

        Args:
            project_id (str): The ID of the project.
            filename (str): The name of the file to be created.

          Raises:
            AddFileErrorOnDatabase: If there's an error creating the file.
        """
        project = LocalProject.get(project_id)
        if not project:
            return False

        relative_path_file = sanitize_filename_path(filename)

        try:
            project, _ = local_storage.add_content(project, relative_path_file, LocalStorageSyncState.UNSTAGED)
            project.update(db)
        except (sqlite3.IntegrityError, Exception) as ex:
            logger.error(f"Failure while creating filename, {ex}")
            raise AddFileErrorOnDatabase from ex

        return True

    async def add_project_files(self, project_id: str, filenames: List[Dict]) -> bool:
        """
        Adds files to a project. This function is designed to be atomic,
        meaning it will either successfully add all files or make no
        changes if an error occurs.

        Args:
            project_id (str): Project's ID to which files should be added.
            filenames (list): A list of dictionaries, where each dictionary contains
                "src_path" and "dest_path" keys describing the file to add.

        Returns:
            bool: True if all files were successfully added, False otherwise.

        Raises:
            LocalProjectNotFoundError: If the project with the given ID does not exist.
            AddProjectFilesError: If there is an error adding the files (e.g., invalid payload,
                failed to copy files).
            AddFileErrorOnDatabase: If there is an error adding a file to the database.
        """

        def _is_payload_valid(filenames: list) -> bool:
            """
            Validates if all files in the given payload are valid.

            Args:
                filenames (list): A list of file information dictionaries, each containing 'src_path' and 'dest_path' keys.

            Returns:
                bool: True if all files are valid (i.e., both source and destination paths are present), False otherwise.
            """
            for file_info in filenames:
                if not file_info["src_path"] or not file_info["dest_path"]:
                    return False
            return True

        project = LocalProject.get(project_id)
        if not project:
            raise LocalProjectNotFoundError

        # check if payload is safe
        if not _is_payload_valid(filenames):
            raise AddProjectFilesError

        # First tries to copy all files to the project's folder
        try:
            added_filename_paths = local_storage.copy_files(project_id, filenames)
        except (FailedToCopyProjectFiles, ProjectFolderNotFound, Exception) as ex:
            raise AddProjectFilesError from ex

        # Then, add all successful copied files to the DB
        try:
            for filename_path in added_filename_paths:
                project, _ = local_storage.add_content(
                    project, filename_path, LocalStorageSyncState.UNSTAGED, db_save=False
                )
            project.update(db)

        except sqlite3.IntegrityError as ex:
            logger.error(f"Failed to add files into the database, {ex}")
            raise AddFileErrorOnDatabase from ex

        except Exception as ex:
            logger.error(f"Unexpected Exception while adding project files. {ex}")
            raise AddFileErrorOnDatabase from ex

        return True

    async def upload_project_files(self, project_id: str, dir_name: str, filenames: list):
        """
        Add files to a project's directory.

        This method adds or updates files in a specified directory within a project.
        It first retrieves the project and creates a deep copy of its contents to ensure
        rollback if any issues occur during the file addition process. Then, it iterates
        over each filename and checks if the file already exists on the database.
        If not, it adds the file; otherwise, it updates the existing file. The method then
        tries to save all changes to the database. If any exceptions occur, it reverts
        the changes and raises an error.

        Args:
        - project_id (str): The ID of the project where files will be added or updated.
        - dir_name (str): The name of the directory where files will be added or updated.
        - filenames (Dict): A dict of file objects to add or update. Defaults to None.

        Returns:
        - None

        Raises:
        - AddFileErrorOnDatabase: If there is an error adding or updating files on the database.
        - AddFileErrorOnLocalStorage: If there is an error saving files in local storage.
        """
        old_content = None
        if project_id:
            try:
                # retrieve the project
                project = LocalProject.get(project_id)
                if not project:
                    return None

                # create a deep copy just in case for a rollback
                old_content = copy.deepcopy(project.data.contents)
                dir_name = sanitize_directory_path(dir_name)

                for f in filenames:
                    relative_path_file = os.path.join(dir_name, f.filename)
                    project, _ = local_storage.add_content(project, relative_path_file, LocalStorageSyncState.UNSTAGED)

            except (sqlite3.IntegrityError, Exception) as ex:
                # revert the changes
                if old_content is not None:
                    project.data.contents = old_content
                    project.update(db)
                logger.error(f"Fail to add filenames, {ex}")

                raise AddFileErrorOnDatabase from ex

            # Try to add all files in the local storage
            try:
                await local_storage.add_files(project_id, dir_name, filenames)
            except (AddFileErrorOnLocalStorage, Exception) as ex:
                # revert the changes
                if old_content is not None:
                    project.data.contents = old_content
                    project.update(db)
                logger.error(f"Fail to save filenames in the local storage, {ex}")

                raise AddFileErrorOnLocalStorage from ex

        return None

    async def delete_project_file(self, project_id: str, filename_paths: list):
        """
        Delete one or more files from a project.

        This method deletes the specified files from the project with the given Project ID.
        The file paths are passed as a list of strings. Each string should be a valid file path within the project.
        Also, all file path are sanitized to prevent XSS.

        Args:
            project_id (str): The ID of the project to delete files from.
            filename_paths (list): A list of file paths to delete.

        Returns:
            None
        """

        if project := LocalProject.get(project_id):
            for filename_path in filename_paths:
                # sanitize the filename path and convert it into a relative filename path
                filename_key = sanitize_filename_path(filename_path)
                if project.remove_file_or_folder(filename_key, db_save=True):
                    await local_storage.remove_object(project_id, filename_key)

    async def update_cloud_project(
        self,
        project_id: str,
        project_name: Optional[str] = None,
        project_title: Optional[str] = None,
        project_description: Optional[str] = None,
        project_metadata: Optional[ProjectMetadata] = None,
        **kwargs,
    ) -> CloudProject:
        """
        Uses the project service client to update a specific project on the cloud.

        Args:
            project_id: a string representing the project id;
            project_name: a string representing the new project name;
            project_title: a string representing the new project title;
            project_description: a string representing the new project description;
            project_metadata a schemas.projects.ProjectMetadata instance representing the new project metadata.

        Returns: an updated instance of schemas.cloud_project.CloudProject in case it exists on the cloud.
        """
        payload = {}

        if project_name:
            payload["name"] = project_name

        if project_title:
            payload["title"] = project_title

        if project_description:
            payload["description"] = project_description

        if project_metadata:
            payload["metadata"] = project_metadata.to_dict()

        return await self.client.update_project(project_id, payload, **kwargs)

    async def list_project_permission(
        self, project_id: str, user_organization_names: List[str]
    ) -> CloudProjectPermissions:
        """
        List projects permissions.
        Args:
            project_id: project id
            user_organization_names: list of organization names that the users belong to

        Returns: CloudProjectPermissions

        """
        coros = {org_name: self.client.list_organization_users(org_name) for org_name in user_organization_names}
        coros["permissions"] = self.client.list_project_permissions(project_id)
        result = await run_concurrently(coros, timeout=30)  # ProjectAPI might take a lot of time to respond
        cloud_permissions: CloudProjectPermissions = result.get("permissions")
        all_orgs_members: List[OrganizationMembers] = [result.get(org_name) for org_name in user_organization_names]

        if not cloud_permissions or not all_orgs_members:
            return CloudProjectPermissions(permissions=[])

        permission: ProjectPermission = None  # noqa
        org_member: OrganizationMember = None  # noqa

        # Projects API does not return the user information, merging it with the result from organization members
        for permission in cloud_permissions.permissions:
            if permission.type == "user_id" and permission.object.id == "*":
                # The endpoint that list projects permission always return a * user - filtering out
                cloud_permissions.permissions.remove(permission)
                continue
            for org_members in all_orgs_members:
                for org_member in org_members.members:
                    if permission.object.id == org_member.id:
                        permission.object.email = org_member.email
                        permission.object.first_name = org_member.first_name
                        permission.object.last_name = org_member.last_name
                        break

        return cloud_permissions

    async def update_project_permissions(self, project_id: str, permissions: List[ProjectPermissionUpdate]) -> bool:
        """
        Updates project's permission for a given organization
        Args:
        project_id: project's id
        """
        return await self.client.update_project_permissions(project_id, permissions)

    async def update_file_metadata(
        self,
        project_id: str,
        file_relative_path: str,
        metadata: Dict,
    ) -> bool:
        """
        Updates file's metadata
        Args:
            project_id: id of the project that the file belongs to
            file_relative_path: file's filename (key)
            metadata: dict containing all the metadata
        Returns: bool

        Raises: CodeSnippetsSchemaError
        """
        code_snippet_metadata = metadata.get(CODE_SNIPPET_TAG, None)
        if code_snippet_metadata:
            try:
                if not CodeSnippetMetadata(**code_snippet_metadata).validate():
                    raise CodeSnippetsSchemaError
            except TypeError:
                raise CodeSnippetsSchemaError
        if local_storage.update_file_metadata(project_id, file_relative_path, metadata):
            return True
        return False

    async def update_project_metadata(
        self,
        project_id: str,
        metadata: Dict,
    ) -> bool:
        """
        Updates a local project's metadata

        Args:
            project_id: id of the project
            metadata: dict containing all the metadata

        Returns: bool
        """
        return local_storage.update_project_metadata(project_id, metadata)

    async def list_user_permissions(self, project_id: str) -> UserProjectPermissions:
        """
        List all permissions of all users in a project. Only owners can read this information
        Args:
            project_id: id of the project which the permissions will be checked

        Returns: UserProjectPermissions
        """
        return await self.client.list_user_permissions(project_id)

    async def list_my_permissions(self, project_id: str) -> MyProjectPermissions:
        """
        List user's project permissions
        Args:
            project_id: id of the project which the permissions will be checked

        Returns: MyProjectPermissions
        """
        return await self.client.list_my_permissions(project_id)


project_service = ProjectService()
