from dataclasses import asdict, dataclass, field, fields
from datetime import datetime
from typing import Dict, List, Optional

from aext_project_filebrowser_server.consts import (
    UNDEFINED_ENVIRONMENT,
    VALID_PROJECT_TAGS,
    FileType,
    LocalStorageSyncState,
    ProjectPermissionActionType,
    ProjectPermissionRoles,
)
from aext_project_filebrowser_server.exceptions import FailedToInstantiateSchema
from aext_project_filebrowser_server.schemas.local_project import (
    LocalProject,
    LocalProjectContent,
    LocalProjectData,
    ProjectMetadata,
    ProjectOwner,
)

from aext_shared.logger import logger


@dataclass
class CloudProject:
    id: str
    name: str
    title: str
    description: str
    created_at: datetime
    updated_at: datetime
    metadata: ProjectMetadata
    owner: ProjectOwner

    @classmethod
    def create(cls, data: Dict) -> "CloudProject":
        metadata = ProjectMetadata(**data["metadata"])
        owner = ProjectOwner(**data["owner"])

        return cls(
            id=data["id"],
            name=data["name"],
            title=data["title"],
            description=data["description"],
            created_at=data["created_at"],
            updated_at=data["updated_at"],
            metadata=metadata,
            owner=owner,
        )

    def to_dict(self) -> dict:
        return asdict(self)

    def to_local_project(
        self,
        contents: List[LocalProjectContent],
        default_environment: str = UNDEFINED_ENVIRONMENT,
        status: LocalStorageSyncState = LocalStorageSyncState.UNSTAGED,
    ) -> LocalProject:
        """
        Transforms a project that has been retrieved from the cloud to its respective LocalProject schema, which
        can be used for local database operations.

        Args:
            contents: A list of LocalProjectData object represeting the project's `row.data` entry in the local database;
            default_environment: A string representing the default environment to add to the LocalProject;
            status: A LocalStorageSyncState string representing the project's status.

        Returns: The project's respective LocalProject object.
        """
        data = LocalProjectData(
            name=self.name,
            title=self.title,
            created_at=self.created_at,
            updated_at=self.updated_at,
            metadata=self.metadata,
            owner=self.owner,
            contents=LocalProject._create_content(self.id, contents, status),
        )

        return LocalProject(
            id=self.id,
            name=self.name,
            owner=self.owner.id,
            default_environment=default_environment,
            tags=self.metadata.tags,
            data=data,
            status=status,
        )

    def validate_tags(self) -> bool:
        """
        Checks if a project contains either a `data` or `notebook` tag.

        Args: (None)

        Returns: True if a project contains either a `data` or `notebook` tag; False otherwise.
        """
        return any(tag in self.metadata.tags for tag in VALID_PROJECT_TAGS)


@dataclass
class CloudProjectFileDetails:
    file_version_id: str
    description: str
    etag: str
    url: str
    signed_url: str
    deleted: bool
    metadata: Dict = field(default_factory=dict)

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class CloudProjectContent:
    key: str
    name: str
    type: FileType
    writtenby: str
    size: int
    last_modified: str
    details: Optional[CloudProjectFileDetails] = None
    contents: Optional["CloudProjectContent"] = None

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class CloudProjectFileTree:
    project_id: str
    stored_files: List[CloudProjectContent]

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class CloudProjectFile:
    name: str
    description: str
    writtenby: str
    size: int
    etag: str
    last_modified: datetime
    file_version_id: str
    url: str
    signed_url: str
    metadata: Dict
    project_id: str
    deleted: bool
    type: FileType.FILE
    pending: bool = False

    def __init__(self, **kwargs):
        """
        Dynamically assign only allowed fields while ignoring extra kwargs.
        """
        allowed_fields = {f.name for f in fields(self)}

        # Set only allowed attributes
        for field_name in allowed_fields:
            if field_name in kwargs:
                setattr(self, field_name, kwargs[field_name])
            else:
                raise TypeError(f"Missing required argument: {field_name}")

        # Call post-init modifications if defined
        if hasattr(self, "__post_init__"):
            self.__post_init__()

    def __post_init__(self):
        # For some reason Projects API returns the etag wrapped in double quotes which should be removed
        self.etag = self.etag.replace('"', "")

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class CloudProjectCheckpoint:
    id: str
    project_id: str
    path: str
    description: str
    writtenby: str
    metadata: Dict
    created_at: datetime
    s3_path: str
    deleted: bool
    pending: bool

    def to_dict(self) -> dict:
        return asdict(self)


@dataclass
class CloudProjectFileList:
    items: List[CloudProjectFile]
    num_items: int

    @classmethod
    def create_from_cloud_response(cls, cloud_response):
        return cls(
            num_items=cloud_response["num_items"], items=[CloudProjectFile(**item) for item in cloud_response["items"]]
        )


@dataclass
class ObjectPermissionInfo:
    id: str
    email: Optional[str] = None
    first_name: Optional[str] = None
    last_name: Optional[str] = None


@dataclass
class ProjectPermission:
    type: str
    object: ObjectPermissionInfo
    roles: List[ProjectPermissionRoles]


@dataclass
class CloudProjectPermissions:
    permissions: List[ProjectPermission]

    @classmethod
    def create_from_cloud_response(cls, cloud_response: Dict) -> "CloudProjectPermissions":
        try:
            items = cloud_response["items"]
            permission_per_object_id = {}
            for item in items:
                object_id = item["id"]  # it can be id of both users and orgs
                if object_id in permission_per_object_id:
                    # object has multiple roles/relation, just need to append the next one
                    permission_obj = permission_per_object_id[object_id]
                    permission_obj.roles.append(item["relation"])
                else:
                    # first role/relation found for object, create an entire new object
                    permission_per_object_id[object_id] = ProjectPermission(
                        type=item["type"], object=ObjectPermissionInfo(id=item["id"]), roles=[item["relation"]]
                    )

            return cls(permissions=list(permission_per_object_id.values()))
        except Exception as ex:
            logger.error("Failed to create CloudProjectPermissions from cloud response")
            raise FailedToInstantiateSchema from ex

    def get_object_permission(self, object_id: str) -> ProjectPermission:
        for permission in self.permissions:
            if permission.object.id == object_id:
                return permission


@dataclass
class ProjectPermissionUpdate:
    action: str
    permission: str
    user_email: str
    user_id: str
    org_id: Optional[str] = None

    def create_payload(self):
        _type = ProjectPermissionActionType.ORGANIZATION if self.org_id else ProjectPermissionActionType.USER
        _object_id = self.org_id if self.org_id else self.user_id
        return {
            "type": _type,
            "relation": self.permission,
            "id": _object_id,
            "action": self.action,
        }


@dataclass
class UserProjectPermission:
    type: str
    relation: str
    id: str


@dataclass
class UserProjectPermissions:
    items: List[UserProjectPermission]
    num_items: int


@dataclass
class MyProjectPermissions:
    own: bool
    modify: bool
    read: bool
    share: bool
    find: bool
    delete: bool

    @classmethod
    def create(cls, permissions: dict):
        """
        Validates the data by ignoring extra permissions not mapped by this dataclass
        """
        valid_fields = {f.name for f in fields(cls)}
        valid_data = {key: value for key, value in permissions.items() if key in valid_fields}
        return cls(**valid_data)
