import hashlib
import os
from pathlib import Path
from typing import Optional

from aext_project_filebrowser_server.exceptions import LocalFileNotFound


class FileDiff:
    @classmethod
    async def calculate_md5(cls, file_path: str, chunk_size: int = 4096) -> str:
        """
        Calculate the MD5 checksum of a file.

        Args:
            file_path (str): The path to the file.
            chunk_size (int): The size of each chunk read from the file. Default is 4096 bytes.

        Returns:
            str: The MD5 checksum of the file in hexadecimal format.
        """
        md5 = hashlib.md5()
        with open(file_path, "rb") as file:
            for chunk in iter(lambda: file.read(chunk_size), b""):
                md5.update(chunk)

        return md5.hexdigest()

    @classmethod
    async def calculate_etag(cls, file_path: str, num_chunks: Optional[int] = None) -> str:
        """
        Calculates AWS S3 file ETAG
        Args:
            file_path: path to the file
            num_chunks: number of chunks that the file needs to be split

        Returns: string containing hex digest

        """
        if num_chunks is None:
            # Assuming that the file was uploaded as a single piece (non-multipart)
            return await cls.calculate_md5(file_path)

        md5_digests = []
        chunk_size = cls.next_chunk_size(file_path, num_chunks)
        with open(file_path, "rb") as file:
            for chunk in iter(lambda: file.read(next(chunk_size)), b""):
                md5_digests.append(hashlib.md5(chunk).digest())

        # E-Tag format when file is uploaded using multi-part
        return f"{hashlib.md5(b''.join(md5_digests)).hexdigest()}-{len(md5_digests)}"

    @classmethod
    def get_file_size(cls, file_path: str) -> int:
        """
        Return file size
        Args:
            file_path: path to the file

        Returns: int representing the file byte size

        """
        file = Path(file_path)
        return file.stat().st_size

    @classmethod
    def next_chunk_size(cls, file_path: str, num_parts: int) -> int:
        """
        An iterator that given the file path and the number of parts it should be split returns
        the next file chunk size
        Args:
            file_path: path to the file
            num_parts: number of parts that the file should be split

        Returns: int containing the next chunk size in bytes

        """
        if num_parts == 0:
            num_parts = 1

        file_size = cls.get_file_size(file_path)
        full_part_size = file_size // num_parts
        partial_part_size = file_size % num_parts

        for idx in range(num_parts):
            chunk_size = full_part_size + (1 if idx < partial_part_size else 0)
            yield chunk_size

    @classmethod
    async def are_files_equal(cls, local_file_absolute_path: str, cloud_file_etag: str) -> bool:
        """
        Check if files are equal
        Args:
            local_file_absolute_path (str): The path to the first file
            cloud_file_etag (str): The etag value stored on the cloud

        Returns: boolean

        """
        if not os.path.exists(local_file_absolute_path):
            raise LocalFileNotFound

        if "-" in cloud_file_etag:
            # this means that the file was uploaded in multi-part mode
            num_chunks = int(cloud_file_etag.split("-")[-1])
            return await cls.calculate_etag(local_file_absolute_path, num_chunks=num_chunks) == cloud_file_etag

        return await cls.calculate_etag(local_file_absolute_path) == cloud_file_etag
