import sqlite3
from typing import List, Optional, Tuple

from aext_project_filebrowser_server.consts import OS_PLATFORM
from aext_project_filebrowser_server.exceptions import DatabaseCursorError
from aext_project_filebrowser_server.migrations import alembic_config, migration_upgrade
from aext_project_filebrowser_server.utils import file_exists

from aext_shared import logger as custom_logger

logger = custom_logger.logger


class SQLite:
    """
    Wrapper to deal with SQLite. This class is used to interact with the database from connection
    to queries/statements.
    """

    DEFAULT_DB_PATH = alembic_config.DB_PATH

    def __init__(self, db_path: str = None):
        if db_path is None:
            db_path = self.DEFAULT_DB_PATH
        try:
            self.db_path = db_path
            if not file_exists(self.db_path):
                logger.warning("Database not initialized by migration_upgrade(). Running it again")
                migration_upgrade()
            self.connection = sqlite3.connect(self.db_path)
            self.cursor = self.connection.cursor()
        except sqlite3.OperationalError as ex:
            logger.error(f"Could not open database at {db_path} {ex}")
            if not file_exists(self.db_path):
                logger.error(f"Database file does not exist {self.db_path}")
        except Exception as ex:
            logger.error(f"Could not connect to the database or create a cursor {ex}")

    def execute(self, raw_sql: str, sql_args: Tuple = (), auto_commit: bool = True) -> sqlite3.Cursor:
        def _execute():
            response = self.cursor.execute(raw_sql, sql_args)
            if auto_commit:
                self.connection.commit()
            return response

        try:
            return _execute()
        except AttributeError:
            # retrying the connection and query
            try:
                self.connection = sqlite3.connect(self.DEFAULT_DB_PATH)
                self.cursor = self.connection.cursor()
                return _execute()
            except Exception as ex:
                logger.exception(f"Error while executing query. Possibly due to a lack of cursor {ex}")
                raise DatabaseCursorError from ex

    def executemany(
        self, raw_sql: str, sql_args_sequence: Optional[List[Tuple]] = None, auto_commit: bool = True
    ) -> sqlite3.Cursor:
        if sql_args_sequence is None:
            sql_args_sequence = []
        response = self.cursor.executemany(raw_sql, sql_args_sequence)
        if auto_commit:
            self.connection.commit()
        return response

    def fetchall(self):
        return self.cursor.fetchall()

    def fetchone(self):
        return self.cursor.fetchone()

    def commit(self):
        self.connection.commit()

    def rollback(self):
        self.connection.rollback()


class Database:
    """
    The Database class acts as an abstraction to basic database operations
    having a driver as the underlying engine for actually deal with particular database flavors
    """

    def __init__(self, db_driver):
        self.driver = db_driver

    def execute(self, statement: str, statement_args: Tuple = (), auto_commit: bool = True):
        return self.driver.execute(statement, statement_args, auto_commit)

    def executemany(
        self, statement: str, statement_args_sequence: Optional[List[Tuple]] = None, auto_commit: bool = True
    ):
        return self.driver.executemany(statement, statement_args_sequence, auto_commit)

    def _describe_columns(self, record):
        if record:
            column_names = [description[0] for description in self.driver.cursor.description]
            record_dict = dict(zip(column_names, record))
            return record_dict

    def fetchone(self):
        record = self.driver.fetchone()
        if record:
            return self._describe_columns(record)

    def fetchall(self) -> List:
        records = self.driver.fetchall()
        if records:
            return [self._describe_columns(record) for record in records]
        return []


db = Database(SQLite())
