import asyncio
import os
import shutil
import sys
import unittest
from pathlib import Path

import pytest
import respx
import tornado
from aext_project_filebrowser_server.consts import PROJECTS_FILES_DIR
from aext_project_filebrowser_server.services.watchdog import watchdog_service
from tests.setup_tests import BasicDBAppHTTPTests

pytestmark = pytest.mark.skipif(sys.platform.startswith("win"), reason="Skipping tests on Windows")


class WatchdogTestCase(BasicDBAppHTTPTests):
    token = "abc"

    def setUp(self):
        super().setUp()

        os.environ["JUPYTERHUB_SERVICE_URL"] = "http://random_host"
        os.environ["JUPYTERHUB_API_TOKEN"] = self.token

        # add pre-populated data
        self.project_id = "d44bcf47-5a1b-4aa0-9858-61cacd93edbf"
        self.create_project_with_sync_data(self.project_id, "tmp_project", [])

        self.project_path = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")
        if self.project_path.exists() and self.project_path.is_dir():
            shutil.rmtree(self.project_path)
        os.makedirs(self.project_path, exist_ok=True)

        watchdog_service.watch(restart_on_stop=False)

    def tearDown(self):
        self._stop_watcher_and_cleanup()
        super().tearDown()

    def _stop_watcher_and_cleanup(self):
        watchdog_service.stop()

        os.environ["JUPYTERHUB_SERVICE_URL"] = ""
        os.environ["JUPYTERHUB_API_TOKEN"] = ""

        try:
            shutil.rmtree(self.project_path)
        except FileNotFoundError:
            pass

    def copy_asset_file(self):
        asset_name = "notebook-custom.ipynb"
        current_path = Path().cwd()
        asset_filepath_relative = f"tests/test_assets/{asset_name}"

        asset_filepath = current_path / asset_filepath_relative
        destination_filepath = self.project_path / asset_name

        shutil.copy(asset_filepath, destination_filepath)
        return asset_name

    def delete_asset_file(self):
        asset_name = "notebook-custom.ipynb"
        destination_filepath = self.project_path / asset_name

        destination_filepath.unlink()

    def create_directory_in_project(self):
        asset_name = "a_directory"
        destination_filepath = self.project_path / asset_name
        os.makedirs(destination_filepath, exist_ok=True)

    def create_tmp_file_in_project(self):
        asset_name = ".tmp_file"
        destination_filepath = self.project_path / asset_name
        destination_filepath.touch()

    @respx.mock
    @tornado.testing.gen_test
    async def test_watcher_detects_new_file(self):
        # define `file/create` httpx mock
        url = f"/aext_project_filebrowser_server/project/{self.project_id}/file/create"
        create_file_route = respx.post(url).respond(status_code=200)

        # manually insert new file into the project dir
        self.copy_asset_file()
        await asyncio.sleep(1)

        # verify `file/create` request is triggered
        assert create_file_route.called
        assert create_file_route.call_count == 1

    @respx.mock
    @tornado.testing.gen_test
    async def test_watcher_detects_file_deleted(self):
        # define `file/create` httpx mock
        url = f"/aext_project_filebrowser_server/project/{self.project_id}/file/create"
        respx.post(url).respond(status_code=200)

        # copy file into relevant dir
        self.copy_asset_file()

        # define `file/delete` httpx mock
        url = f"/aext_project_filebrowser_server/project/{self.project_id}/file/delete"
        delete_file_route = respx.post(url).respond(status_code=201)

        # delete newly copied file
        self.delete_asset_file()
        await asyncio.sleep(1)

        # verify `file/delete` request is triggered
        assert delete_file_route.called
        assert delete_file_route.call_count == 1

    @tornado.testing.gen_test
    async def test_watcher_ignores_directories(self):
        # define `file/create` httpx mock
        url = f"/aext_project_filebrowser_server/project/{self.project_id}/file/create"
        create_file_route = respx.post(url).respond(status_code=200)

        # manually insert new file into the project dir
        self.create_directory_in_project()
        await asyncio.sleep(1)

        # verify `file/create` request is triggered
        assert not create_file_route.called
        assert create_file_route.call_count == 0

    @tornado.testing.gen_test
    async def test_watcher_ignores_checkpoint_files(self):
        # define `file/create` httpx mock
        url = f"/aext_project_filebrowser_server/project/{self.project_id}/file/create"
        create_file_route = respx.post(url).respond(status_code=200)

        # manually insert new file into the project dir
        self.create_tmp_file_in_project()
        await asyncio.sleep(1)

        # verify `file/create` request is triggered
        assert not create_file_route.called
        assert create_file_route.call_count == 0

    @unittest.mock.patch(
        "aext_project_filebrowser_server.services.watchdog.watchdog_service._create_new_observer_thread"
    )
    def test_watcher_init_without_restart_on_stop(self, mock_create_new_observer_thread):
        assert len(watchdog_service.active_observers) == 1
        self._stop_watcher_and_cleanup()
        assert len(watchdog_service.active_observers) == 0
        assert not mock_create_new_observer_thread.called

    @unittest.mock.patch(
        "aext_project_filebrowser_server.services.watchdog.watchdog_service._create_new_observer_thread"
    )
    def test_watcher_init_with_restart_on_stop(self, mock_create_new_observer_thread):
        assert len(watchdog_service.active_observers) == 1
        self._stop_watcher_and_cleanup()
        assert len(watchdog_service.active_observers) == 0

        watchdog_service.watch(restart_on_stop=True)
        assert len(watchdog_service.active_observers) == 1

        watchdog_service.stop()
        assert len(watchdog_service.active_observers) == 0
        assert mock_create_new_observer_thread.called
