import sqlite3
import sys
from unittest import mock

import pytest
import tornado
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.services.local_storage import local_storage
from tests.test_assets.responses import EXPECTED_LIST_PROJECTS

from .setup_tests import BasicDBAppHTTPTests
from .test_assets.project_data import LOCAL_PROJECT

pytestmark = pytest.mark.skipif(sys.platform.startswith("win"), reason="Skipping tests on Windows")


class ListProject(BasicDBAppHTTPTests):
    def tearDown(self):
        super().tearDown()

    @pytest.fixture(scope="function", autouse=True)
    def pre_populated_db(self):
        self.setUp()

        # Add pre-populated data
        project = LocalProject.create(LOCAL_PROJECT)
        project.save()

    @tornado.testing.gen_test
    async def test_list_projects(self):

        # Test data
        expected_projects = [LOCAL_PROJECT]

        # Act: Fetch the list of projects for the given owner_id
        projects = await local_storage.list_projects()

        # Assert:  Number of projects
        self.assertEqual(
            len(projects),
            len(expected_projects),
            msg=f"Expected {len(expected_projects)} projects, but got {len(projects)}",
        )

        # Assert: Validate the contents of each project
        for project, expected_project in zip(projects, expected_projects):
            self.assertEqual(
                project.id,
                expected_project["id"],
                msg=f"Expected project id {expected_project['id']}, but got {project.id}",
            )
            self.assertEqual(
                len(project.data.contents),
                len(expected_project["contents"]),
                msg=f"Expected {len(expected_project['contents'])} contents, but got {len(project.data.contents)}",
            )

    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.list_projects")
    @tornado.testing.gen_test
    async def test_list_projects_db_error(self, mock_local_storage_list_projects):
        mock_local_storage_list_projects.side_effect = sqlite3.IntegrityError("Mock Integrity Error")

        # Act & Assert: Validate the contents
        with self.assertRaises(sqlite3.IntegrityError) as context:
            await local_storage.list_projects()

        # Assert: Check the exception message
        self.assertEqual(str(context.exception), "Mock Integrity Error")

    @mock.patch("aext_project_filebrowser_server.services.local_storage.LocalStorage.list_projects")
    @tornado.testing.gen_test
    async def test_list_projects_general_error(self, mock_local_storage_list_projects):
        mock_local_storage_list_projects.side_effect = Exception("Mock General Exception")

        # Assert: Validate the contents
        with self.assertRaises(Exception) as context:
            await local_storage.list_projects()

        # Assert: Check the exception message
        self.assertEqual(str(context.exception), "Mock General Exception")

    @tornado.testing.gen_test
    async def test_project_to_frontend(self):

        projects = await local_storage.list_projects()
        assert [project.to_frontend() for project in projects] == EXPECTED_LIST_PROJECTS
