import os
import unittest

import httpx
import pytest
import tornado
from aext_project_filebrowser_server.consts import FILE_MAX_SIZE_100_MB
from aext_project_filebrowser_server.schemas.cloud_project import CloudProject
from aext_project_filebrowser_server.utils import (
    FileValidation,
    add_query_params,
    evaluate_feature_flag,
    generate_normalized_name,
    get_intermediate_directories,
    write_file_from_http_response,
)
from tests.setup_tests import BasicAppHTTPTests

from aext_shared.utils import get_boolean_env_var

from .test_assets.project_data import CLOUD_PROJECT


class FileValidationTestCase(unittest.TestCase):

    def setUp(self):
        self.file = {"filename": "my-file.txt", "content": b"aaaaaaaaaaa", "content_type": "plain/text"}

    def test_validate_file_size_should_return_false_for_files_bigger_than_the_limit(self):
        file_validation = FileValidation(
            filename=self.file.get("filename"),
            content=self.file.get("content"),
            content_type=self.file.get("content_type"),
        )

        self.assertFalse(file_validation.validate_size(max_size_bytes=1))

    def test_validate_file_size_should_return_true_for_files_below_the_limit(self):
        file_validation = FileValidation(
            filename=self.file.get("filename"),
            content=self.file.get("content"),
            content_type=self.file.get("content_type"),
        )

        self.assertTrue(file_validation.validate_size(max_size_bytes=FILE_MAX_SIZE_100_MB))

    def test_is_valid_should_return_true_for_files_below_the_limit(self):
        file_validation = FileValidation(
            filename=self.file.get("filename"),
            content=self.file.get("content"),
            content_type=self.file.get("content_type"),
        )

        self.assertTrue(file_validation.is_valid())


@pytest.mark.parametrize(
    "size,expected",
    [
        (0, "0 Bytes"),
        (1, "1 Bytes"),
        (1023, "1023 Bytes"),
        (1024, "1 KB"),
        (1536, "1 KB"),
        (1048576, "1 MB"),
        (1073741824, "1 GB"),
        (1099511627776, "1 TB"),
        (1125899906842624, "1 PB"),
        (1234567890123456, "1 PB"),
    ],
)
class UtilsTestCase(BasicAppHTTPTests):
    def test_generate_normalized_name(self):
        assert generate_normalized_name("My Project") == "my-project"
        assert generate_normalized_name("This is an App") == "this-is-an-app"

    def test_cloud_project_validate_tags(self):
        cloud_data = CLOUD_PROJECT

        cloud_project_no_tags = CloudProject.create(cloud_data)
        cloud_project_no_tags.metadata.tags = []

        cloud_project_data_tag = CloudProject.create(cloud_data)
        cloud_project_data_tag.metadata.tags = ["data"]

        cloud_project_notebook_tag = CloudProject.create(cloud_data)
        cloud_project_notebook_tag.metadata.tags = ["notebook"]

        cloud_project_both_tags = CloudProject.create(cloud_data)
        cloud_project_both_tags.metadata.tags = ["data", "notebook"]

        cloud_project_contains_code_snippet_tag = CloudProject.create(cloud_data)
        cloud_project_contains_code_snippet_tag.metadata.tags = ["code_snippet"]

        cloud_project_invalid_tag = CloudProject.create(cloud_data)
        cloud_project_invalid_tag.metadata.tags = ["invalid_tag"]

        cloud_project_contains_data_tag = CloudProject.create(cloud_data)
        cloud_project_contains_data_tag.metadata.tags = ["invalid_tag", "data"]

        cloud_project_contains_notebook_tag = CloudProject.create(cloud_data)
        cloud_project_contains_notebook_tag.metadata.tags = ["invalid_tag", "notebook"]

        self.assertFalse(cloud_project_no_tags.validate_tags())
        self.assertTrue(cloud_project_data_tag.validate_tags())
        self.assertTrue(cloud_project_notebook_tag.validate_tags())
        self.assertTrue(cloud_project_both_tags.validate_tags())
        self.assertTrue(cloud_project_contains_code_snippet_tag.validate_tags())
        self.assertFalse(cloud_project_invalid_tag.validate_tags())
        self.assertTrue(cloud_project_contains_data_tag.validate_tags())
        self.assertTrue(cloud_project_contains_notebook_tag.validate_tags())

    @tornado.testing.gen_test
    @unittest.mock.patch("aext_project_filebrowser_server.utils.cloud_request")
    async def test_evaluate_feature_flag_nonexistent_flag(self, mock_cloud_request):
        mock_cloud_request.return_value = httpx.Response(status_code=404)
        result = await evaluate_feature_flag("this-flag-does-not-exist", user_credentials={})
        self.assertFalse(result)

    @tornado.testing.gen_test
    @unittest.mock.patch("aext_project_filebrowser_server.utils.cloud_request")
    async def test_evaluate_feature_flag_disabled_flag(self, mock_cloud_request):
        mock_cloud_request.return_value = httpx.Response(status_code=400, text="false")
        result = await evaluate_feature_flag("disabled-flag", user_credentials={})
        self.assertFalse(result)

    @tornado.testing.gen_test
    @unittest.mock.patch("aext_project_filebrowser_server.utils.cloud_request")
    async def test_evaluate_feature_flag_enabled_flag(self, mock_cloud_request):
        mock_cloud_request.return_value = httpx.Response(status_code=200, text="true")
        result = await evaluate_feature_flag("enabled-flag", user_credentials={})
        self.assertTrue(result)

    @tornado.testing.gen_test
    @unittest.mock.patch("aext_project_filebrowser_server.utils.cloud_request")
    async def test_evaluate_feature_flag_error_default_false(self, mock_cloud_request):
        mock_cloud_request.side_effect = Exception("Boom!")
        result = await evaluate_feature_flag("a-flag", user_credentials={})
        self.assertFalse(result)

    @tornado.testing.gen_test
    @unittest.mock.patch("aext_project_filebrowser_server.utils.cloud_request")
    async def test_evaluate_feature_flag_error_default_true(self, mock_cloud_request):
        mock_cloud_request.side_effect = Exception("Boom!")
        result = await evaluate_feature_flag("a-flag", user_credentials={}, default=True)
        self.assertTrue(result)


@pytest.mark.parametrize("env_var_value", ["true", "True", "TRUE", "1", "y", "yes", "YES", "Yes"])
def test_get_boolean_env_var(env_var_value):
    os.environ["test_env_var"] = env_var_value
    assert get_boolean_env_var("test_env_var", env_var_value)


def test_add_query_params():
    url = "https://localhost/test"
    params = {"foo": "1", "bar": "2"}
    new_url = add_query_params(url, params)
    assert "?foo=1" in new_url
    assert "&bar=2" in new_url


def test_get_intermediate_directories():
    result = get_intermediate_directories("tmp/images/misc/jpeg/low-res")
    for file_path, file_type in result:
        assert file_path in [
            "tmp",
            "tmp/images",
            "tmp/images/misc",
            "tmp/images/misc/jpeg",
            "tmp/images/misc/jpeg/low-res",
        ]


@pytest.mark.parametrize(
    "response_args",
    [
        {"status_code": 200, "text": "foo", "headers": {"Content-Type": "text"}},
        {"status_code": 200, "content": b"foo", "headers": {"Content-Type": ""}},
    ],
)
def test_write_file_from_http_response(response_args):
    response = httpx.Response(**response_args)
    file_path = os.path.join(os.getcwd(), "test")
    write_file_from_http_response(response, os.path.join(os.getcwd(), "test"))
    assert os.path.exists(file_path)
    os.remove(file_path)
