import asyncio
import functools
import inspect
import json
import os
from dataclasses import asdict
from typing import Any, Dict, List, Optional, Type
from uuid import UUID

import httpx
import tornado
from aext_project_filebrowser_server.clients.http import ProjectAPIClientHTTP
from aext_project_filebrowser_server.consts import (
    CODE_SNIPPET_TAG,
    DEFAULT_PROJECT_TAGS,
    UNDEFINED_ENVIRONMENT,
    USER_JLAB_HOME_PATH,
    InternalOrigins,
    ProjectPermissionAction,
    ProjectPermissionRoles,
    WebsocketMessages,
)
from aext_project_filebrowser_server.exceptions import (
    AddFileErrorOnDatabase,
    AddProjectFilesError,
    CloudProjectNameAlreadyExistsError,
    CloudProjectNotFoundError,
    CloudProjectUpdatePayloadError,
    CodeSnippetsSchemaError,
    FailedToStartSyncPull,
    ListUserPermissionsError,
    LocalFileAlreadyExists,
    LocalFileNotFound,
    LocalFilePermissionError,
    LocalProjectNotFoundError,
    ProjectFailCreate,
    ProjectNameConflict,
    UserDoesNotBelongToOrganization,
)
from aext_project_filebrowser_server.metrics import metrics
from aext_project_filebrowser_server.migrations import migration_list, migration_upgrade
from aext_project_filebrowser_server.schemas.cloud_project import (
    ProjectPermissionUpdate,
)
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.schemas.websocket import WebSocketResponse
from aext_project_filebrowser_server.services.filesystem_operations import (
    filesystem_operations,
)
from aext_project_filebrowser_server.services.local_storage import local_storage
from aext_project_filebrowser_server.services.projects import ProjectService
from aext_project_filebrowser_server.services.sync import ProjectSync
from aext_project_filebrowser_server.utils import (
    DataclassEncoder,
    FileValidation,
    file_exists,
    generate_normalized_name,
    get_file_full_path,
    get_file_size,
    get_latest_kernel_spec_name,
    get_username,
    list_kernel_specs_names,
    sanitize_filename_path,
)
from jupyter_server.base.handlers import APIHandler
from sentry_sdk import configure_scope
from tornado.routing import _RuleList
from tornado.web import Application, HTTPError
from tornado.websocket import WebSocketHandler

from aext_shared import logger as custom_logger
from aext_shared.config import SHARED_CONFIG
from aext_shared.errors import (
    BackendError,
    BadRequest,
    NotFoundError,
    UnauthorizedError,
)
from aext_shared.handler import BackendHandler, create_rules
from aext_shared.utils import feature_is_enabled

logger = custom_logger.logger

config = SHARED_CONFIG

project_api_client = ProjectAPIClientHTTP()
project_service = ProjectService()
project_sync = ProjectSync()


class ProjectFileBrowserHandler(BackendHandler):
    sync = project_sync  # abstraction over push/pull operations - uses the service as base
    project_service = project_service  # class that communicates with projects API
    local_storage = local_storage

    def __init__(self, application: Application, request, **handler_kwargs):
        super().__init__(application, request, **handler_kwargs)
        project_api_client.proxy = self.anaconda_proxy
        self.project_service.client = project_api_client
        self.sync.project_service = self.project_service

    def is_valid_project_id(self, id: Optional[str]) -> bool:
        try:
            UUID(id)
            return True
        except (ValueError, TypeError):
            return False

    async def authenticate_project_service(self, accept_internal_origin: bool = False) -> Dict:
        # allow internal watchdog instance to make requests without requiring user credentials
        if (
            accept_internal_origin
            and (origin := self.request.headers.get("X-Origin-Internal", None)) == InternalOrigins.WATCHDOG.value
        ):
            return {"client": origin}

        try:
            user_credentials = await self.get_user_access_credentials()
            user_credentials = asdict(user_credentials)
            cloudflare_client_id = os.environ.get("NUCLEUS_CLOUDFLARE_CLIENT_ID", "")
            cloudflare_client_secret = os.environ.get("NUCLEUS_CLOUDFLARE_CLIENT_SECRET", "")
            user_credentials.update(
                {"CF-Access-Client-Id": cloudflare_client_id, "CF-Access-Client-Secret": cloudflare_client_secret}
            )
            self.project_service.set_user_credentials(user_credentials)
            return user_credentials
        except UnauthorizedError as ex:
            self.set_status(401, f"User does not have permissions to perform this action: {ex}")
            return self.finish()
        except Exception:
            raise


class HealthzRouteHandler(BackendHandler):
    """
    This is a healthcheck endpoint
    """

    @tornado.web.authenticated
    async def get(self):
        """
        Used to check if the server is already responding

        Returns: json containing the "live" key and True value

        """
        self.finish({"live": True})


class ListUserFilesHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def get(self):
        """
        Returns user's files stored locally

        Returns:
            {
                "files": {
                    "path": str
                    "size": int
                    "last_modified": str
                }
            }
        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            user_files = await filesystem_operations.list_all_user_files()
            self.set_status(201, "Success")
            self.set_header("Content-Type", "application/json")
            self.finish(json.dumps(user_files, cls=DataclassEncoder))


class ProjectHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self):
        """
        Creates a new project locally and in the cloud
        Body (JSON):
          title (str): project title
          default_environment (str) - Default (undefined_environment) : one of the pre-created environments we have in Anaconda Notebooks
        Returns: LocalProjectData

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            payload = self.get_json_body()
            project_title = payload.get("title")
            project_default_environment = payload.get("default_environment") or UNDEFINED_ENVIRONMENT
            user_credentials = await self.authenticate_project_service()

            if not (tags := payload.get("tags", [])):
                tags = DEFAULT_PROJECT_TAGS

            if not project_title:
                logger.info("A project title must be provided")
                raise BadRequest(
                    {
                        "reason": "A project title must be provided",
                        "remote_status_code": 400,
                    }
                )

            try:
                project = await self.project_service.create_project(project_title, project_default_environment, tags)

                if project is None:
                    logger.error("Could not create project")
                    self.set_status(500, "Could not create project")
                else:
                    await metrics.send(
                        user_credentials=user_credentials,
                        metric_data={
                            "event": metrics.EVENT_CREATE_PROJECT,
                            "event_params": {},
                            "service_id": "aext-cloud",
                            "user_environment": config["environment"].value,
                        },
                    )
                    self.set_status(201, "Success")
                    self.set_header("Content-Type", "application/json")
                    self.write(asdict(project))
            # TODO: Has to be double checked
            except (ProjectFailCreate, ProjectNameConflict) as ex:
                logger.error(f"Error creating Project. {ex}")
                self.set_status(409, "Project name already taken")
            except Exception as ex:
                logger.exception(f"Could not create Project {ex}")
                self.set_status(500, "Could not create Project")
            finally:
                self.finish()

    @tornado.web.authenticated
    async def delete(self, project_id: str):
        """
        Deletes a project. The action is also propagated to anaconda.cloud (after migration: anaconda.com)
        Args:
            project_id: Project's id

        Returns:
            {
                "success": boolean
            }

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            try:
                user_credentials = await self.authenticate_project_service()
                # validate project_id type
                if not self.is_valid_project_id(project_id):
                    raise BadRequest(
                        {
                            "reason": "Invalid project id",
                            "remote_status_code": 400,
                        }
                    )
                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_DELETE_PROJECT,
                        "event_params": {},
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )
                self.finish({"success": await self.project_service.delete_project(project_id)})
            except (json.JSONDecodeError, HTTPError) as ex:
                message = "Invalid request"
                logger.error(f"{message}: {ex}")
                raise BadRequest(
                    {
                        "reason": message,
                        "remote_status_code": 400,
                    }
                ) from ex

    @tornado.web.authenticated
    async def get(self, project_id: str):
        """
        Fetches a specific project
        Args:
            project_id: id of the project

        Returns:
            {
                "id": str,
                "name": str,
                "title": str,
                "created_at": datetime,
                "updated_at": datetime,
                "metadata": object (ProjectMetadata),
                "owner": object (ProjectOwner),
                "contents": list objects (LocalProjectContent),
                "default_environment": str,
                "status": str (LocalStorageSyncState),
            }

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            if not self.is_valid_project_id(project_id):
                raise BadRequest(
                    {
                        "reason": "Invalid project id",
                        "remote_status_code": 400,
                    }
                )

            project = LocalProject.get(project_id)
            if not project:
                self.set_status(404, "Project not found")
                return self.finish()

            self.set_status(200, "Success")
            self.set_header("Content-Type", "application/json")

            self.finish(project.to_frontend())

    @tornado.web.authenticated
    async def patch(self, project_id: str):
        """
        Updates a project
        Args:
            project_id (str): project's id
        Body (Json):
            title (str): project's title
            default_environment (str): project's default conda environment

        Returns:
            {
                "id": str,
                "name": str,
                "title": str,
                "created_at": datetime,
                "updated_at": datetime,
                "metadata": object (ProjectMetadata),
                "owner": object (ProjectOwner),
                "contents": list objects (LocalProjectContent),
                "default_environment": str,
                "status": str (LocalStorageSyncState),
            }

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            user_credentials = await self.authenticate_project_service()
            # validate project_id
            if not self.is_valid_project_id(project_id):
                raise BadRequest(
                    {
                        "reason": "Invalid project id",
                        "remote_status_code": 400,
                    }
                )
            payload = self.get_json_body() or {}
            project_title = payload.get("title", None)
            project_default_environment = payload.get("default_environment", None)

            # validate at least one update field has been provided
            if not project_title and not project_default_environment:
                logger.info("A project title or default environment must be provided")
                raise BadRequest(
                    {
                        "reason": "A project title or default environment must be provided",
                        "remote_status_code": 400,
                    }
                )

            local_project = LocalProject.get(project_id)

            # validate project exists in local db
            if not local_project:
                raise NotFoundError(
                    {"reason": f"Unable to locate project {project_id} locally", "remote_status_code": 404}
                )

            if project_title:
                try:
                    project_name = generate_normalized_name(project_title)
                    await self.project_service.update_cloud_project(
                        project_id, project_title=project_title, project_name=project_name
                    )
                    local_project.name = project_name
                    local_project.data.title = project_title
                except CloudProjectNotFoundError:
                    msg = f"Unable to locate project {project_id} on the cloud"
                    logger.warning(msg)
                    self.set_status(404, msg)
                    return self.finish({})
                except CloudProjectUpdatePayloadError:
                    msg = f"Unable to update project with the following field(s): title={project_title}"
                    logger.error(msg)
                    self.set_status(422, msg)
                    return self.finish(local_project.to_frontend())
                except CloudProjectNameAlreadyExistsError:
                    msg = f"A project with title={project_title} already exists"
                    logger.warning(msg)
                    self.set_status(409, msg)
                    return self.finish(local_project.to_frontend())
                except Exception:
                    msg = f"Unable to update project {local_project.id}"
                    logger.error(msg)
                    self.set_status(500, msg)
                    return self.finish({})

            if project_default_environment:
                local_project.default_environment = project_default_environment

            # write entire project transaction to db
            local_project.update()

            await metrics.send(
                user_credentials=user_credentials,
                metric_data={
                    "event": metrics.EVENT_UPDATE_PROJECT,
                    "event_params": {
                        "title": project_title,
                        "default_environment": project_default_environment,
                    },
                    "service_id": "aext-cloud",
                    "user_environment": config["environment"].value,
                },
            )
            self.set_status(200, "Success")
            return self.finish(local_project.to_frontend())


class ProjectPullAllRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def get(self):
        """
        Pulls all user's projects from anaconda.cloud (after migration: anaconda.com)

        It creates the missing one locally and deletes any projects that may have been removed
        from the cloud

        Returns: [
            {
                "id": str,
                "name": str,
                "title": str,
                "created_at": datetime,
                "updated_at": datetime,
                "metadata": object (ProjectMetadata),
                "owner": object (ProjectOwner),
                "contents": list objects (LocalProjectContent),
                "default_environment": str,
                "status": str (LocalStorageSyncState),
            }
        ]

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )

            try:
                user_credentials = await self.authenticate_project_service()

                projects = await self.sync.pull_all()
                projects_dict_list = [project.to_frontend() for project in projects]
                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_PULL_ALL,
                        "event_params": {
                            "num_projects": len(projects_dict_list),
                        },
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )

                self.finish(json.dumps(projects_dict_list))
            except BackendError:
                self.set_status(500, "Error while pulling all projects")
                self.finish()
            except FailedToStartSyncPull:
                self.set_status(500, "Error while starting pull process for all projects.")
                self.finish()
            except FileNotFoundError:
                self.set_status(404, "Project not found")
                return self.finish()
            except Exception:
                self.set_status(500, "An unexpected error occurred")
                return self.finish()


class ProjectPushAllRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def get(self):
        """
        Pushes all projects to the cloud containing the latest changes in the projects

        Returns: {
            "items": {
                "project_id": boolean
            }
        }

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            user_credentials = await self.authenticate_project_service()
            projects = LocalProject.list()
            if not projects:
                self.set_status(404, "There are no projects to be pushed")
                return self.finish()
            else:
                try:
                    contents = await self.sync.push_all()
                except BackendError:
                    self.set_status(403, "User can't modify project")
                    return self.finish()
                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_PUSH_ALL,
                        "event_params": {
                            "num_projects": len(projects),
                        },
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )
                self.finish({"items": contents})


class ProjectPullOneRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def get(self, project_id: str):
        """
        Pull changes from the cloud for a specific project.
        Files that are present on the cloud but not locally will be added to the project
        Files that are different from the cloud will be overwritten locally
        Files that are NOT present on the cloud will be deleted locally
        Args:
            project_id: project's id

        Returns:
            {
                "id": str,
                "name": str,
                "title": str,
                "created_at": datetime,
                "updated_at": datetime,
                "metadata": object (ProjectMetadata),
                "owner": object (ProjectOwner),
                "contents": list objects (LocalProjectContent),
                "default_environment": str,
                "status": str (LocalStorageSyncState),
            }

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            if not self.is_valid_project_id(project_id):
                raise BadRequest(
                    {
                        "reason": "Invalid project id",
                        "remote_status_code": 400,
                    }
                )

            user_credentials = await self.authenticate_project_service()
            try:
                project = await self.sync.pull(project_id)
                if not project:
                    return self.finish({})
                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_PULL_ONE,
                        "event_params": {
                            "project_id": project_id,
                        },
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )
                return self.finish(project.to_frontend())
            except FailedToStartSyncPull:
                self.set_status(500, "Filed to start sync process")
                return self.finish()
            except FileNotFoundError:
                self.set_status(404, "Project not found")
                return self.finish()


class ProjectPushOneRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def get(self, project_id: str):
        """
        Pushes one single project to the cloud, including the changes made to it.
        Args:
            project_id (str): project's id

        Returns: {
            "items": {
                "project_id": boolean
            }
        }

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            if not self.is_valid_project_id(project_id):
                raise BadRequest(
                    {
                        "reason": "Invalid project id",
                        "remote_status_code": 400,
                    }
                )
            user_credentials = await self.authenticate_project_service()
            project = LocalProject.get(project_id)
            if project is None:
                self.set_status(404, f"Project {project_id} not found")
                return self.finish()
            else:
                try:
                    sync_started = await self.sync.push(project_id, user_credentials=user_credentials)
                except BackendError:
                    self.set_status(403, "User can't modify project")
                    return self.finish()
                if sync_started:
                    await metrics.send(
                        user_credentials=user_credentials,
                        metric_data={
                            "event": metrics.EVENT_PUSH_ONE,
                            "event_params": {
                                "project_id": project_id,
                            },
                            "service_id": "aext-cloud",
                            "user_environment": config["environment"].value,
                        },
                    )
                    self.finish({"items": {f"{project_id}": True}})
                else:
                    raise BackendError(
                        status_code=500,
                        data={
                            "reason": "Failed to start push process",
                            "remote_status_code": 500,
                        },
                    )


class DBMigration(APIHandler):
    async def get(self):
        output = migration_upgrade()
        if output is not None:
            return self.finish(output)

        self.set_status(500)
        return self.finish()

    async def post(self):
        output = migration_list()
        if output is not None:
            return self.finish(output)

        self.set_status(500)
        return self.finish()


class ProjectDownloadFileHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def get(self, project_id: str, filename: str):
        """
        Downloads a file that belongs to a project
        Args:
            project_id (str): project's id
            filename (str): filename of the project that will be downloaded

        Returns (bytes): file content

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            if not self.is_valid_project_id(project_id):
                raise BadRequest(
                    {
                        "reason": "Invalid project id",
                        "remote_status_code": 400,
                    }
                )

            logger.info(f"Downloading {filename} from {project_id}")
            file_content = await self.project_service.download_file(project_id, filename, parse_response=False)

            if file_content:
                self.set_header("Content-Type", "text/plain")
                self.set_header("Content-Disposition", f"attachment; filename={filename}")
                return self.finish(file_content)

            logger.error(f"Error retrieving {filename} from {project_id}")
            self.set_status(500)
            self.finish()


def websocket_authenticated(method):
    """
    Decorator to authenticate WebSocket handler methods.

    Checks if the user is authenticated before executing the method.
    If the user is not authenticated, sends a message and closes the connection.
    """

    @functools.wraps(method)
    async def wrapper(self, *args, **kwargs):
        logger.debug(f"user token: {self.token}. username: {get_username()}")
        if self.token is None or get_username() == "unknown":
            logger.error("User not found or token not valid. Closing connection.")
            message = kwargs.get("message", None)
            response = WebSocketResponse(
                http_status=self.get_status(),
                http_message="User not found or token not valid. Closing connection.",
                message=message,
                data=None,
            )
            await self.write_message(response.to_json())
            self.close()
            return
        return await method(self, *args, **kwargs)

    return wrapper


class ProjectListWebSocketRouteHandler(ProjectFileBrowserHandler, WebSocketHandler):
    connections = set()

    def initialize(self):
        self.list_projects_update_interval = 10  # in secs
        self.close_connection_timeout = 120  # in secs
        self.last_message_time = None
        self.timeout_task = None

    @websocket_authenticated
    async def open(self) -> None:
        """
        Handles the WebSocket connection open event.
        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            self.last_message_time = asyncio.get_event_loop().time()
            self.timeout_task = asyncio.ensure_future(self.check_timeout())
            self.connections.add(self)

    @websocket_authenticated
    async def send_data(self, message: WebsocketMessages, data: Any) -> None:
        """
        Fetches the data associated with a request message and sends it over WebSocket.
        """
        response = WebSocketResponse(http_status=200, message=message, data=data)
        await self.write_message(response.to_json())

    async def check_timeout(self) -> None:
        """
        Checks for timeout and closes the connection if no message is received for 120 seconds
        """
        while True:
            current_time = asyncio.get_event_loop().time()
            logger.debug(f"counting: {current_time - self.last_message_time}")
            if current_time - self.last_message_time > self.close_connection_timeout:
                self.close()
                break
            await asyncio.sleep(10)

    @websocket_authenticated
    async def on_message(self, message):
        """
        Handle incoming WebSocket messages if needed
        """
        self.last_message_time = asyncio.get_event_loop().time()
        data = None
        if message == WebsocketMessages.START:
            with configure_scope() as sentry_scope:
                sentry_scope.set_transaction_name(
                    f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
                )
                projects = await self.local_storage.list_projects()
                data = [project.to_frontend() for project in projects]

        await self.send_data(message, data)

    def on_close(self):
        """
        Handles the WebSocket connection close event.
        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            if self.timeout_task:
                self.timeout_task.cancel()

            self.connections.remove(self)
            self.close()
            print("WebSocket connection closed.")

    @classmethod
    async def broadcast(cls, message: WebsocketMessages, data: Any) -> None:
        """
        Send an update message to all opened connections
        Args:
            message: string containing the message
            data: any serializable data to include in the message

        Returns: None

        """
        logger.debug(f"Sending WS update: ({str(message)}, {data})")
        for connection in cls.connections:
            try:
                response = WebSocketResponse(http_status=200, message=message, data=data)
                connection.write_message(response.to_json())
            except tornado.websocket.WebSocketError:
                logger.debug("Websocket closed. Skipping")
            except Exception as ex:
                logger.warning(f"Failed to send websocket updates {ex}")

    @classmethod
    async def send_projects_update(cls) -> None:
        """
        Send a list containing the most update info of all users' projects

        Returns: None

        """
        projects = await cls.local_storage.list_projects()
        data = [project.to_frontend() for project in projects]
        await cls.broadcast(WebsocketMessages.START, data)


class ProjectCreateFileRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self, project_id: str):
        """
        Creates a new project's file
        Args:
            project_id (str): Projects that the file belongs to
        Body:
            filename (str): newly created filename
        Returns (str): "File created"

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )

            if not (local_project := LocalProject.get(project_id)):
                self.set_status(404, f"Local project (id={project_id}) not found")
                return self.finish()

            payload = self.get_json_body() or {}
            filename = payload.get("filename", None)
            if not filename:
                self.set_status(400, "Filename not provided")
                return self.finish()

            metadata = payload.get("metadata", {})
            user_credentials = await self.authenticate_project_service(accept_internal_origin=True)

            file_relative_path = sanitize_filename_path(filename)
            file_size = get_file_size(get_file_full_path(file_relative_path, project_id))
            file_is_code_snippet = bool(metadata.get(CODE_SNIPPET_TAG, False))

            if not await project_service.create_project_file(project_id, filename):
                self.set_status(500, "File not created")
            else:
                try:
                    if metadata:
                        updated_metadata = await project_service.update_file_metadata(
                            project_id, file_relative_path, metadata
                        )
                        if updated_metadata:
                            self.set_status(200, "File created")
                        else:
                            # Since metadata failed the file must be removed
                            await project_service.delete_project_file(project_id, filename_paths=[filename])
                            self.set_status(500, "Failed to update metadata")
                except PermissionError:
                    self.set_status(403, f"User does not have permissions to change {filename}")
                except CloudProjectNotFoundError:
                    self.set_status(404, "Project not found on cloud")
                except CodeSnippetsSchemaError:
                    self.set_status(400, "Code snippets schema is invalid")
                    file_is_code_snippet = False
                except BackendError as ex:
                    self.set_status(500, "Could not update file metadata")
                    logger.exception(ex)

            await metrics.send(
                user_credentials=user_credentials,
                metric_data={
                    "event": metrics.EVENT_CREATE_FILE,
                    "event_params": {
                        "file_size": file_size,
                        "num_files": 1,
                    },
                    "service_id": "aext-cloud",
                    "user_environment": config["environment"].value,
                },
            )

            auto_push = self.get_query_argument("auto_push", False) == "true"
            if file_is_code_snippet:
                await self.project_service.sync_push_project_files(project_id, [file_relative_path])

                project_metadata = local_project.data.metadata
                if CODE_SNIPPET_TAG not in project_metadata.tags:
                    project_metadata.tags += [CODE_SNIPPET_TAG]
                    await self.project_service.update_project_metadata(project_id, project_metadata)

                await self.project_service.update_cloud_project(project_id, project_metadata=project_metadata)
            elif auto_push:
                await self.project_service.sync_push_project_files(project_id)

            await ProjectListWebSocketRouteHandler.send_projects_update()

            self.finish()


class ProjectAddFileRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self, project_id: str):
        """
        Adds already created files in JupyterLab to the project
        Args:
            project_id (str): project's id
        Body:
            filename_paths (list): all the files that will be added to the project
        Returns (str): "Files added to the Project"

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            user_credentials = await self.authenticate_project_service()
            payload = self.get_json_body()
            filenames = payload.get("filename_paths", [])
            if not filenames:
                self.set_status(400, "File paths not provided")
                self.finish()
                return

            try:
                for file_info in filenames:
                    file_info["src_path"] = os.path.join(USER_JLAB_HOME_PATH, file_info["src_path"])

                    base_filename = os.path.basename(file_info["src_path"])
                    if not file_exists(file_info["src_path"]):
                        self.set_status(400, f"File {base_filename} does not exist")
                        return self.finish()

                    if not FileValidation(file_full_path=file_info["src_path"]).is_valid():
                        self.set_status(413, f"File {base_filename} is too big")
                        return self.finish()

                await project_service.add_project_files(project_id, filenames)
                file_sizes = {}
                total_size = 0
                for idx, file_src_dest_dict in enumerate(filenames):
                    file_size = get_file_size(
                        get_file_full_path(sanitize_filename_path(file_src_dest_dict["dest_path"]), project_id)
                    )
                    file_sizes[f"file-{idx + 1}"] = file_size
                    total_size += file_size

                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_ADD_FILE,
                        "event_params": {
                            "file_sizes": file_sizes,
                            "num_files": len(filenames),
                            "total_size": total_size,
                        },
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )
                self.set_status(201, "Files added to the Project")
            except LocalProjectNotFoundError:
                self.set_status(404, "Project not found")
            except (AddFileErrorOnDatabase, AddProjectFilesError):
                self.set_status(400, "Failed to add Project files")

            auto_push = self.get_query_argument("auto_push", False) == "true"
            if auto_push:
                await self.project_service.sync_push_project_files(project_id)

            await ProjectListWebSocketRouteHandler.send_projects_update()

            self.finish()


class ProjectPushFileRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self, project_id: str):
        """
        Pushes a file that was created on a project to cloud
        Args:
            project_id (str): Projects that the file belongs to
        Body:
            filename (str): filename of the file to be pushed
        Returns (str): "File pushed"

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )

            if not (LocalProject.get(project_id)):
                self.set_status(404, f"Local project (id={project_id}) not found")
                return self.finish()

            payload = self.get_json_body() or {}
            filename = payload.get("filename", None)
            if not filename:
                self.set_status(400, "Filename not provided")
                return self.finish()

            file_relative_path = sanitize_filename_path(filename)

            await self.project_service.sync_push_project_files(project_id, [file_relative_path])

            await ProjectListWebSocketRouteHandler.send_projects_update()

            self.finish()


class ProjectUploadFileRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self, project_id: str, dir_name: Optional[str] = None):
        """
        Upload files from users machine to the project
        Args:
            project_id (str): project's id
            dir_name (str): directory that the file will be added to
        Body:
            filenames (list): all the files that will be added to the project
        Returns (str): "done"

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            user_credentials = await self.authenticate_project_service()
            dir_name = self.get_argument("dir_name", None)
            if self.request.files:
                file_sizes = {}
                total_size = 0
                filenames = self.request.files.get("filenames", [])
                for idx, filename in enumerate(filenames):
                    validation = FileValidation(
                        filename=filename["filename"], content=filename["body"], content_type=filename["content_type"]
                    )
                    file_size = len(filename["body"])
                    file_sizes[f"file_{idx + 1}"] = file_size
                    total_size += file_size
                    if not validation.is_valid():
                        self.set_status(413, f"File {validation.filename} is too big")
                        return self.finish()

                await project_service.upload_project_files(project_id, dir_name, filenames)
                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_CREATE_FILE,
                        "event_params": {
                            "file_sizes": file_sizes,
                            "num_files": len(filenames),
                            "total_size": total_size,
                        },
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )

                auto_push = self.get_query_argument("auto_push", False) == "true"
                if auto_push:
                    await self.project_service.sync_push_project_files(project_id)

                await ProjectListWebSocketRouteHandler.send_projects_update()

            self.finish("done")


class ProjectDeleteFileRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self, project_id: str):
        """
        Deletes file from the project
        Args:
            project_id (str): project's id
        Body:
            filenames (list): list containing the filenames that will be deleted
        Returns:

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )

            user_credentials = await self.authenticate_project_service(accept_internal_origin=True)

            payload = self.get_json_body()
            filename_paths = payload.get("filenames", [])

            if filename_paths and project_id:
                await project_service.delete_project_file(project_id, filename_paths)

            await metrics.send(
                user_credentials=user_credentials,
                metric_data={
                    "event": metrics.EVENT_DELETE_FILE,
                    "event_params": {},
                    "service_id": "aext-cloud",
                    "user_environment": config["environment"].value,
                },
            )

            if local_project := LocalProject.get(project_id):
                # we need to check for additional code snippet files
                # if they are present, do nothing (no-op)
                # if not, then remove `code_snippet` tag from project
                if not local_project.contains_snippet_files():
                    project_metadata = local_project.data.metadata
                    # only remove the `code_snippet` tag if present in the project's metadata tags
                    # then update the project with the new metadata tags
                    if CODE_SNIPPET_TAG in project_metadata.tags:
                        project_metadata.tags.remove(CODE_SNIPPET_TAG)
                        await self.project_service.update_project_metadata(project_id, project_metadata)

            await ProjectListWebSocketRouteHandler.send_projects_update()

            self.set_status(201, "Success")
            self.finish()


class ProjectUpdateFileRouteHandler(ProjectFileBrowserHandler):
    @tornado.web.authenticated
    async def post(self, project_id: str):
        """
        Update a file's project information
        Args:
            project_id (str): project's id
        Body:
            file_relative_path (str): file's relative path
            new_filename (str): the new file's name

        Returns:

        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            user_credentials = await self.authenticate_project_service()
            payload = self.get_json_body()
            current_file_relative_path = payload.get("file_relative_path")
            new_filename = payload.get("new_filename")
            if not current_file_relative_path:
                self.set_status(400, "File relative path not specified")
            if not new_filename:
                self.set_status(400, "The new filename should be specified")
            else:
                try:
                    local_storage.rename_object(project_id, current_file_relative_path, new_filename)
                    await metrics.send(
                        user_credentials=user_credentials,
                        metric_data={
                            "event": metrics.EVENT_UPDATE_FILE,
                            "event_params": {},
                            "service_id": "aext-cloud",
                            "user_environment": config["environment"].value,
                        },
                    )
                    self.set_status(204, "File renamed")
                except LocalFileNotFound:
                    self.set_status(404, "Local file not found")
                except LocalFilePermissionError:
                    self.set_status(403, f"Use does not have permission to rename file {current_file_relative_path}")
                except LocalFileAlreadyExists:
                    self.set_status(409, f"The file {new_filename} already exists")
                except LocalProjectNotFoundError:
                    self.set_status(404, "Project not found")
                except Exception:
                    self.set_status(500, "Unexpected error while renaming file")
            self.finish()


class ProjectSharingPermissionsRouteHandler(ProjectFileBrowserHandler):
    def _validate_project_ownership(self, project_id: str, user_credentials: Dict) -> bool:
        project = LocalProject.get(project_id)
        if not project:
            self.set_status(404, "Project not found")
            return False
        if not project.is_owned_by(user_credentials["username"]):
            self.set_status(403, "Only owners can manage permissions")
            return False
        return True

    def _validate_update_payload(self, payload: Dict) -> Optional[List[ProjectPermissionUpdate]]:
        permissions = []
        for payload_permission in payload.get("permissions", []):
            if payload_permission["permission"] not in ProjectPermissionRoles.list_values():
                self.set_status(400, f"Invalid permission {payload_permission['permission']}")
                return
            if payload_permission["action"] not in ProjectPermissionAction.list_values():
                self.set_status(400, f"Invalid action {payload_permission['action']}")
                return
            permissions.append(ProjectPermissionUpdate(**payload_permission))
        return permissions

    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def get(self, project_id: str):
        """
        List all sharing permissions for a specific project taking into account
        the organization scope
        Returns: A list containing the permissions
        Args:
            project_id: project's id
            organization_names: list of organization names that the users belong to
        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            try:
                user_credentials = await self.authenticate_project_service()
                organization_names = self.get_arguments("organizations") or []
                if not organization_names:
                    self.set_status(400, "At least one organization should be provided")
                    return self.finish()
                if not self._validate_project_ownership(project_id, user_credentials):
                    return self.finish()
                permissions = await project_service.list_project_permission(project_id, organization_names)
                self.finish(asdict(permissions))
            except PermissionError:
                self.set_status(403, "Permission denied")
            except CloudProjectNotFoundError:
                self.set_status(404, "Project not found")
            except Exception as ex:
                logger.error(ex)
                self.set_status(500, "Unknown error")

    @tornado.web.authenticated
    @feature_is_enabled("cloud_notebooks")
    async def post(self, project_id: str):
        """
        Updates project's permission for a given organization
        Args:
            project_id: project's id
        """
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            try:
                user_credentials = await self.authenticate_project_service()
                if not self._validate_project_ownership(project_id, user_credentials):
                    return self.finish()

                payload = self.get_json_body()
                permissions = self._validate_update_payload(payload)
                if permissions is None:
                    return self.finish()
                if len(permissions) == 0:
                    self.set_status(400, "Missing permissions")
                    return self.finish()
                updated = await project_service.update_project_permissions(project_id, permissions)
                if not updated:
                    self.set_status(500, "Fail to update permissions")
                    return self.finish()

                await metrics.send(
                    user_credentials=user_credentials,
                    metric_data={
                        "event": metrics.EVENT_UPDATE_PROJECT_PERMISSIONS,
                        "event_params": {
                            "project_id": project_id,
                            "permissions": [
                                {
                                    "action": perm.get("action"),
                                    "permission": perm.get("permission"),
                                    "org_id": perm.get("org_id"),
                                }
                                for perm in payload.get("permissions", [])
                            ],
                        },
                        "service_id": "aext-cloud",
                        "user_environment": config["environment"].value,
                    },
                )
                self.set_status(200, "Permissions updated")
                return self.finish()
            except PermissionError:
                self.set_status(403, "Permission denied")
            except CloudProjectNotFoundError:
                self.set_status(404, "Project not found")
            except UserDoesNotBelongToOrganization:
                self.set_status(403, "One of the users does not belong to the company")
            except Exception as ex:
                logger.error(ex)
                self.set_status(500, "Unknown error")


class UserProjectPermissions(ProjectFileBrowserHandler):
    async def get(self, project_id: str):
        with configure_scope() as sentry_scope:
            sentry_scope.set_transaction_name(
                f"{self.__class__.__name__} {inspect.currentframe().f_code.co_name.upper()}"
            )
            try:
                permissions = await project_service.list_my_permissions(project_id)
                self.set_status(httpx.codes.OK, "User's permission")
                return self.finish(asdict(permissions))
            except ListUserPermissionsError as ex:
                logger.error(ex)
                self.set_status(httpx.codes.INTERNAL_SERVER_ERROR, "Could not list user's permissions")
            except UnauthorizedError as ex:
                self.set_status(
                    httpx.codes.UNAUTHORIZED, f"User does not have permissions to perform this action: {ex}"
                )
            except PermissionError:
                self.set_status(httpx.codes.FORBIDDEN, "Permission denied")
            except CloudProjectNotFoundError:
                self.set_status(httpx.codes.NOT_FOUND, "Project not found")
            except Exception as ex:
                logger.exception(ex)
                self.set_status(httpx.codes.INTERNAL_SERVER_ERROR, "Unknown error")


def get_routes(base_url: str) -> _RuleList:
    handlers: Dict[str, Type[BackendHandler]] = {
        "healthz": HealthzRouteHandler,
        "jlab/files/list": ListUserFilesHandler,
        "project": ProjectHandler,
        "project/pull": ProjectPullAllRouteHandler,
        "project/push": ProjectPushAllRouteHandler,
        "project/list": ProjectListWebSocketRouteHandler,
        r"project/pull/([^/]+)": ProjectPullOneRouteHandler,
        r"project/push/([^/]+)": ProjectPushOneRouteHandler,
        r"project/([^/]+)": ProjectHandler,
        r"project/([^/]+)/permissions/my": UserProjectPermissions,
        r"project/(.*)/file/upload": ProjectUploadFileRouteHandler,
        r"project/(.*)/file/create": ProjectCreateFileRouteHandler,
        r"project/(.*)/file/push": ProjectPushFileRouteHandler,
        r"project/(.*)/file/add": ProjectAddFileRouteHandler,
        r"project/(.*)/file/delete": ProjectDeleteFileRouteHandler,
        r"project/(.*)/file/update": ProjectUpdateFileRouteHandler,
        r"project/(.*)/download/file/(.*)": ProjectDownloadFileHandler,
        r"project/(.*)/permissions": ProjectSharingPermissionsRouteHandler,
        "db/migrations": DBMigration,
    }
    return create_rules(base_url, "aext_project_filebrowser_server", handlers)
