from typing import Dict, List, Optional, Tuple

import httpx
from aext_project_filebrowser_server.consts import (
    PROJECTS_API_VERSION,
    THIRTY_SECONDS_TIMEOUT,
    RequestMethods,
)

from aext_shared.backend_proxy import CERTIFICATES_LOCATION
from aext_shared.config import SHARED_CONFIG

config = SHARED_CONFIG


def get_headers(user_credentials: Dict) -> Dict:
    """
    Given user credentials it returns a dict containing headers that can be passed to the Request
    Args:
        user_credentials: Dict containing user credentials

    Returns:

    """
    headers = {
        "Authorization": f"Bearer {user_credentials['access_token']}",
        "Api-Version": PROJECTS_API_VERSION,
        "User-Agent": httpx._client.USER_AGENT,
    }

    if user_credentials["CF-Access-Client-Id"] and user_credentials["CF-Access-Client-Secret"]:
        headers.update(
            {
                "CF-Access-Client-Id": user_credentials["CF-Access-Client-Id"],
                "CF-Access-Client-Secret": user_credentials["CF-Access-Client-Secret"],
            }
        )

    return headers


async def make_request(request: httpx.Request, timeout=None) -> httpx.Response:
    """
    Makes a request using HTTPX
    Args:
        request: httpx Request object
        timeout: Request timeout

    Returns: httpx Response
    """
    async with httpx.AsyncClient(follow_redirects=True, timeout=timeout, verify=CERTIFICATES_LOCATION) as client:
        response: httpx.Response = await client.send(request)
        return response


async def cloud_request(
    uri: str,
    method: RequestMethods,
    user_credentials: Dict,
    files: Optional[List[Tuple[str, Tuple]]] = None,
    timeout: Optional[int] = THIRTY_SECONDS_TIMEOUT,
    json_payload: Dict = None,
) -> httpx.Response:
    """
    Makes a request to anaconda.cloud (after migration: anaconda.com) - it can be used in scenarios where it is not possible to use
    the cloud_backend_proxy.
    Args:
        uri: URI (without the domain) that will be used in the request
        method: HTTP method
        user_credentials: dict containing use credentials
        files: files in case they are needed
        timeout: request timeout
        json_payload: dict containing the json payload

    Returns: httpx Response

    """
    if uri.startswith("/"):
        raise ValueError("URI should NOT start with /")

    base_url = config["api"]["url"]
    url = f"{base_url}/{uri}"

    if json_payload is None:
        json_payload = {}

    if files:
        request = httpx.Request(method, url, headers=get_headers(user_credentials), files=files)
    else:
        request = httpx.Request(method, url, json=json_payload, headers=get_headers(user_credentials))

    return await make_request(request, timeout)
