# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for tools provided with CloudAPI."""

from __future__ import annotations

__all__ = ()

import itertools
import json
import typing
from unittest import mock
import requests


def test_error_parsers():
    """Check behavior of :class:`~anaconda_navigator.api.cloud.tools.error_parsers.HttpErrorHandlers`."""
    from anaconda_navigator.api.cloud import tools

    exception_types: typing.Final[typing.Sequence[typing.Type[BaseException]]] = [
        OSError, TypeError, FileNotFoundError, Exception, BaseException, ValueError,
    ]
    http_contexts: typing.Final[typing.Sequence[typing.Tuple[typing.Any, ...]]] = [
        (409,), (409, 'environment_already_exists'), (409, 'environment_max_count_reached'),
        (),
        (401, 'environment_already_exists'), (401,), (401, 'environment_max_count_reached')
    ]
    handlers: typing.Final[typing.Mapping[typing.Any, mock.Mock]] = {
        key: mock.Mock()
        for key in itertools.chain(exception_types, http_contexts)
    }

    # initialization

    instance: tools.HttpErrorHandlers = tools.HttpErrorHandlers()

    exception_type: typing.Type[BaseException]
    for exception_type in exception_types:
        instance.register_handler(exception_type, handlers[exception_type])

    http_context: typing.Tuple[typing.Any, ...]
    for http_context in http_contexts:
        instance.register_http_handler(handlers[http_context], *http_context)

    # helpers

    def assert_called(item: typing.Any, *args: typing.Any, **kwargs: typing.Any) -> None:
        key: typing.Any
        value: mock.Mock
        for key, value in handlers.items():
            if key == item:
                value.assert_called_once_with(*args, **kwargs)
                value.reset_mock()
            else:
                value.assert_not_called()

    # checks

    exception: BaseException
    for exception_type in exception_types:
        exception: BaseException = exception_type()
        instance.handle(exception)
        assert_called(exception_type, exception)

    exception = FileExistsError()
    instance.handle(exception)
    assert_called(OSError, exception)

    for http_context in http_contexts:
        response: requests.Response = requests.Response()
        if len(http_context) >= 1:
            response.status_code = http_context[0]
        if len(http_context) >= 2:
            response.encoding = 'utf-8'
            response._content = json.dumps({  # pylint: disable=protected-access
                'error': {
                    'code': http_context[1],
                    'message': 'ERROR!',
                },
            }).encode(response.encoding)
            response._content_consumed = True  # pylint: disable=protected-access

        exception = requests.RequestException(response=response)
        instance.handle(exception)
        assert_called(http_context, exception)
