# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

"""Tests for worker infrastructure."""

from __future__ import annotations

__all__ = ()

import time
import typing
import pytest
from qtpy import QtCore


@pytest.fixture(scope='function')
def function_task():
    """Prepare a simple task function."""
    from anaconda_navigator.utils import workers

    @workers.Task
    def result(
            *args: typing.Any,
            **kwargs: typing.Any,
    ) -> typing.Tuple[typing.Tuple[typing.Any, ...], typing.Mapping[str, typing.Any]]:
        """Return provided arguments."""
        return args, kwargs

    return result


@pytest.fixture(scope='function')
def task_collection():
    """Prepare a class with tasks methods."""
    from anaconda_navigator.utils import workers

    class Result:
        """Simple class with a collection of different tasks."""

        __slots__ = ()

        @workers.Task()
        def success_task(  # pylint: disable=no-self-argument
                *args: typing.Any,
                **kwargs: typing.Any,
        ) -> typing.Tuple[typing.Tuple[typing.Any, ...], typing.Mapping[str, typing.Any]]:
            """Return provided arguments."""
            return args, kwargs

        @workers.Task
        def failure_task(self) -> typing.NoReturn:
            """Raise an exception."""
            raise TypeError()

        @workers.Task(workers.AddCancelContext())
        def cancellable_task(self, context: workers.CancelContext) -> None:
            """Do nothing and cancel it if required."""
            time.sleep(1)
            context.abort_if_canceled()

        @workers.Task
        def sleep_task(self) -> None:
            """Do nothing."""
            time.sleep(1)

        @workers.Task(workers.InsertArgument('q', 0), workers.InsertArgument(True, 'new_arg'))
        def modified_task(  # pylint: disable=no-self-argument
                *args: typing.Any,
                **kwargs: typing.Any,
        ) -> typing.Tuple[typing.Tuple[typing.Any, ...], typing.Mapping[str, typing.Any]]:
            """Return provided arguments."""
            return args, kwargs

    return Result


def raise_error() -> typing.NoReturn:
    """Raise an error on call."""
    raise TypeError()


class SignalReader(QtCore.QObject):  # pylint: disable=too-few-public-methods
    """Helper for accessing signal result."""

    def __init__(self, signal: typing.Any) -> None:
        """Initialize new :class:`~SignalReader` instance."""
        super().__init__()
        self.__result: typing.Optional[typing.Tuple[typing.Any, ...]] = None
        signal.connect(self.__fetch)

    @property
    def result(self) -> typing.Tuple[typing.Any, ...]:  # noqa: D401
        """Fetched signal result."""
        if self.__result is None:
            raise TypeError()
        return self.__result

    def __fetch(self, *args: typing.Any) -> None:
        """Fetch signal result."""
        self.__result = args


def test_task_direct_call(function_task):  # pylint: disable=redefined-outer-name
    args: typing.Tuple[typing.Any, ...]
    kwargs: typing.Mapping[str, typing.Any]
    args, kwargs = function_task(1, 'two', 3.14, left='L', right='->')
    assert list(args) == [1, 'two', 3.14]
    assert dict(kwargs) == {'left': 'L', 'right': '->'}


def test_task_worker_call(qtbot, function_task):  # pylint: disable=redefined-outer-name
    from anaconda_navigator.utils import workers

    worker: workers.TaskWorker = function_task.worker('one', 2, 'pi', up='top', down='bottom')
    done: SignalReader = SignalReader(worker.signals.sig_done)
    succeeded: SignalReader = SignalReader(worker.signals.sig_succeeded)
    worker.signals.sig_failed.connect(raise_error)
    worker.signals.sig_canceled.connect(raise_error)
    with qtbot.waitSignals(
            [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_succeeded],
            timeout=2000,
    ):
        worker.start()

    assert list(worker.call.args) == ['one', 2, 'pi']
    assert dict(worker.call.kwargs) == {'up': 'top', 'down': 'bottom'}
    assert done.result[0] is worker.result
    assert succeeded.result[0] is worker.result

    assert worker.result.call is worker.call
    assert worker.result.duration is not None
    assert worker.result.exception is None
    assert list(worker.result.result[0]) == ['one', 2, 'pi']
    assert dict(worker.result.result[1]) == {'up': 'top', 'down': 'bottom'}
    assert worker.result.status == workers.TaskStatus.SUCCEEDED


def test_success_task(qtbot, task_collection):  # pylint: disable=redefined-outer-name
    from anaconda_navigator.utils import workers

    instance = task_collection()

    # unbound call
    args: typing.Tuple[typing.Any, ...]
    kwargs: typing.Mapping[str, typing.Any]
    args, kwargs = task_collection.success_task('1', '22', '333', four='4', five='V')
    assert list(args) == ['1', '22', '333']
    assert dict(kwargs) == {'four': '4', 'five': 'V'}

    # bound call
    args, kwargs = instance.success_task('111', '2', '33', four='IV', five=5)
    assert list(args) == [instance, '111', '2', '33']
    assert dict(kwargs) == {'four': 'IV', 'five': 5}

    # unbound worker
    worker = task_collection.success_task.worker('11', '222', '3', four=4, five='5')
    worker.signals.sig_failed.connect(raise_error)
    worker.signals.sig_canceled.connect(raise_error)
    with qtbot.waitSignals(
            [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_succeeded],
            timeout=2000,
    ):
        worker.start()

    assert list(worker.call.args) == ['11', '222', '3']
    assert dict(worker.call.kwargs) == {'four': 4, 'five': '5'}

    assert worker.result.exception is None
    assert list(worker.result.result[0]) == ['11', '222', '3']
    assert dict(worker.result.result[1]) == {'four': 4, 'five': '5'}
    assert worker.result.status == workers.TaskStatus.SUCCEEDED

    # bound worker
    worker = instance.success_task.worker('I', 'II', 'III', four='8/2', five='2+3')
    worker.signals.sig_failed.connect(raise_error)
    worker.signals.sig_canceled.connect(raise_error)
    with qtbot.waitSignals(
            [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_succeeded],
            timeout=2000,
    ):
        worker.start()

    assert list(worker.call.args) == [instance, 'I', 'II', 'III']
    assert dict(worker.call.kwargs) == {'four': '8/2', 'five': '2+3'}

    assert worker.result.exception is None
    assert list(worker.result.result[0]) == [instance, 'I', 'II', 'III']
    assert dict(worker.result.result[1]) == {'four': '8/2', 'five': '2+3'}
    assert worker.result.status == workers.TaskStatus.SUCCEEDED


def test_failure_task(qtbot, task_collection):  # pylint: disable=redefined-outer-name
    from anaconda_navigator.utils import workers

    instance = task_collection()

    # unbound call
    with pytest.raises(TypeError):
        task_collection.failure_task(instance)

    # bound call
    with pytest.raises(TypeError):
        instance.failure_task()

    # workers
    worker: workers.TaskWorker
    for worker in [task_collection.failure_task.worker(instance), instance.failure_task.worker()]:
        worker.signals.sig_succeeded.connect(raise_error)
        worker.signals.sig_canceled.connect(raise_error)
        with qtbot.waitSignals(
                [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_failed],
                timeout=2000,
        ):
            worker.start()

        assert list(worker.call.args) == [instance]
        assert not dict(worker.call.kwargs)

        assert isinstance(worker.result.exception, TypeError)
        with pytest.raises(TypeError):
            _ = worker.result.result
        assert worker.result.status == workers.TaskStatus.FAILED


def test_cancellable_task(qtbot, task_collection):  # pylint: disable=redefined-outer-name
    from anaconda_navigator.utils import workers

    instance = task_collection()

    # unbound call
    task_collection.cancellable_task(instance)  # pylint: disable=no-value-for-parameter

    task_collection.cancellable_task(instance, context=workers.CancelContext(canceled=False))

    with pytest.raises(workers.TaskCanceledError):
        task_collection.cancellable_task(instance, context=workers.CancelContext(canceled=True))

    # bound call
    instance.cancellable_task()  # pylint: disable=no-value-for-parameter

    instance.cancellable_task(context=workers.CancelContext(canceled=False))

    with pytest.raises(workers.TaskCanceledError):
        instance.cancellable_task(context=workers.CancelContext(canceled=True))

    # workers
    worker: workers.TaskWorker
    for worker in [
        task_collection.cancellable_task.worker(instance),
        task_collection.cancellable_task.worker(instance, context=workers.CancelContext(canceled=False)),
        instance.cancellable_task.worker(),
        instance.cancellable_task.worker(context=workers.CancelContext(canceled=False)),
    ]:
        worker.signals.sig_failed.connect(raise_error)
        worker.signals.sig_canceled.connect(raise_error)
        with qtbot.waitSignals(
                [worker.signals.sig_start, worker.signals.sig_succeeded, worker.signals.sig_done],
                timeout=3000,
        ):
            worker.start()

        assert worker.result.exception is None
        assert worker.result.result is None
        assert worker.result.status == workers.TaskStatus.SUCCEEDED

    for worker in [
        task_collection.cancellable_task.worker(instance),
        task_collection.cancellable_task.worker(instance, context=workers.CancelContext(canceled=False)),
        instance.cancellable_task.worker(),
        instance.cancellable_task.worker(context=workers.CancelContext(canceled=False)),
    ]:
        worker.signals.sig_succeeded.connect(raise_error)
        worker.signals.sig_failed.connect(raise_error)
        with qtbot.waitSignals(
                [worker.signals.sig_start, worker.signals.sig_canceled, worker.signals.sig_done],
                timeout=3000,
        ):
            worker.start()
            worker.cancel()

        assert isinstance(worker.result.exception, workers.TaskCanceledError)
        with pytest.raises(workers.TaskCanceledError):
            _ = worker.result.result
        assert worker.result.status == workers.TaskStatus.CANCELED

    for worker in [
        task_collection.cancellable_task.worker(instance, context=workers.CancelContext(canceled=True)),
        instance.cancellable_task.worker(context=workers.CancelContext(canceled=True)),
    ]:
        worker.signals.sig_succeeded.connect(raise_error)
        worker.signals.sig_failed.connect(raise_error)
        with qtbot.waitSignals(
                [worker.signals.sig_start, worker.signals.sig_canceled, worker.signals.sig_done],
                timeout=3000,
        ):
            worker.start()

        assert isinstance(worker.result.exception, workers.TaskCanceledError)
        with pytest.raises(workers.TaskCanceledError):
            _ = worker.result.result
        assert worker.result.status == workers.TaskStatus.CANCELED


def test_sleep_task(qtbot, task_collection):  # pylint: disable=redefined-outer-name
    from anaconda_navigator.utils import workers

    instance = task_collection()

    # unbound call
    task_collection.sleep_task(instance)

    # bound call
    instance.sleep_task()

    # workers
    worker: workers.TaskWorker
    for worker in [task_collection.sleep_task.worker(instance), instance.sleep_task.worker()]:
        worker.signals.sig_failed.connect(raise_error)
        worker.signals.sig_canceled.connect(raise_error)
        with qtbot.waitSignals(
                [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_succeeded],
                timeout=3000,
        ):
            worker.start()

        assert worker.result.exception is None
        assert worker.result.result is None
        assert worker.result.status == workers.TaskStatus.SUCCEEDED

    for worker in [task_collection.sleep_task.worker(instance), instance.sleep_task.worker()]:
        worker.signals.sig_succeeded.connect(raise_error)
        worker.signals.sig_failed.connect(raise_error)
        with qtbot.waitSignals(
                [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_canceled],
                timeout=3000,
        ):
            worker.start()
            worker.cancel()

        assert isinstance(worker.result.exception, workers.TaskCanceledError)
        with pytest.raises(workers.TaskCanceledError):
            _ = worker.result.result
        assert worker.result.status == workers.TaskStatus.CANCELED


def test_modified_task(qtbot, task_collection):  # pylint: disable=redefined-outer-name
    from anaconda_navigator.utils import workers

    instance = task_collection()

    # unbound call
    args: typing.Tuple[typing.Any, ...]
    kwargs: typing.Mapping[str, typing.Any]
    args, kwargs = task_collection.modified_task('A', 'b', old_arg=False)
    assert list(args) == ['q', 'A', 'b']
    assert dict(kwargs) == {'old_arg': False, 'new_arg': True}

    # bound call
    args, kwargs = instance.modified_task('a', 'B', old_arg=True)
    assert list(args) == [instance, 'q', 'a', 'B']
    assert dict(kwargs) == {'old_arg': True, 'new_arg': True}

    # unbound worker
    worker: workers.TaskWorker = task_collection.modified_task.worker('x', 'Y', other_arg=17)
    worker.signals.sig_failed.connect(raise_error)
    worker.signals.sig_canceled.connect(raise_error)
    with qtbot.waitSignals(
            [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_succeeded],
            timeout=2000,
    ):
        worker.start()

    assert list(worker.call.args) == ['q', 'x', 'Y']
    assert dict(worker.call.kwargs) == {'other_arg': 17, 'new_arg': True}

    assert worker.result.exception is None
    assert list(worker.result.result[0]) == ['q', 'x', 'Y']
    assert dict(worker.result.result[1]) == {'other_arg': 17, 'new_arg': True}
    assert worker.result.status == workers.TaskStatus.SUCCEEDED

    # bound worker
    worker = instance.modified_task.worker('X', 'y', other=-1)
    worker.signals.sig_failed.connect(raise_error)
    worker.signals.sig_canceled.connect(raise_error)
    with qtbot.waitSignals(
            [worker.signals.sig_start, worker.signals.sig_done, worker.signals.sig_succeeded],
            timeout=2000,
    ):
        worker.start()

    assert list(worker.call.args) == [instance, 'q', 'X', 'y']
    assert dict(worker.call.kwargs) == {'other': -1, 'new_arg': True}

    assert worker.result.exception is None
    assert list(worker.result.result[0]) == [instance, 'q', 'X', 'y']
    assert dict(worker.result.result[1]) == {'other': -1, 'new_arg': True}
    assert worker.result.status == workers.TaskStatus.SUCCEEDED
