# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring,redefined-outer-name

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for channels popup."""

from __future__ import annotations

__all__ = ()

import os
from unittest import mock
import pytest
from qtpy.QtCore import Qt, QTimer
from tests.current.widgets import common_extra


@pytest.fixture(scope='function')
def botchannels(qtbot, qt_styles, temporary_file):  # pylint: disable=unused-argument
    from anaconda_navigator.widgets.dialogs import channels as dialogs_channels

    widget = dialogs_channels.DialogChannels()
    qtbot.addWidget(widget)

    validator_patcher = mock.patch.object(widget.api._download_api, '_is_valid_url')  # pylint: disable=protected-access
    validator_patcher.start()

    temp_rc_content = 'channels:\n  - chdoig\n  - sean'
    with open(temporary_file, 'w', encoding='utf-8') as stream:
        stream.write(temp_rc_content)

    # Mock conda config data
    config_data = {'config_sources': {temporary_file: {'channels': ['chdoig', 'sean']}}}
    with qtbot.waitSignal(signal=widget.sig_setup_ready, timeout=5000, raising=True):
        widget.setup(None, config_data, None)

    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


def test_init(botchannels):
    assert botchannels.widget.list.count() == 2


def test_add_defaults(botchannels):
    botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
    item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
    botchannels.qtbot.keyClicks(item.text_channel, 'defaults')
    with botchannels.qtbot.waitSignal(signal=botchannels.widget.sig_check_ready, timeout=5000, raising=True):
        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
        botchannels.qtbot.keyClick(item.text_channel, Qt.Key_Enter)
    item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
    botchannels.qtbot.mouseClick(botchannels.widget.button_ok, Qt.LeftButton)

    assert item.text_channel.text() == 'defaults'


def test_add_valid_channels(botchannels, temporary_folder):
    from anaconda_navigator import config as navigator_config

    if navigator_config.WIN:
        file_channel = 'file:///' + temporary_folder.replace('\\', '/')
    else:
        file_channel = 'file://' + temporary_folder

    channels = {
        'goanpeca': 'goanpeca',
        'https://anaconda.org/sean': 'https://conda.anaconda.org/sean',
        'https://conda.anaconda.org/chdoig': 'https://conda.anaconda.org/chdoig',
        file_channel: file_channel,
        'https://conda.anaconda.org/goanpeca': 'https://conda.anaconda.org/goanpeca',
    }
    botchannels.widget.api._download_api._is_valid_url.return_value = True  # pylint: disable=protected-access

    for channel, channel_text in channels.items():
        botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
        item.text_channel.clear()
        botchannels.qtbot.keyClicks(item.text_channel, channel)

        with botchannels.qtbot.waitSignal(signal=botchannels.widget.sig_check_ready, timeout=20000, raising=True):
            count = botchannels.widget.list.count()
            item = botchannels.widget.list.item(count - 1)
            botchannels.widget.list.validate_channel(item)

        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)

        assert item.text_channel.text() == channel_text

    assert botchannels.widget.list.count() == len(channels) + 2


def test_add_invalid_channel(botchannels):
    botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
    item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
    botchannels.qtbot.keyClicks(item.text_channel, 'whatever-invalid-channel')
    botchannels.widget.api._download_api._is_valid_url.return_value = False  # pylint: disable=protected-access

    with botchannels.qtbot.waitSignal(signal=botchannels.widget.sig_check_ready, timeout=20000, raising=True):
        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
        botchannels.qtbot.keyClick(item.text_channel, Qt.Key_Enter)
    count = botchannels.widget.list.count()
    item = botchannels.widget.list.item(count - 1)
    assert item.label_info.isVisible()
    assert not botchannels.widget.button_ok.isEnabled()
    assert botchannels.widget.button_cancel.isEnabled()
    assert not botchannels.widget.button_add.isEnabled()


def test_add_empty_channel(botchannels):
    botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
    with botchannels.qtbot.waitSignal(signal=botchannels.widget.sig_check_ready, timeout=20000, raising=True):
        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
        botchannels.qtbot.keyClick(item.text_channel, Qt.Key_Enter)
    count = botchannels.widget.list.count()
    item = botchannels.widget.list.item(count - 1)
    assert item.label_info.isVisible()
    assert not botchannels.widget.button_ok.isEnabled()
    assert botchannels.widget.button_cancel.isEnabled()
    assert not botchannels.widget.button_add.isEnabled()


def test_add_duplicate_channel(botchannels):
    for channel in ['chdoig', 'chdoig']:
        botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
        botchannels.qtbot.keyClicks(item.text_channel, channel)
        with botchannels.qtbot.waitSignal(signal=botchannels.widget.sig_check_ready, timeout=20000, raising=True):
            item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
            botchannels.qtbot.keyClick(item.text_channel, Qt.Key_Enter)
    count = botchannels.widget.list.count()
    item = botchannels.widget.list.item(count - 1)
    assert item.label_info.isVisible()
    assert not botchannels.widget.button_ok.isEnabled()
    assert botchannels.widget.button_cancel.isEnabled()
    assert not botchannels.widget.button_add.isEnabled()


@pytest.mark.skipif(os.name == 'nt', reason='does not work on windows')
def test_remove_button_focus_signal(botchannels):
    assert botchannels.widget.list.currentRow() == -1
    with botchannels.qtbot.waitSignal(signal=botchannels.widget.list.sig_focus_fixed, timeout=20000, raising=True):
        botchannels.qtbot.keyClick(botchannels.widget, Qt.Key_Tab)
    assert botchannels.widget.list.currentRow() == 0

    with botchannels.qtbot.waitSignal(signal=botchannels.widget.list.sig_focus_fixed, timeout=20000, raising=True):
        botchannels.qtbot.keyClick(botchannels.widget, Qt.Key_Tab)
    assert botchannels.widget.list.currentRow() == 1


def test_escape_pressed(botchannels):
    botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
    item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
    botchannels.qtbot.keyClicks(item.text_channel, 'invalid-channel-name')
    botchannels.widget.api._download_api._is_valid_url.return_value = False  # pylint: disable=protected-access

    with botchannels.qtbot.waitSignal(signal=botchannels.widget.sig_check_ready, timeout=20000, raising=True):
        item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
        botchannels.qtbot.keyClick(item.text_channel, Qt.Key_Enter)
        count = botchannels.widget.list.count()
        item = botchannels.widget.list.item(count - 1)
        assert count == 3
        assert item.label_info.isVisible()

    item = botchannels.widget.list.item(botchannels.widget.list.count() - 1)
    botchannels.qtbot.keyClick(item.text_channel, Qt.Key_Escape)
    assert botchannels.widget.list.count() == 2


def test_menu_copy(botchannels):
    from qtpy.QtCore import QCoreApplication  # pylint: disable=import-outside-toplevel
    app = QCoreApplication.instance()
    item = botchannels.widget.list.item(0)

    def _triggered():
        with botchannels.qtbot.waitSignal(signal=item.text_channel.sig_copied, timeout=2000, raising=True):
            botchannels.qtbot.keyClick(item.text_channel.menu, Qt.Key_Down)
            botchannels.qtbot.keyClick(item.text_channel.menu, Qt.Key_Return)
        assert app.clipboard().text() == 'chdoig'

    timer = QTimer()
    timer.timeout.connect(_triggered)
    timer.setInterval(2000)
    timer.start()
    botchannels.qtbot.mouseClick(item.text_channel, Qt.RightButton)


def test_paste_channel(botchannels):
    from qtpy.QtCore import QCoreApplication  # pylint: disable=import-outside-toplevel
    app = QCoreApplication.instance()
    botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
    app.clipboard().setText('Metropolis')
    count = botchannels.widget.list.count()
    item = botchannels.widget.list.item(count - 1)
    botchannels.qtbot.keyClick(item.text_channel, Qt.Key_V, modifier=Qt.ControlModifier)
    assert item.text_channel.text() == 'Metropolis'


def test_edit_buttons_disabled(botchannels):
    botchannels.qtbot.mouseClick(botchannels.widget.button_add, Qt.LeftButton)
    assert not botchannels.widget.button_ok.isEnabled()
    assert botchannels.widget.button_cancel.isEnabled()


def test_empty_list(botchannels):
    assert botchannels.widget.button_ok.isEnabled()
    item = botchannels.widget.list.item(0)
    botchannels.qtbot.mouseClick(item.button_remove, Qt.LeftButton)
    assert botchannels.widget.list.count() == 1

    item = botchannels.widget.list.item(0)
    botchannels.qtbot.mouseClick(item.button_remove, Qt.LeftButton)
    assert botchannels.widget.list.count() == 0
    assert not botchannels.widget.button_ok.isEnabled()


def test_update_style_sheet(botchannels):
    botchannels.widget.update_style_sheet()


def test_reject(botchannels):
    with botchannels.qtbot.waitSignal(signal=botchannels.widget.rejected, timeout=2000, raising=True):
        botchannels.widget.update_channels()


def test_reject_2(botchannels):
    with botchannels.qtbot.waitSignal(signal=botchannels.widget.rejected, timeout=2000, raising=True):
        botchannels.qtbot.keyClick(botchannels.widget, Qt.Key_Escape)
