# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for environment-related dialogs."""

from __future__ import annotations

__all__ = ()

import contextlib
import typing
from unittest import mock
import pytest
from qtpy.QtCore import QPoint, Qt
from tests.current.widgets import common_extra


BASE_ENVIRONMENTS = {
    '/usr/anaconda': 'root',
    '/usr/anaconda/envs/test1': 'test1',
    '/usr/anaconda/envs/test2': 'test2',
}
BASE_INFO = {
    '__environments': BASE_ENVIRONMENTS,
    '__envs_dirs_writable': ['/usr/anaconda/envs/'],
}

BASE_DATA = {
    'info': BASE_INFO,
    'processed_info': BASE_INFO,
    'packages': {
        'python': {
            'versions': ['2.7', '3.6', '3.9', '3.7', '3.8', '3.5'],
        },
        'r-base': {
            'versions': ['3.1.2', '3.1.3', '3.2.0', '3.2.1', '3.2.2', '3.3.1', '3.3.2', '3.4.1', '3.4.2', '3.4.3'],
        },
    }
}


class MockVersionInfo:  # pylint: disable=too-few-public-methods
    """Mock of the sys.version_infomodule to get fake higher python version."""

    major = '3'
    minor = '7'


@contextlib.contextmanager
def mocked_compat(
        path: str,
        selected_filter: str,
        target: str = 'anaconda_navigator.widgets.dialogs.environment.import_dialogs.compat',
) -> typing.Iterator[None]:
    """Mock compat library """
    new_compat = mock.Mock()
    new_compat.getopenfilename.return_value = path, selected_filter

    with mock.patch(target, new_compat):
        yield


@contextlib.contextmanager
def mocked_sys(
        target: str = 'anaconda_navigator.widgets.dialogs.environment.create_dialogs.sys',
) -> typing.Iterator[None]:
    """Mock sys library."""
    new_sys = mock.Mock()
    new_sys.version_info = MockVersionInfo

    with mock.patch(target, new_sys):
        yield


def widget_pos(widget):
    return QPoint(2, widget.height() // 2)


@pytest.fixture(scope='function')
def botremove(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs import environment as environment_dialogs

    widget = environment_dialogs.RemoveDialog(name='test', prefix='/usr/anaconda')
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botcreate(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs import environment as environment_dialogs

    with mocked_sys():
        widget = environment_dialogs.CreateDialog()
        qtbot.addWidget(widget)

        with qtbot.waitSignals(
            [package_group.sig_refreshed for package_group in widget.package_groups.values()],
            timeout=1200_000,
        ):
            widget.setup(None, BASE_DATA, None)
            widget.show()

        yield common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botclone(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs import environment as environment_dialogs

    widget = environment_dialogs.CloneDialog(parent=None, clone_from_name='test1')
    qtbot.addWidget(widget)
    widget.setup(None, BASE_DATA, None)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botconflict(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs import environment as environment_dialogs

    widget = environment_dialogs.ConflictDialog(parent=None, package='conflict-package')
    qtbot.addWidget(widget)
    widget.setup(None, BASE_DATA, None)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


@pytest.fixture(scope='function')
def botbase(qtbot, qt_styles):  # pylint: disable=unused-argument
    """Return bot and widget."""
    from anaconda_navigator.widgets.dialogs.environment import common as common_dialogs

    widget = common_dialogs.EnvironmentActionsDialog(parent=None)
    qtbot.addWidget(widget)
    widget.show()
    return common_extra.BotContext(qtbot=qtbot, widget=widget)


def test_create_environment_dialog_refresh(botcreate):  # pylint: disable=redefined-outer-name
    assert not botcreate.widget.button_ok.isEnabled()

    botcreate.qtbot.keyClicks(botcreate.widget.text_name, 'some-env-name')
    assert botcreate.widget.button_ok.isEnabled()

    botcreate.widget.text_name.setText('')
    botcreate.qtbot.keyClicks(botcreate.widget.text_name, 'test1')
    assert not botcreate.widget.button_ok.isEnabled()


def test_create_environment_dialog_packages(botcreate):  # pylint: disable=redefined-outer-name
    botcreate.qtbot.keyClicks(botcreate.widget.text_name, 'some-env-name')
    assert botcreate.widget.button_ok.isEnabled()
    assert botcreate.widget.package_groups['python'].checked
    assert not botcreate.widget.package_groups['r'].checked
    assert {item.split('=', 1)[0] for item in botcreate.widget.packages} == {'python'}

    botcreate.qtbot.mouseClick(
        botcreate.widget.package_groups['python'].checkbox,
        Qt.LeftButton,
        pos=widget_pos(botcreate.widget.package_groups['python'].checkbox),
    )
    assert not botcreate.widget.button_ok.isEnabled()
    assert not botcreate.widget.package_groups['python'].checked
    assert not botcreate.widget.package_groups['r'].checked
    assert {item.split('=', 1)[0] for item in botcreate.widget.packages} == set()

    botcreate.qtbot.mouseClick(
        botcreate.widget.package_groups['r'].checkbox,
        Qt.LeftButton,
        pos=widget_pos(botcreate.widget.package_groups['r'].checkbox),
    )
    assert botcreate.widget.button_ok.isEnabled()
    assert not botcreate.widget.package_groups['python'].checked
    assert botcreate.widget.package_groups['r'].checked
    assert {item.split('=', 1)[0] for item in botcreate.widget.packages} == {'r-base', 'r-essentials'}

    botcreate.qtbot.mouseClick(
        botcreate.widget.package_groups['python'].checkbox,
        Qt.LeftButton,
        pos=widget_pos(botcreate.widget.package_groups['python'].checkbox),
    )
    assert botcreate.widget.button_ok.isEnabled()
    assert botcreate.widget.package_groups['python'].checked
    assert botcreate.widget.package_groups['r'].checked
    assert {item.split('=', 1)[0] for item in botcreate.widget.packages} == {'python', 'r-base', 'r-essentials'}


def test_create_environment_dialog_create(botcreate):  # pylint: disable=redefined-outer-name
    botcreate.qtbot.keyClicks(botcreate.widget.text_name, 'some-env-name')
    with botcreate.qtbot.waitSignal(botcreate.widget.accepted, timeout=1000, raising=True):
        botcreate.qtbot.mouseClick(botcreate.widget.button_ok, Qt.LeftButton)


def test_create_environment_dialog_cancel(botcreate):  # pylint: disable=redefined-outer-name
    with botcreate.qtbot.waitSignal(botcreate.widget.rejected, timeout=1000, raising=True):
        botcreate.qtbot.mouseClick(botcreate.widget.button_cancel, Qt.LeftButton)


def test_clone_environment_dialog_name(botclone):  # pylint: disable=redefined-outer-name
    # Check repeated env name is disallowed
    for env_prefix, env_name in BASE_ENVIRONMENTS.items():  # pylint: disable=unused-variable
        botclone.widget.text_name.setText('')
        botclone.qtbot.keyClicks(botclone.widget.text_name, env_name)
        assert not botclone.widget.button_ok.isEnabled()

    botclone.widget.text_name.setText('')
    botclone.qtbot.keyClicks(botclone.widget.text_name, 'validenvname')
    assert botclone.widget.button_ok.isEnabled()


def test_clone_environment_dialog_clone(botclone):  # pylint: disable=redefined-outer-name
    botclone.qtbot.keyClicks(botclone.widget.text_name, 'validenvname')

    with botclone.qtbot.waitSignal(botclone.widget.accepted, timeout=1000, raising=True):
        botclone.qtbot.mouseClick(botclone.widget.button_ok, Qt.LeftButton)


def test_clone_environment_dialog_cancel(botclone):  # pylint: disable=redefined-outer-name
    with botclone.qtbot.waitSignal(botclone.widget.rejected, timeout=1000, raising=True):
        botclone.qtbot.mouseClick(botclone.widget.button_cancel, Qt.LeftButton)


def test_remove_environment_dialog_remove(botremove):  # pylint: disable=redefined-outer-name
    with botremove.qtbot.waitSignal(botremove.widget.accepted, timeout=1000, raising=True):
        botremove.qtbot.mouseClick(botremove.widget.button_ok, Qt.LeftButton)


def test_remove_environment_dialog_cancel(botremove):  # pylint: disable=redefined-outer-name
    with botremove.qtbot.waitSignal(botremove.widget.rejected, timeout=1000, raising=True):
        botremove.qtbot.mouseClick(botremove.widget.button_cancel, Qt.LeftButton)


def test_environment_conflict_name(botconflict):  # pylint: disable=redefined-outer-name
    assert botconflict.widget.name == 'conflict-package'


def test_environment_dialog_base_refresh(botbase):  # pylint: disable=redefined-outer-name
    with pytest.raises(NotImplementedError):
        botbase.widget.refresh()


def test_environment_dialog_base_prefix(botbase):  # pylint: disable=redefined-outer-name
    assert not botbase.widget.prefix
    assert not botbase.widget.name
