# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for preferences dialog."""

from __future__ import annotations

__all__ = ()

from unittest import mock
import pytest
from flaky import flaky
from qtpy.QtCore import Qt
from tests.current.widgets import common_extra

CHECK_OPTIONS = [
    'provide_analytics', 'hide_quit_dialog', 'hide_update_dialog', 'hide_running_apps_dialog',
    'enable_high_dpi_scaling',
]
ALL_OPTIONS = [*CHECK_OPTIONS, 'anaconda_api_url']


@pytest.fixture(scope='function')
def main_section():
    from anaconda_navigator.config import main as config_main

    return config_main.DEFAULTS[0][-1]


@pytest.fixture(scope='function')
def botpreferences(qtbot, qt_styles):  # pylint: disable=unused-argument
    from anaconda_navigator.widgets.dialogs import preferences as dialogs_preferences
    from anaconda_navigator.api import external_apps
    from anaconda_navigator import config as navigator_config

    external_apps.get_applications(navigator_config.CONF, mock.Mock())

    widget = dialogs_preferences.PreferencesDialog(environments={})
    qtbot.addWidget(widget)
    widget.show()

    return common_extra.BotContext(qtbot=qtbot, widget=widget)


def test_simple(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    assert botpreferences.widget.widgets
    assert not botpreferences.widget.button_ok.isEnabled()  # no change yet
    assert botpreferences.widget.button_reset.isEnabled()
    assert botpreferences.widget.button_cancel.isEnabled()

    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            assert widget.get_value() == default_value, f'unexpected value for {option!r}'


def test_options_accept(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            widget.set_value(not default_value)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.accepted, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)

    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            assert widget.get_value() is not default_value
            assert botpreferences.widget.get_option(option) is not default_value
    assert not botpreferences.widget.isVisible()


def test_options_cancel(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            widget.set_value(not default_value)
            print(widget.option, not default_value)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.rejected, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_cancel, Qt.LeftButton)

    print('\n')
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            print(widget.option, botpreferences.widget.get_option(option))
            assert botpreferences.widget.get_option(option) == default_value
    assert not botpreferences.widget.isVisible()


def test_get_defaults(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            assert default_value == botpreferences.widget.get_option_default(option)


def test_set_defaults(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            widget.set_value(not default_value)
            botpreferences.widget.set_option_default(option)
            assert botpreferences.widget.get_option(option) == default_value


def test_options_changed_back(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            botpreferences.widget.widgets_changed.add(widget)
            botpreferences.widget.options_changed(value=default_value, widget=widget)


@pytest.mark.xfail
def test_api_valid(botpreferences):  # pylint: disable=redefined-outer-name
    botpreferences.qtbot.keyClicks(botpreferences.widget.widgets[0], 'https://api.beta.anaconda.org')
    with botpreferences.qtbot.waitSignal(botpreferences.widget.accepted, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert not botpreferences.widget.isVisible()


def test_api_invalid(botpreferences):  # pylint: disable=redefined-outer-name
    botpreferences.qtbot.keyClicks(botpreferences.widget.widgets[0], '*&**&')
    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_check_ready, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert botpreferences.widget.widgets[0].label_information.toolTip()
    assert botpreferences.widget.button_ok.isEnabled() is False
    assert botpreferences.widget.isVisible()


@pytest.mark.xfail
def test_api_ssl(botpreferences):  # pylint: disable=redefined-outer-name
    botpreferences.qtbot.keyClicks(botpreferences.widget.widgets[0], 'https://api.beta.anaconda.org')
    widget = botpreferences.widget.widget_for_option('ssl_verification')
    widget.set_value(False)
    widget.check_value(False)
    botpreferences.widget.widgets_changed.add(widget)
    with botpreferences.qtbot.waitSignal(botpreferences.widget.accepted, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert not botpreferences.widget.isVisible()


def test_api_close(botpreferences):  # pylint: disable=redefined-outer-name
    botpreferences.qtbot.keyClicks(botpreferences.widget.widgets[0], 'https://beta.anaconda.org')  # no .api
    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_check_ready, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert botpreferences.widget.widgets[0].label_information.toolTip()
    assert botpreferences.widget.button_ok.isEnabled() is False
    assert botpreferences.widget.isVisible()


def test_api_close_2(botpreferences):  # pylint: disable=redefined-outer-name
    botpreferences.qtbot.keyClicks(botpreferences.widget.widgets[0], 'https://beta.anaconda.org/')  # no .api
    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_check_ready, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert botpreferences.widget.button_ok.isEnabled() is False
    assert botpreferences.widget.isVisible()


def test_options_reset_cancel(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            widget.set_value(not default_value)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_check_ready, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_reset_ready, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_reset, Qt.LeftButton)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.rejected, timeout=5000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_cancel, Qt.LeftButton)

    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            assert botpreferences.widget.get_option(option) != default_value

    assert not botpreferences.widget.isVisible()


@flaky(max_runs=2, min_passes=1)
def test_options_reset_accept(botpreferences, main_section):  # pylint: disable=redefined-outer-name
    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            widget = botpreferences.widget.widget_for_option(option)
            widget.set_value(not default_value)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_check_ready, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_reset_ready, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_reset, Qt.LeftButton)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.accepted, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)

    for option, default_value in main_section.items():
        if option in CHECK_OPTIONS:
            assert botpreferences.widget.get_option(option) == default_value

    assert not botpreferences.widget.isVisible()


def test_ssl_options_true(botpreferences):  # pylint: disable=redefined-outer-name
    assert botpreferences.widget.widget_for_option('ssl_verification').isChecked()
    assert botpreferences.widget.widget_for_option('ssl_certificate').isEnabled()


def test_ssl_options_false(botpreferences):  # pylint: disable=redefined-outer-name
    ssl_verify = botpreferences.widget.widget_for_option('ssl_verification')
    ssl_verify.set_value(False)
    assert not botpreferences.widget.widget_for_option('ssl_verification').isChecked()
    assert not botpreferences.widget.widget_for_option('ssl_certificate').isEnabled()


def test_ssl_options_path_invalid(botpreferences):  # pylint: disable=redefined-outer-name
    ssl_cert = botpreferences.widget.widget_for_option('ssl_certificate')
    ssl_cert.set_value('rubishpath/')

    with botpreferences.qtbot.waitSignal(botpreferences.widget.sig_check_ready, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert not botpreferences.widget.button_ok.isEnabled()


def test_ssl_options_path_valid(botpreferences, temporary_file):  # pylint: disable=redefined-outer-name
    ssl_cert = botpreferences.widget.widget_for_option('ssl_certificate')
    ssl_cert.set_value(temporary_file)

    with botpreferences.qtbot.waitSignal(botpreferences.widget.accepted, timeout=10000, raising=True):
        botpreferences.qtbot.mouseClick(botpreferences.widget.button_ok, Qt.LeftButton)
    assert botpreferences.widget.button_ok.isEnabled()
