# -*- coding: utf-8 -*-

"""Shortcuts to common Qt testing utilities."""

from __future__ import annotations

__all__ = ['QtHelper']

import typing

from tests.utilities import functional_utils
from . import qwidget_collection
from . import wait


class QtHelper:
    """Helper class with shortcuts to primary Qt testing utilities."""

    __slots__ = ()

    @staticmethod
    @typing.overload
    def find_many(arg1: 'qwidget_collection.Source') -> qwidget_collection.QWidgetCollection:
        """Search for Qt widgets in the application."""

    @staticmethod
    @typing.overload
    def find_many(arg1: 'qwidget_collection.Predicate', **kwargs: typing.Any) -> qwidget_collection.QWidgetCollection:
        """Search for Qt widgets in the application."""

    @staticmethod
    @typing.overload
    def find_many(
            arg1: 'qwidget_collection.Source',
            arg2: 'qwidget_collection.Predicate',
            **kwargs: typing.Any,
    ) -> qwidget_collection.QWidgetCollection:
        """Search for Qt widgets in the application."""

    @staticmethod
    def find_many(
            arg1: typing.Union['qwidget_collection.Source', 'qwidget_collection.Predicate'],
            arg2: typing.Optional['qwidget_collection.Predicate'] = None,
            **kwargs: typing.Any,
    ) -> qwidget_collection.QWidgetCollection:
        """
        Search for Qt widgets in the application.

        May take:

        - Instance of a Qt widget or :class:`~QWidgetCollection`. In this case it should just return the instance.

        - Instance of a Qt widget or :class:`~QWidgetCollection` and a predicate. This would look for children of
          instance which passes the predicate check.

        - Just a predicate. This would look for all widgets in an application that pass the predicate check.

        Additionally, this function may also take additional parameters to use with :meth:`~QWidgetCollection.find`:

        :param min_depth: Include widgets that are located on at least particular depth.
        :param max_depth: Don't look further than particular depth.
        :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.
        """
        return qwidget_collection.search_for(
            arg1,
            arg2,  # type: ignore
            **kwargs,
        ).value

    @classmethod
    @typing.overload
    def find_one(cls, arg1: 'qwidget_collection.Source') -> 'qwidget_collection.Item':
        """Search for a single Qt widget in the application."""

    @classmethod
    @typing.overload
    def find_one(cls, arg1: 'qwidget_collection.Predicate', **kwargs: typing.Any) -> 'qwidget_collection.Item':
        """Search for a single Qt widget in the application."""

    @classmethod
    @typing.overload
    def find_one(
            cls,
            arg1: 'qwidget_collection.Source',
            arg2: 'qwidget_collection.Predicate',
            **kwargs: typing.Any,
    ) -> 'qwidget_collection.Item':
        """Search for a single Qt widget in the application."""

    @classmethod
    def find_one(
            cls,
            arg1: typing.Union['qwidget_collection.Source', 'qwidget_collection.Predicate'],
            arg2: typing.Optional['qwidget_collection.Predicate'] = None,
            **kwargs: typing.Any,
    ) -> 'qwidget_collection.Item':
        """
        Search for a single Qt widget in the application.

        If there would be no or more than a one widget - method would raise an exception.

        May take:

        - Instance of a Qt widget or :class:`~QWidgetCollection`. In this case it should just return the instance.

        - Instance of a Qt widget or :class:`~QWidgetCollection` and a predicate. This would look for children of
          instance which passes the predicate check.

        - Just a predicate. This would look for all widgets in an application that pass the predicate check.

        Additionally, this function may also take additional parameters to use with :meth:`~QWidgetCollection.find`:

        :param min_depth: Include widgets that are located on at least particular depth.
        :param max_depth: Don't look further than particular depth.
        :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.
        """
        return cls.find_many(
            arg1,
            arg2,  # type: ignore
            **kwargs,
        ).single

    @classmethod
    def sleep(cls, interval: int = 500) -> None:
        """Wait for some time (in msec)."""
        wait.Wait().for_timeout(interval=interval).exec_()

    @staticmethod
    def wait() -> 'wait.Wait[typing.Any]':
        """Create a new :class:`~tests.utilities.qt_utils.wait.Wait` instance for further chaining."""
        return wait.Wait()

    @classmethod
    @typing.overload
    def wait_for_many(cls, arg1: 'qwidget_collection.Source') -> qwidget_collection.QWidgetCollection:
        """Wait before multiple widgets of expected state become available."""

    @classmethod
    @typing.overload
    def wait_for_many(
            cls,
            arg1: 'qwidget_collection.Predicate',
            **kwargs: typing.Any,
    ) -> qwidget_collection.QWidgetCollection:
        """Wait before multiple widgets of expected state become available."""

    @classmethod
    @typing.overload
    def wait_for_many(
            cls,
            arg1: 'qwidget_collection.Source',
            arg2: 'qwidget_collection.Predicate',
            **kwargs: typing.Any,
    ) -> qwidget_collection.QWidgetCollection:
        """Wait before multiple widgets of expected state become available."""

    @classmethod
    def wait_for_many(
            cls,
            arg1: typing.Union['qwidget_collection.Source', 'qwidget_collection.Predicate'],
            arg2: typing.Optional['qwidget_collection.Predicate'] = None,
            *,
            at_least: int = 1,
            at_most: int = 2 ** 63,
            **kwargs: typing.Any,
    ) -> qwidget_collection.QWidgetCollection:
        """
        Wait for multiple widgets with expected description.

        May take:

        - Instance of a Qt widget or :class:`~QWidgetCollection`. In this case it should just return the instance.

        - Instance of a Qt widget or :class:`~QWidgetCollection` and a predicate. This would look for children of
          instance which passes the predicate check.

        - Just a predicate. This would look for all widgets in an application that pass the predicate check.

        To specify an exact amount of awaited widgets, you may set:

        :param at_least: Lower bound of expected widgets count (inclusive).
        :param at_most: Upper bound of expected widgets count (inclusive).

        Additionally, this function may also take additional parameters to use with :meth:`~QWidgetCollection.find`:

        :param min_depth: Include widgets that are located on at least particular depth.
        :param max_depth: Don't look further than particular depth.
        :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.

        And you can also modify the polling behavior of the waiting process:

        :param wait_before: additional pause (in msec) before checking a condition for the first time.
        :param wait_after: additional pause (in msec) after condition is met. It expects a condition to still return a
                           result during all this waiting period. If condition breaks during this waiting period
                           (returns :code:`None`) - this would restore the waiting for the condition.
        :param poll_interval: interval between condition checks.
        """
        at_least = min(at_least, at_most)

        def condition() -> typing.Optional[qwidget_collection.QWidgetCollection]:
            output: qwidget_collection.QWidgetCollection = cls.find_many(
                arg1,
                arg2,  # type: ignore
                **kwargs,
            )
            if at_least <= len(output) <= at_most:
                return output
            return None

        waiter: wait.Wait[qwidget_collection.QWidgetCollection] = cls.wait()
        waiter_kwargs: typing.Mapping[str, typing.Any] = functional_utils.extract_kwargs(kwargs, waiter.for_condition)
        waiter = waiter.for_condition(
            condition=condition,
            identity='find_many',
            raise_exception=True,
            **waiter_kwargs,
        )

        result: qwidget_collection.QWidgetCollection
        _, result = waiter.exec_()

        return result

    @classmethod
    @typing.overload
    def wait_for_none(cls, arg1: 'qwidget_collection.Source') -> None:
        """Wait for multiple widgets to stop meeting expected description."""

    @classmethod
    @typing.overload
    def wait_for_none(cls, arg1: 'qwidget_collection.Predicate', **kwargs: typing.Any) -> None:
        """Wait for multiple widgets to stop meeting expected description."""

    @classmethod
    @typing.overload
    def wait_for_none(
            cls,
            arg1: 'qwidget_collection.Source',
            arg2: 'qwidget_collection.Predicate',
            **kwargs: typing.Any,
    ) -> None:
        """Wait for multiple widgets to stop meeting expected description."""

    @classmethod
    def wait_for_none(
            cls,
            arg1: typing.Union['qwidget_collection.Source', 'qwidget_collection.Predicate'],
            arg2: typing.Optional['qwidget_collection.Predicate'] = None,
            **kwargs: typing.Any,
    ) -> None:
        """
        Wait for multiple widgets to stop meeting expected description.

        May take:

        - Instance of a Qt widget or :class:`~QWidgetCollection`. In this case it should just return the instance.

        - Instance of a Qt widget or :class:`~QWidgetCollection` and a predicate. This would look for children of
          instance which passes the predicate check.

        - Just a predicate. This would look for all widgets in an application that pass the predicate check.

        Additionally, this function may also take additional parameters to use with :meth:`~QWidgetCollection.find`:

        :param min_depth: Include widgets that are located on at least particular depth.
        :param max_depth: Don't look further than particular depth.
        :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.

        And you can also modify the polling behavior of the waiting process:

        :param wait_before: additional pause (in msec) before checking a condition for the first time.
        :param wait_after: additional pause (in msec) after condition is met. It expects a condition to still return a
                           result during all this waiting period. If condition breaks during this waiting period
                           (returns :code:`None`) - this would restore the waiting for the condition.
        :param poll_interval: interval between condition checks.
        """
        cls.wait_for_many(
            arg1,
            arg2,  # type: ignore
            at_least=0,
            at_most=0,
            **kwargs,
        )

    @classmethod
    @typing.overload
    def wait_for_one(cls, arg1: 'qwidget_collection.Source') -> 'qwidget_collection.Item':
        """Wait for a single widget with expected description."""

    @classmethod
    @typing.overload
    def wait_for_one(cls, arg1: 'qwidget_collection.Predicate', **kwargs: typing.Any) -> 'qwidget_collection.Item':
        """Wait for a single widget with expected description."""

    @classmethod
    @typing.overload
    def wait_for_one(
            cls,
            arg1: 'qwidget_collection.Source',
            arg2: 'qwidget_collection.Predicate',
            **kwargs: typing.Any,
    ) -> 'qwidget_collection.Item':
        """Wait for a single widget with expected description."""

    @classmethod
    def wait_for_one(
            cls,
            arg1: typing.Union['qwidget_collection.Source', 'qwidget_collection.Predicate'],
            arg2: typing.Optional['qwidget_collection.Predicate'] = None,
            **kwargs: typing.Any,
    ) -> 'qwidget_collection.Item':
        """
        Wait for a single widget with expected description.

        If multiple widgets with the same description are available - this method will raise an error. This should be
        useful when you are sure that there must be only one widget with particular description.

        If you want to wait for at least one widget to appear or only one widget left, consider using
        :meth:`~QtHelper.wait_for_many` with corresponding `at_least` and `at_most` argument values.

        May take:

        - Instance of a Qt widget or :class:`~QWidgetCollection`. In this case it should just return the instance.

        - Instance of a Qt widget or :class:`~QWidgetCollection` and a predicate. This would look for children of
          instance which passes the predicate check.

        - Just a predicate. This would look for all widgets in an application that pass the predicate check.

        Additionally, this function may also take additional parameters to use with :meth:`~QWidgetCollection.find`:

        :param min_depth: Include widgets that are located on at least particular depth.
        :param max_depth: Don't look further than particular depth.
        :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.

        And you can also modify the polling behavior of the waiting process:

        :param wait_before: additional pause (in msec) before checking a condition for the first time.
        :param wait_after: additional pause (in msec) after condition is met. It expects a condition to still return a
                           result during all this waiting period. If condition breaks during this waiting period
                           (returns :code:`None`) - this would restore the waiting for the condition.
        :param poll_interval: interval between condition checks.
        """
        return cls.wait_for_many(
            arg1,
            arg2,  # type: ignore
            **kwargs,
        ).single
