# -*- coding: utf-8 -*-

"""Collection to operate on widgets."""

from __future__ import annotations

__all__ = ['QWidgetCollection']

import typing

import attr
from qtpy import QtWidgets

from tests.utilities import iteration_utils

Item = typing.Union[QtWidgets.QLayout, QtWidgets.QWidget]

if typing.TYPE_CHECKING:
    from .predicates import core as predicate_core

    Predicate = predicate_core.Predicate[Item]
    Source = typing.Union[Item, typing.Iterable[Item]]


def flatten(args: typing.Tuple[typing.Union[Item, typing.Iterable[Item]], ...]) -> typing.Iterator[Item]:
    """Iterate through items provided to the :class:`~QWidgetCollection` constructor."""
    arg: typing.Union[Item, typing.Iterable[Item]]
    for arg in args:
        if isinstance(arg, Item.__args__):  # type: ignore
            yield arg
        else:
            yield from arg


def get_children_for_main_window(item: QtWidgets.QMainWindow) -> typing.Iterator[Item]:
    """Iterate through children of QMainWindow `item`."""
    central_widget: typing.Optional[QtWidgets.QWidget] = item.centralWidget()
    if central_widget is not None:
        yield central_widget


def get_children_for_widget(item: QtWidgets.QWidget) -> typing.Iterator[Item]:
    """Iterate through children of QWidget `item`."""
    layout: typing.Optional[QtWidgets.QLayout] = item.layout()
    if layout is not None:
        yield layout


def get_children_for_layout(item: QtWidgets.QLayout) -> typing.Iterator[Item]:
    """Iterate through children of QLayout `item`."""
    index: int
    for index in range(item.count()):
        child: typing.Optional[QtWidgets.QLayoutItem] = item.itemAt(index)
        if child is None:
            continue

        widget: typing.Optional[QtWidgets.QWidget] = child.widget()
        if widget is not None:
            yield widget
            continue

        layout: typing.Optional[QtWidgets.QLayout] = child.layout()
        if layout is not None:
            yield layout

        # spacer children are ignored


def get_children(item: Item) -> typing.Iterator[Item]:
    """Iterate through children of a :class:`~QWidgetCollection` item."""
    if isinstance(item, QtWidgets.QMainWindow):
        return get_children_for_main_window(item)

    if isinstance(item, QtWidgets.QWidget):
        return get_children_for_widget(item)

    if isinstance(item, QtWidgets.QLayout):
        return get_children_for_layout(item)

    raise TypeError(f'Could not get children for {type(item).__name__} item')


# Widget collection

class QWidgetCollection(typing.Sequence[Item]):
    """
    Collection of Qt widgets.

    This collection contains extra interfaces that should help to search for a particular widget in the UI control tree.
    """

    __slots__ = ('__content',)

    def __init__(self, *args: typing.Union[Item, typing.Iterable[Item]]) -> None:
        """Initialize new :class:`~QWidgetCollection` instance."""
        self.__content: typing.Final[typing.Tuple[Item, ...]] = tuple(
            iteration_utils.unique(flatten(args)),
        )

    @staticmethod
    def from_application(application: typing.Union[QtWidgets.QApplication] = None) -> 'QWidgetCollection':
        """
        Retrieve top-level widgets for the `application`.

        If no `application` provided - then active one will be used.
        """
        if application is None:
            application = QtWidgets.QApplication.instance()
        if application is None:
            return QWidgetCollection()

        return QWidgetCollection(application.topLevelWidgets())

    @property
    def single(self) -> Item:  # noqa: D401
        """Retrieve single item from the collection."""
        if len(self.__content) != 1:
            raise ValueError('object is not single')
        return self.__content[0]

    def children(self) -> QWidgetCollection:
        """Retrieve a collection with all children of widgets in current collection."""
        return QWidgetCollection(
            child
            for item in self.__content
            for child in get_children(item)
        )

    def find(
            self,
            predicate: 'Predicate',
            *,
            min_depth: int = 1,
            max_depth: int = 2 ** 63,
            look_in: typing.Optional['Predicate'] = None,
    ) -> QWidgetCollection:
        """
        Search for a particular widgets, that are nested in widgets from the collection.

        :param predicate: Condition to filter widgets by.
        :param min_depth: Include widgets that are located on at least particular depth.
        :param max_depth: Don't look further than particular depth.
        :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.
        :return: Collection of requested widgets.
        """
        min_depth = max(min_depth, 1)
        if min_depth > max_depth:
            return QWidgetCollection()
        max_depth -= min_depth

        current: QWidgetCollection = self
        while (min_depth > 1) and current:
            if look_in is not None:
                current = current.only(look_in)
            current = current.children()
            min_depth -= 1

        args: typing.List[typing.Iterable[Item]] = [current.only(predicate)]
        while (max_depth > 0) and current:
            if look_in is not None:
                current = current.only(look_in)
            current = current.children()
            args.append(current.only(predicate))
            max_depth -= 1

        return QWidgetCollection(*args)

    def only(self, predicate: 'Predicate') -> QWidgetCollection:
        """
        Filter current collection and leave only particular widgets.

        :param predicate: Condition to filter widgets by.
        :return: Collection of requested widgets.
        """
        return QWidgetCollection(filter(predicate, self.__content))

    def parents(self) -> QWidgetCollection:
        """Prepare collection of parent items."""
        result: typing.List[Item] = []

        item: Item
        for item in self.__content:
            parent: typing.Any = item.parent()
            if isinstance(parent, Item.__args__):  # type: ignore
                result.append(parent)

        return QWidgetCollection(result)

    def print(self) -> str:
        """Print content of the collection."""
        result: str = f'{len(self.__content)} items:'

        item: Item
        for item in self.__content:
            result += f'\n  {type(item).__name__}'

        return result

    def __add__(self, other: QWidgetCollection) -> QWidgetCollection:
        """Merge content of two collections."""
        if isinstance(other, QWidgetCollection):
            return QWidgetCollection(self, other)
        return NotImplemented  # type: ignore

    def __contains__(self, item: typing.Any) -> bool:
        """Check if item is in the collection."""
        return item in self.__content

    @typing.overload
    def __getitem__(self, item: int) -> Item:
        """Retrieve single widget from the collection."""

    @typing.overload
    def __getitem__(self, item: slice) -> QWidgetCollection:
        """Retrieve subset of items."""

    def __getitem__(
            self,
            item: typing.Union[int, slice],
    ) -> typing.Union[Item, QWidgetCollection]:
        """
        Retrieve content from the collection.

        :param item: Index or slice of widgets to retrieve.
        :return: Requested content.
        """
        if isinstance(item, int):
            return self.__content[item]
        return QWidgetCollection(self.__content[item])

    def __iter__(self) -> typing.Iterator[Item]:
        """Iterate through widgets in the collection."""
        return iter(self.__content)

    def __len__(self) -> int:
        """Get total number of widgets in the collection."""
        return len(self.__content)

    def __sub__(self, other: QWidgetCollection) -> QWidgetCollection:
        """Remove items, that are in `other` collection."""
        if isinstance(other, QWidgetCollection):
            return QWidgetCollection(
                item
                for item in self.__content
                if item not in other
            )
        return NotImplemented  # type: ignore


# shortcuts

@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class SearchResult:  # pylint: disable=too-few-public-methods
    """
    Result of searching for Qt widgets.

    Common base that allows result (:attr:`~SearchResult.value`) to be dynamic.
    """

    source: QWidgetCollection

    @property
    def value(self) -> QWidgetCollection:  # noqa: D401
        """Current result of a widget search."""
        return self.source


@attr.s(auto_attribs=True, collect_by_mro=True, eq=False, frozen=True)
class DynamicSearchResult(SearchResult):  # pylint: disable=too-few-public-methods
    """Search result that runs additional filtering each time result is requested."""

    predicate: 'Predicate'
    kwargs: typing.Mapping[str, typing.Any] = attr.ib(factory=dict)

    @property
    def value(self) -> QWidgetCollection:  # noqa: D401
        """Current result of a widget search."""
        return self.source.find(self.predicate, **self.kwargs)


@typing.overload
def search_for(arg1: 'Source') -> SearchResult:
    """Search for Qt widgets in the application."""


@typing.overload
def search_for(
        arg1: 'Predicate',
        *,
        min_depth: int = ...,
        max_depth: int = ...,
        look_in: typing.Optional['Predicate'] = ...,
) -> SearchResult:  # DynamicSearchResult
    """Search for Qt widgets in the application."""


@typing.overload
def search_for(
        arg1: 'Source',
        arg2: 'Predicate',
        *,
        min_depth: int = ...,
        max_depth: int = ...,
        look_in: typing.Optional['Predicate'] = ...,
) -> DynamicSearchResult:
    """Search for Qt widgets in the application."""


def search_for(
        arg1: typing.Union['Source', 'Predicate'],
        arg2: typing.Optional['Predicate'] = None,
        **kwargs: typing.Any,
) -> SearchResult:
    """
    Search for Qt widgets in the application.

    May take:

    - Instance of a Qt widget or :class:`~QWidgetCollection`. In this case it should just return the instance.

    - Instance of a Qt widget or :class:`~QWidgetCollection` and a predicate. This would look for children of instance
      which passes the predicate check.

    - Just a predicate. This would look for all widgets in an application that pass the predicate check.

    Additionally, this function may also take additional parameters to use with :meth:`~QWidgetCollection.find`:

    :param min_depth: Include widgets that are located on at least particular depth.
    :param max_depth: Don't look further than particular depth.
    :param look_in: Condition to limit widgets in which we are looking for `predicate`-compliant widgets.
    """
    if callable(arg1):
        arg2 = arg1
        arg1 = QWidgetCollection.from_application()
    elif not isinstance(arg1, QWidgetCollection):
        arg1 = QWidgetCollection(arg1)

    if arg2 is None:
        return SearchResult(source=arg1)
    return DynamicSearchResult(source=arg1, predicate=arg2, kwargs=kwargs)
