# -*- coding: utf-8 -*-

"""
Internal preferences of the Navigator.

This module should contain all preferences for the Navigator components, that are constant for the current Navigator
release.

.. note::

    The primary goal of this file - is to have a single place with configurations, instead of spreading them across the
    whole application. Thus, if we need to change a single URL, period, behavior - we may just look into a single file
    instead of looking across the related components to what should be changed.

.. warning::

    If you need any additional data structure for any preference - put it in the
    :mod:`~anaconda_navigator.config.structures`.

    The :mod:`~anaconda_navigator.config.preferences` should contain only preference, which should make it much easier
    to navigate through the file.
"""

from __future__ import annotations

__all__ = ()

import collections.abc
import os
import typing

from . import base
from . import structures


MILLISECONDS: typing.Final[int] = 1
SECONDS: typing.Final[int] = 1000 * MILLISECONDS
MINUTES: typing.Final[int] = 60 * SECONDS
HOURS: typing.Final[int] = 60 * MINUTES
DAYS: typing.Final[int] = 24 * HOURS

BYTES: typing.Final[int] = 1
KILOBYTES: typing.Final[int] = 1024 * BYTES
MEGABYTES: typing.Final[int] = 1024 * KILOBYTES
GIGABYTES: typing.Final[int] = 1024 * MEGABYTES


# ╠════════════════════════════════════════════════════════════════════════════════════════════════════════╡ Conda ╞═══╣

CONDA_INDEX_UPDATE_INTERVAL: typing.Final[int] = 7 * DAYS
FEATURED_CHANNELS: typing.Final[collections.abc.Sequence[str]] = ()


# ╠══════════════════════════════════════════════════════════════════════════════════════════════════╡ Main Window ╞═══╣

SIDEBAR_LINKS: typing.Final[collections.abc.Sequence[structures.SidebarLink]] = (
    structures.SidebarLink(
        'Documentation', 'https://www.anaconda.com/docs/tools/anaconda-navigator/main', utm_medium='nav-docs'),
    structures.SidebarLink('Anaconda Blog', 'https://www.anaconda.com/blog', utm_medium='nav-blog'),
)

SIDEBAR_SOCIALS: typing.Final[collections.abc.Sequence[structures.SidebarSocial]] = (
    structures.SidebarSocial('Twitter', url='https://twitter.com/AnacondaInc'),
    structures.SidebarSocial('Github', url='https://github.com/anaconda'),
    structures.SidebarSocial('Youtube', url='https://www.youtube.com/@AnacondaInc.'),
    structures.SidebarSocial('Linkedin', url='https://www.linkedin.com/company/anacondainc/'),
)


# ╠════════════════════════════════════════════════════════════════════════════════════════════╡ Application tiles ╞═══╣

BIGGEST_APPLICATION_CONFIGURATION: typing.Final[int] = 64 * KILOBYTES

USER_APPLICATIONS_ROOT: typing.Final[str] = base.get_conf_path('applications')

REMOTE_APPLICATION_SOURCES: typing.Final[structures.ApplicationSources] = structures.ApplicationSources(
    root=os.path.join(USER_APPLICATIONS_ROOT, 'remote'),
)
REMOTE_APPLICATION_SOURCES.push(group='base', url='https://anaconda.cloud/api/navigator/v1/applications/base')
REMOTE_APPLICATION_SOURCES.push(group='overrides', url='https://anaconda.cloud/api/navigator/v1/applications/overrides')
REMOTE_APPLICATION_SOURCES.lock()


# ╠═══════════════════════════════════════════════════════════════════════════════════════════════╡ Advertisements ╞═══╣

def __init_ad_configuration_paths() -> collections.abc.MutableSequence[str]:
    """Initialize sequence of paths to search configurations in."""
    result: list[str] = []

    if os.name == 'nt':
        # pylint: disable=import-outside-toplevel
        from anaconda_navigator.external.knownfolders import get_folder_path, FOLDERID  # type: ignore
        result.extend((
            os.path.join(get_folder_path(FOLDERID.ProgramData)[0], 'Anaconda3', 'etc', 'partner.yml'),
            os.path.join(get_folder_path(FOLDERID.ProgramData)[0], 'Miniconda3', 'etc', 'partner.yml'),
            os.path.join(os.path.expanduser('~'), 'Anaconda3', 'etc', 'partner.yml'),
            os.path.join(os.path.expanduser('~'), 'Miniconda3', 'etc', 'partner.yml'),
        ))

    else:
        result.extend((
            os.path.join('/', 'etc', 'anaconda', 'partner.yml'),
            os.path.join('opt', 'anaconda3', 'etc', 'partner.yml'),
            os.path.join('opt', 'miniconda3', 'etc', 'partner.yml'),
            os.path.join(os.path.expanduser('~'), 'anaconda3', 'etc', 'partner.yml'),
            os.path.join(os.path.expanduser('~'), 'miniconda3', 'etc', 'partner.yml'),
        ))

    result.append(base.get_conf_path('partner.yml'))

    return result


AD_CONFIGURATION_PATHS: typing.Final[collections.abc.Sequence[str]] = __init_ad_configuration_paths()
AD_SLIDESHOW_TIMEOUT: typing.Final[int] = 60 * SECONDS
AD_SOURCES: typing.Final[collections.abc.Sequence[str]] = (
    'https://anaconda.cloud/api/navigator/v1/simple-advertisement',
)


# ╠══════════════════════════════════════════════════════════════════════════════════════════════════════╡ Updates ╞═══╣

UPDATES_LINK: typing.Final[str] = 'https://anaconda.cloud/api/navigator/v1/whats-new/list'


# ╠═══════════════════════════════════════════════════════════════════════════════════════════════╡ Login reminder ╞═══╣

LOGIN_REMINDER_OFFLINE_CHECK_INTERVAL: typing.Final[int] = 10 * MINUTES


# ╠═══════════════════════════════════════════════════════════════════════════════════════╡ Third party installers ╞═══╣

ALLOW_UNSAFE_INSTALLERS: typing.Final[bool] = False
SAFE_INSTALLER_DOMAINS: typing.Final[collections.abc.Collection[str]] = {  # each must start with a dot
    '.anaconda.cloud', '.anaconda.com',
}
