"""Hardware monitoring utilities."""

from __future__ import annotations

import platform
import subprocess  # nosec
import typing

SYSTEM = platform.system()


def __parse_wmic_output(output: str) -> str:
    """Parse the output of wmic util."""
    return ';'.join(part for line in output.split('\n')[1:] if (part := line.strip()))


def __call_command(command: typing.Sequence[str] | str, *args: typing.Any, **kwargs: typing.Any) -> str:
    """Safe call subprocess."""
    try:
        return subprocess.check_output(command, *args, **kwargs).decode().strip()  # nosec
    except Exception:  # pylint: disable=broad-exception-caught
        return '-'


def get_cpu_model() -> str:
    """Try to get CPU model name using platform specific command."""

    if SYSTEM == 'Windows':
        return __parse_wmic_output(__call_command('wmic CPU get NAME'))

    if SYSTEM == 'Darwin':
        return __call_command('/usr/sbin/sysctl -n machdep.cpu.brand_string', shell=True)  # nosec

    if SYSTEM == 'Linux':
        try:
            with open('/proc/cpuinfo', 'r', encoding='utf-8') as f:
                info = f.readlines()
        except Exception:  # pylint: disable=broad-exception-caught
            info = []
        return next((x.split(':', 1)[1] for x in info if 'model name' in x), '-').strip()

    return '-'


def get_gpu_model() -> str:
    """Try to get GPU model name using platform specific command."""

    if SYSTEM == 'Windows':
        return __parse_wmic_output(__call_command('wmic path win32_VideoController get name'))

    if SYSTEM == 'Darwin':
        return __call_command(
            "system_profiler SPDisplaysDataType | grep -w 'Chipset Model'| sed -n 's/Chipset Model: //p'",
            shell=True) or '-'  # nosec

    if SYSTEM == 'Linux':
        return __call_command(
            '/usr/bin/lspci | grep -i VGA', shell=True
        ).split(': ', 1)[-1].strip() or '-'  # nosec

    return '-'
