# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Common configuration for tests."""

from __future__ import annotations

__all__ = ()

import contextlib
import os
import shutil
import tempfile
import typing
from unittest import mock

import pytest

from tests.mocks import *  # pylint: disable=unused-wildcard-import,wildcard-import
from tests.utilities import import_utils
from tests.utilities import qt_utils
from tests.utilities import snapshot_utils

if typing.TYPE_CHECKING:
    import pytestqt.plugin


# Utilities to make sure that each test is isolated
#
# Such tools should make sure that each test runs in a "fresh" environment, and clean everything up afterwards.


IMPORTS: typing.Final[import_utils.ModuleManager] = import_utils.ModuleManager(
    'anaconda_navigator', 'anaconda_navigator.*',
    'binstar_client', 'binstar_client.*',
    'conda', 'conda.*',
    'conda_token', 'conda_token.*',
    'navigator_updater', 'navigator_updater.*',
)


@contextlib.contextmanager
def isolated_environment() -> typing.Iterator[None]:
    """
    Make sure that context is isolated.

    This will check that all primary packages are not imported globally, and clean up after all actions performed.

    .. note::

        If we would just clean environment before each run - it won't clean top level imports. And these would still be
        shared between runs, which is not something we want.
    """
    if IMPORTS.check():
        raise AssertionError('project-related modules should not be imported on top level in tests')
    try:
        yield None
    finally:
        IMPORTS.release()


def cleanup() -> None:
    """
    Helper function to clear after test is finished.

    .. warning::

        Should be executed within :func:`isolated_environment` context.
    """
    from anaconda_navigator.utils import workers
    with qt_utils.Wait().for_signals(workers.MANAGER.instance.sig_empty).for_timeout(30_000, raise_exception=True):
        workers.MANAGER.instance.cancel_all()
    workers.THREAD_POOL.reset()


@pytest.fixture(scope='function', autouse=True)
def isolated_test(qtbot: pytestqt.plugin.QtBot) -> typing.Iterator[None]:  # pylint: disable=unused-argument
    """Make sure each test is isolated."""
    with isolated_environment():
        try:
            yield None
        finally:
            cleanup()


# Utilities to make sure each tests runs in its own HOME directory.
#
# This is required as some configuration files might be stored there. Such files might be updated by one tests and
# fetched by others, which should not be the case.


MANAGER: snapshot_utils.Manager
if os.name == 'nt':
    MANAGER = snapshot_utils.DirectoryManager('APPDATA', 'HOME', 'LOCALAPPDATA', 'USERPROFILE')
else:
    MANAGER = snapshot_utils.DirectoryManager('HOME')

SNAPSHOT: typing.Final[str] = 'base'


@contextlib.contextmanager
def managed_home_environment(*, requires_snapshot: bool = False) -> typing.Iterator[None]:
    """
    Isolate scope in its own home environment.

    :param requires_snapshot: Initialize new environment from a basic snapshot.
    """
    snapshot: typing.Optional[str] = None
    if requires_snapshot:
        snapshot = SNAPSHOT
    MANAGER.create(snapshot=snapshot)

    try:
        yield
    finally:
        MANAGER.remove()


@pytest.fixture(scope='session')
def isolated_home_base() -> typing.Iterator[None]:
    """Prepare snapshots to create isolated home environments from."""
    # Double-check the isolation level, as further :code:`IMPORTS.release()` may interfere with global isolation check.
    with managed_home_environment(requires_snapshot=False):
        with isolated_environment():
            from anaconda_navigator.api.anaconda_api import AnacondaAPI
            from anaconda_navigator.config import CONF

            AnacondaAPI()

            # configuration tweaks
            CONF.set('internal', 'anaconda_toolbox_installed', True)

        MANAGER.save(snapshot=SNAPSHOT)

    try:
        yield
    finally:
        MANAGER.close()


@pytest.fixture(scope='function')
def isolated_home(
        isolated_home_base: None,  # pylint: disable=redefined-outer-name,unused-argument
) -> typing.Iterator[None]:
    """Create an isolated HOME directory for test."""
    with managed_home_environment(requires_snapshot=True):
        try:
            yield
        finally:
            cleanup()


# Common helper fixtures

@pytest.fixture(scope='function')
def qt_styles(qtbot: 'pytestqt.plugin.QtBot') -> typing.Iterator[None]:  # pylint: disable=unused-argument
    """Load styles for better presentation of Qt widgets."""
    from qtpy import QtWidgets
    from anaconda_navigator.utils import styles

    QtWidgets.QApplication.instance().setStyleSheet(styles.load_style_sheet())

    yield None


@pytest.fixture(scope='function')
def disable_analytics() -> typing.Iterator[None]:
    """Mute blocking analytics thread."""
    with mock.patch('anaconda_navigator.utils.telemetry.core.Analytics._Analytics__run'):
        yield None


@pytest.fixture(scope='function')
def synchronous_workers(
        disable_analytics: None,  # pylint: disable=redefined-outer-name,unused-argument
) -> typing.Iterator[None]:
    """Patch all workers to run in a synchronous mode."""
    from anaconda_navigator.utils import workers

    original: typing.Any = workers.TaskWorker.start
    workers.TaskWorker.start = workers.TaskWorker.run  # type: ignore

    yield None

    workers.TaskWorker.stop = original


@pytest.fixture(scope='function')
def temporary_folder() -> typing.Iterator[str]:
    """Create a temporary folder for a single test."""
    result: str = tempfile.mkdtemp()

    yield result

    shutil.rmtree(result)


@pytest.fixture(scope='function')
def temporary_file() -> typing.Iterator[str]:
    """Create a temporary folder for a single test."""
    file_descriptor: int
    result: str
    file_descriptor, result = tempfile.mkstemp()
    os.close(file_descriptor)

    yield result

    os.remove(result)
