# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Api client tests."""

from __future__ import annotations

__all__ = ()

import os
from unittest import mock
import pytest
from flaky import flaky


NAVIGATOR_TEST_USERNAME = os.environ.get('NAVIGATOR_TEST_USERNAME')
NAVIGATOR_TEST_PASSWORD = os.environ.get('NAVIGATOR_TEST_PASSWORD')
USER_KEYS = ['created_at', 'location', 'user_type', 'company', 'url', 'name']
DEFAULT_TIMEOUT = 10000


def test_api_url():
    from anaconda_navigator.api import client_api

    api = client_api.ClientAPI()
    beta = 'https://api.beta.anaconda.org'
    default = 'https://api.anaconda.org'

    # Switch from beta and back
    api.set_api_url(beta)
    assert api.get_api_url() == beta
    api.set_api_url(default)
    assert api.get_api_url() == default


@flaky(max_runs=4, min_passes=1)
def test_user_logout(qtbot):
    from anaconda_navigator.api import client_api

    api = client_api.ClientAPI()
    api.logout()

    worker = api.login(NAVIGATOR_TEST_USERNAME, NAVIGATOR_TEST_PASSWORD, 'Test', '')
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass
    api.logout()

    user = api.user()
    print('Logout user:', user)
    assert isinstance(user, dict)
    assert len(user) == 0


@flaky(max_runs=4, min_passes=1)
@pytest.mark.skipif(NAVIGATOR_TEST_PASSWORD is None, reason='User and password are available as env variables on CI')
def test_user_login(qtbot):
    from anaconda_navigator.api import client_api

    api = client_api.ClientAPI()
    api.logout()
    worker_login = api.login(NAVIGATOR_TEST_USERNAME, NAVIGATOR_TEST_PASSWORD, 'navigator-tests', '')
    with qtbot.waitSignal(worker_login.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass
    user = api.user()
    print(user)
    assert isinstance(user, dict)
    assert user.get('login') == NAVIGATOR_TEST_USERNAME
    for key in USER_KEYS:
        assert key in user


@pytest.mark.skipif(NAVIGATOR_TEST_PASSWORD is None, reason='User and password are available as env variables on CI')
def test_org_login(qtbot):
    from anaconda_navigator.api import client_api

    api = client_api.ClientAPI()
    api.logout()
    worker_login = api.login(NAVIGATOR_TEST_USERNAME, NAVIGATOR_TEST_PASSWORD, 'navigator-tests', '')
    with qtbot.waitSignal(worker_login.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass
    login = 'goanpeca'
    user = api.organizations(login=login)
    print(user)
    assert isinstance(user, dict)
    assert user.get('login') == login
    for key in USER_KEYS:
        assert key in user


def test_ssl():
    from anaconda_navigator.api import client_api

    api = client_api.ClientAPI()
    default = True
    test = False
    # Switch from beta and back
    api.set_ssl(test)
    assert api.get_ssl() == test
    api.set_ssl(default)
    assert api.get_ssl() == default


def test_client_worker_succeeds(qtbot):
    from anaconda_navigator.api import client_api

    exception_text = 'Text of exception'

    def method(*args, **kwargs):
        raise Exception('(' + exception_text + ')')   # pylint: disable=broad-exception-raised

    def worker_ready(_worker, output, error):
        print('Output', output)
        print('Error', error)
        assert output is None
        assert exception_text in error

    worker = client_api.ClientWorker(method, (1,), {'test': 2})
    worker.sig_finished.connect(worker_ready)

    assert not worker.is_finished()

    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        worker.start()

    assert worker.is_finished()


def test_client_worker_fails(qtbot):
    from anaconda_navigator.api import client_api

    def method(*args, **kwargs):
        return [args, kwargs]

    def worker_ready(_worker, output, error):
        print('Output', output)
        print('Error', error)
        assert error is None
        assert [(1, ), {'test': 2}] == output

    worker = client_api.ClientWorker(method, (1, ), {'test': 2})
    worker.sig_finished.connect(worker_ready)

    assert not worker.is_finished()

    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        worker.start()

    assert worker.is_finished()


def test_reload_api_not_logged_in_user():
    """Checks that Binstar client is loaded with the default Anaconda API config."""
    import binstar_client
    from anaconda_navigator import config as anaconda_config
    from anaconda_navigator.api import client_api

    anaconda_config.CONF.set('main', 'logged_api_url', None)

    api = client_api.ClientAPI()
    api.reload_client()
    assert isinstance(api.anaconda_client_api, binstar_client.Binstar)


def test_reload_api_logged_team_edition():
    """Checks that TeamEdition client is loaded while Anaconda Server user is logged in."""
    from anaconda_navigator import config as anaconda_config
    from anaconda_navigator.api import client_api
    from anaconda_navigator.api import team_edition_api

    anaconda_config.CONF.set('main', 'anaconda_server_api_url', 'http://some-team-edition-url.com')
    api = client_api.ClientAPI()

    anaconda_config.CONF.set_logged_data('http://some-team-edition-url.com', anaconda_config.AnacondaBrand.TEAM_EDITION)

    api.reload_client()
    assert isinstance(api.anaconda_client_api, team_edition_api.TeamEditionAPI)


def test_reload_api_logged_repo4():
    """Checks that Binstar client is loaded while Repo4 user is logged in."""
    import binstar_client
    from anaconda_navigator import config as anaconda_config
    from anaconda_navigator.api import client_api

    anaconda_config.CONF.set('main', 'enterprise_4_repo_api_url', 'http://some-repo4-url.com')
    anaconda_config.CONF.set_logged_data('http://some-repo4-url.com', anaconda_config.AnacondaBrand.ENTERPRISE_EDITION)

    api = client_api.ClientAPI()
    api.reload_client()
    assert isinstance(api.anaconda_client_api, binstar_client.Binstar)


@mock.patch('binstar_client.Binstar.user', mock.Mock(return_value={'login': 'some_user_name'}))
def test_reload_api_logged_through_anaconda_client_cli():
    """
    Checks that Binstar client is loaded if no login action was done through the Navigator app
    but there was a login action through the CLI by using Anaconda Client.
    """
    import binstar_client
    from anaconda_navigator.api import client_api

    api = client_api.ClientAPI()
    api.reload_client()
    assert isinstance(api.anaconda_client_api, binstar_client.Binstar)
