# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for conda API functions."""

from __future__ import annotations

__all__ = ()

import os
import random
import re
import secrets
import string
import tempfile
import typing
import pytest
import yaml


DEFAULT_TIMEOUT = 1200000  # 1 minute
OUTPUT_KEYS = ['success', 'actions']
OUTPUT_KEYS_CLONE = ['success']
CONFIG_GET_KEYS = [
    'add_anaconda_token',
    'add_binstar_token',
    'add_pip_as_python_dependency',
    'allow_other_channels',
    'allow_softlinks',
    'always_copy',
    'always_softlink',
    'always_yes',
    'anaconda_upload',
    'auto_update_conda',
    'binstar_upload',
    'changeps1',
    'channel_priority',
    'channels',
    'create_default_packages',
    'default_channels',
    'disallow',
    'envs_dirs',
    'offline',
    'pinned_packages',
    'pkgs_dirs',
    'shortcuts',
    'show_channel_urls',
    'track_features',
    'update_dependencies',
    'use_pip',
]


def test_get_conda_version(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready(_worker, output, _error):
        assert re.fullmatch(r'^\d+\.\d+\.\d+.*$', output)

    worker_1 = api.get_conda_version()
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_split_canonical_name():
    from anaconda_navigator.api import utils

    def rand(mult: int = 10) -> str:
        return str(int(random.random() * mult))  # nosec

    name = f'package{rand(100000000)}'
    version = '.'.join(rand() for _ in range(6))
    build = f'somebuild_{rand(100)}'
    package_canonical_name = '-'.join([name, version, build])

    _name, _version, _build = utils.split_canonical_name(package_canonical_name)

    assert name == _name
    assert version == _version
    assert build == _build


def test_info(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    used_keys = [
        'channels',
        'conda_build_version',
        'conda_env_version',
        # 'conda_location',  # This key is new and not used
        'conda_prefix',
        'conda_version',
        'default_prefix',
        'env_vars',
        'envs',
        'envs_dirs',
        'offline',
        'pkgs_dirs',
        'platform',
        'python_version',
        'rc_path',
        'requests_version',
        'root_prefix',
        'root_writable',
        'site_dirs',
        'sys.executable',
        'sys.prefix',
        'sys.version',
        'sys_rc_path',
        'user_agent',
        'user_rc_path',
    ]

    def worker_ready(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)  # Check that error is empty '' or None
        assert isinstance(output, dict)
        for key in used_keys:
            # Check the keys have not changed
            assert key in output

    worker_1 = api.info()
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_create_invalid():
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    with pytest.raises(TypeError):
        api.create(pkgs=['python'])

    with pytest.raises(TypeError):
        api.create(name='name', prefix='prefix')

    with pytest.raises(TypeError):
        api.create(name='name', prefix='prefix', pkgs=[])

    with pytest.raises(TypeError):
        api.create(prefix='prefix')

    with pytest.raises(TypeError):
        api.create(name='name')


def test_create_remove_name(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready2(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert isinstance(output, dict)

    def worker_ready(worker_2, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert isinstance(output, dict)
        for key in OUTPUT_KEYS:
            assert key in output
        assert output.get('success')

        worker_3 = api.remove_environment(name=worker_2.test_name)
        worker_2.sig_finished.connect(worker_ready2)
        with qtbot.waitSignal(worker_3.sig_finished, timeout=DEFAULT_TIMEOUT):
            pass

    pkgs = ['anaconda-client']
    name = f'testenv_{int(random.random() * 10000000)}'  # nosec
    worker_1 = api.create(name=name, pkgs=pkgs)
    worker_1.test_name = name
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_create_remove_prefix(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready2(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert isinstance(output, dict)

    def worker_ready(worker_2, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert isinstance(output, dict)
        for key in OUTPUT_KEYS:
            assert key in output
        assert output.get('success')

        worker_3 = api.remove_environment(prefix=worker_2.test_prefix)
        worker_2.sig_finished.connect(worker_ready2)
        with qtbot.waitSignal(worker_3.sig_finished, timeout=DEFAULT_TIMEOUT):
            pass

    pkgs = ['anaconda-client']
    name = f'testenv_{int(random.random() * 10000000)}'  # nosec
    prefix = os.path.join(api.ROOT_PREFIX, 'envs', name)
    worker_1 = api.create(prefix=prefix, pkgs=pkgs)
    worker_1.test_prefix = prefix
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_install_dry_keys(qtbot):
    """Conda 4.3 used a list for the actions, 4.4 uses a dict."""
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready_3(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert isinstance(output, dict)
        assert 'success' in output

    def worker_ready_2(worker_3, output, error):
        print('Output:', output)
        print('Error:', error)
        assert isinstance(output, dict)
        assert 'success' in output
        assert 'actions' in output
        actions = output.get('actions')
        assert isinstance(actions, dict)
        assert output.get('success')

        worker_3 = api.remove_environment(prefix=worker_3.test_prefix)
        worker_3.sig_finished.connect(worker_ready_3)
        with qtbot.waitSignal(worker_3.sig_finished, timeout=DEFAULT_TIMEOUT):
            pass

    def worker_ready(worker_2, output, error):
        print('Output:', output)
        print('Error:', error)
        assert isinstance(output, dict)
        assert 'success' in output

        pkgs = ['pylint']
        worker_2 = api.install(prefix=worker_2.test_prefix, pkgs=pkgs, dry_run=True)
        worker_2.test_prefix = prefix
        worker_2.sig_finished.connect(worker_ready_2)
        with qtbot.waitSignal(worker_2.sig_finished, timeout=DEFAULT_TIMEOUT):
            pass

    pkgs = ['anaconda-client']
    name = 'testenv_' + str(int(random.random() * 10000000))  # nosec
    prefix = os.path.join(api.ROOT_PREFIX, 'envs', name)
    worker_1 = api.create(prefix=prefix, pkgs=pkgs)
    worker_1.test_prefix = prefix
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_clone_environment(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert isinstance(output, dict)
        for key in OUTPUT_KEYS_CLONE:
            assert key in output
        assert output.get('success')

        api.remove_environment(prefix=clone_from_prefix)
        api.remove_environment(prefix=clone_prefix)

    pkgs = ['anaconda-client']
    name = 'testenv_' + str(int(random.random() * 10000000))  # nosec
    clone_from_prefix = os.path.join(api.ROOT_PREFIX, 'envs', name)
    worker = api.create(prefix=clone_from_prefix, pkgs=pkgs)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    clone_prefix = clone_from_prefix + '_clone'
    worker_1 = api.clone_environment(clone_from_prefix, prefix=clone_prefix)
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_clone_invalid():
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    with pytest.raises(TypeError):
        api.clone_environment('clone', name='name', prefix='prefix')

    with pytest.raises(TypeError):
        api.clone_environment('clone')


def test_config_set_valid(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert output.get('always_yes')
        tests = []
        for key in CONFIG_GET_KEYS:
            tests.append(key in output)
        assert any(tests)

    worker_1 = api.config_set('always_yes', True)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    worker_2 = api.config_get(*CONFIG_GET_KEYS)
    worker_2.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_2.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    worker_3 = api.config_set('always_yes', False)
    with qtbot.waitSignal(worker_3.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_config_add_remove_delete(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    channel = f'somefakechannel_{int(random.random() * 1000)}'  # nosec

    def worker_ready_get_add(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        channels = output.get('channels')
        assert isinstance(channels, list)
        assert channel in channels

    worker_1 = api.config_add('channels', channel)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    worker_2 = api.config_get(*CONFIG_GET_KEYS)
    worker_2.sig_finished.connect(worker_ready_get_add)
    with qtbot.waitSignal(worker_2.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    def worker_ready_get_remove(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        channels = output.get('channels')
        assert isinstance(channels, list)
        assert channel not in channels

    worker_3 = api.config_remove('channels', channel)
    with qtbot.waitSignal(worker_3.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    worker_4 = api.config_get(*CONFIG_GET_KEYS)
    worker_4.sig_finished.connect(worker_ready_get_remove)
    with qtbot.waitSignal(worker_4.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    def worker_ready_get_delete(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert not bool(error)
        assert output.get('channels') is None

    worker_5 = api.config_delete('channels')
    with qtbot.waitSignal(worker_5.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    worker_6 = api.config_get(*CONFIG_GET_KEYS)
    worker_6.sig_finished.connect(worker_ready_get_delete)
    with qtbot.waitSignal(worker_6.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_environment_exists():
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    with pytest.raises(TypeError):
        api.environment_exists()

    assert api.environment_exists(name='root')
    assert api.environment_exists(name='test')
    assert api.environment_exists(prefix=api.ROOT_PREFIX)


@pytest.mark.xfail
def test_clear_lock(qtbot):
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def worker_ready(_worker, output, error):
        print('Output:', output)
        print('Error:', error)
        assert output.get('success')
        assert not bool(error)

    worker_1 = api.clear_lock()
    worker_1.sig_finished.connect(worker_ready)
    with qtbot.waitSignal(worker_1.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass


def test_environment_export(qtbot):
    """Check conda environment export function."""
    from anaconda_navigator.api import conda_api

    api = conda_api.CondaAPI()

    def assert_no_error(_worker: typing.Any, _output: typing.Any, error: typing.Any) -> None:
        assert not error

    # Common variables
    tempdir: str = tempfile.gettempdir()

    first_name: str = 'test_' + ''.join(secrets.choice(string.ascii_letters + string.digits) for _ in range(8))
    second_name: str = 'test_' + ''.join(secrets.choice(string.ascii_letters + string.digits) for _ in range(8))

    first_yaml: str = os.path.join(tempdir, first_name + '.yaml')
    second_yaml: str = os.path.join(tempdir, second_name + '.yaml')

    # Create first environment
    worker = api.create(name=first_name, pkgs=['anaconda-client', 'flask', 'click', 'tqdm'])
    worker.sig_finished.connect(assert_no_error)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    # Export first environment
    worker = api.export_environment(file=first_yaml, name=first_name)
    worker.sig_finished.connect(assert_no_error)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    # Check yaml of the first environment
    assert os.path.isfile(first_yaml)

    stream: typing.TextIO
    with open(first_yaml, 'rb') as stream:
        first_content: typing.Any = yaml.load(stream, Loader=yaml.SafeLoader)
    assert first_content['name'] == first_name
    assert any(item.startswith('anaconda-client=') for item in first_content['dependencies'])
    assert any(item.startswith('flask=') for item in first_content['dependencies'])
    assert any(item.startswith('click=') for item in first_content['dependencies'])
    assert any(item.startswith('tqdm=') for item in first_content['dependencies'])

    # Create second environment from dump of the first one
    worker = api.create(name=second_name, file=first_yaml)
    worker.sig_finished.connect(assert_no_error)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    # Export second environment
    worker = api.export_environment(file=second_yaml, name=second_name)
    worker.sig_finished.connect(assert_no_error)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    # Check yaml of the second environment
    assert os.path.isfile(second_yaml)

    stream: typing.TextIO
    with open(second_yaml, 'rb') as stream:
        second_content: typing.Any = yaml.load(stream, Loader=yaml.SafeLoader)
    assert second_content['name'] == second_name
    assert second_content['channels'] == first_content['channels']
    assert second_content['dependencies'] == first_content['dependencies']

    # Remove first environment
    worker = api.remove_environment(name=first_name)
    worker.sig_finished.connect(assert_no_error)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass

    # Remove second environment
    worker = api.remove_environment(name=second_name)
    worker.sig_finished.connect(assert_no_error)
    with qtbot.waitSignal(worker.sig_finished, timeout=DEFAULT_TIMEOUT):
        pass
