# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-function-docstring

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2017 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Test launch applications utilities."""

from __future__ import annotations

__all__ = ()

import os
import pytest


def test_expand_environment():
    from anaconda_navigator.utils import launch

    assert launch.expand_environment('${A}${B}', {}) == '${A}${B}'

    assert launch.expand_environment('>>>${A}${B}<<<', {'A': 'X'}) == '>>>X${B}<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'B': 'Y'}) == '>>>${A}Y<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'C': 'Z'}) == '>>>${A}${B}<<<'

    assert launch.expand_environment('>>>${A}${B}<<<', {'A': 'X'}, default='U') == '>>>XU<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'B': 'Y'}, default='U') == '>>>UY<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'C': 'Z'}, default='U') == '>>>UU<<<'

    assert launch.expand_environment('>>>${A}${B}<<<', {'A': 'X', 'B': 'Y'}) == '>>>XY<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'A': None, 'B': 'Y'}) == '>>>${A}Y<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'A': 'X', 'B': None}) == '>>>X${B}<<<'

    assert launch.expand_environment('>>>${A}${B}<<<', {'A': None, 'B': 'Y'}, default='U') == '>>>${A}Y<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'A': 'X', 'B': None}, default='U') == '>>>X${B}<<<'

    assert launch.expand_environment('>>>${A}${B}<<<', {'A': '${B}', 'B': '${C}', 'C': 'X'}) == '>>>${B}${C}<<<'
    assert launch.expand_environment('>>>${A}${B}<<<', {'A': '${B}', 'B': '${C}', 'C': 'X'}, None, True) == '>>>XX<<<'
    assert launch.expand_environment(
        command='>>>${A}${B}<<<',
        environment={'A': '${C}', 'B': '${D}', 'C': 'X', 'D': 'Y'},
        recursive=True,
    ) == '>>>XY<<<'


@pytest.mark.skipif(os.name == 'nt', reason='Skip on windows as used on unix only')
def test_is_program_installed():
    from anaconda_navigator.utils import launch

    assert launch.is_program_installed('python')
    assert not launch.is_program_installed('bloop')
    assert not launch.is_program_installed('bleep')


@pytest.mark.skipif(os.name != 'nt', reason='Skip on unix as used on win only')
def test_get_command_on_win_spaces(temporary_folder: str) -> None:
    from anaconda_navigator.utils import launch

    tmp_folder = os.path.join(temporary_folder, 'dieter oms')
    root_prefix = 'C:\\prefix with spaces'
    _, command_details = launch.get_command_on_win(
        root_prefix=root_prefix,
        prefix=root_prefix + '\\envs\\someenv',
        command='boo',
        package_name='foobar',
        environment=None,
        cwd=os.path.expanduser('~'),
        default_scripts_path=tmp_folder,
    )

    cmd = command_details['cmd']
    assert 'call "C:\\prefix with spaces\\Scripts\\activate" "C:\\prefix with spaces\\envs\\someenv"' in cmd
    assert '"' + tmp_folder in command_details['cmd']


@pytest.mark.skipif(os.name != 'nt', reason='Skip on unix as used on win only')
def test_get_command_on_win_no_spaces(temporary_folder: str) -> None:
    from anaconda_navigator.utils import launch

    tmp_folder = os.path.join(temporary_folder, 'dieter oms')
    root_prefix = 'C:\\prefix-with-no-spaces'
    _, command_details = launch.get_command_on_win(
        root_prefix=root_prefix,
        prefix=root_prefix + '\\envs\\someenv',
        command='boo',
        package_name='foobar',
        environment=None,
        cwd=os.path.expanduser('~'),
        default_scripts_path=tmp_folder,
    )

    cmd = command_details['cmd']
    assert 'call C:\\prefix-with-no-spaces\\Scripts\\activate C:\\prefix-with-no-spaces\\envs\\someenv' in cmd
    assert '"' + tmp_folder in cmd


@pytest.mark.skipif(os.name != 'nt', reason='Skip on unix as used on win only')
def test_get_command_on_win_no_spaces_2(temporary_folder: str) -> None:
    from anaconda_navigator.utils import launch

    tmp_folder = os.path.join(temporary_folder, 'dieter-oms')
    root_prefix = 'C:\\prefix-with-no-spaces'
    _, command_details = launch.get_command_on_win(
        root_prefix=root_prefix,
        prefix=f'{root_prefix}\\envs\\someenv',
        command='boo',
        package_name='foobar',
        environment=None,
        cwd=os.path.expanduser('~'),
        default_scripts_path=tmp_folder,
    )

    cmd = command_details['cmd']
    assert 'call C:\\prefix-with-no-spaces\\Scripts\\activate C:\\prefix-with-no-spaces\\envs\\someenv' in cmd
    assert f'"{tmp_folder}' not in cmd


@pytest.mark.skipif(os.name == 'nt', reason='Skip on windows as used on unix only')
def test_get_command_on_unix_spaces(temporary_folder: str) -> None:
    from anaconda_navigator.utils import launch

    tmp_folder = os.path.join(temporary_folder, 'dieter oms')
    root_prefix = '/prefix with spaces'
    _, command_details = launch.get_command_on_unix(
        root_prefix=root_prefix,
        prefix=root_prefix + '/envs/someenv',
        command='boo',
        package_name='foobar',
        environment=None,
        cwd=os.path.expanduser('~'),
        default_scripts_path=tmp_folder,
    )

    assert '. "/prefix with spaces/bin/activate" "/prefix with spaces/envs/someenv"' in command_details['cmd']
    assert f'"{tmp_folder}' in command_details['cmd']


@pytest.mark.skipif(os.name == 'nt', reason='Skip on windows as used on unix only')
def test_get_command_on_unix_no_spaces(temporary_folder: str) -> None:
    from anaconda_navigator.utils import launch

    tmp_folder = os.path.join(temporary_folder, 'dieter oms')
    root_prefix = '/prefix-with-no-spaces'
    _, command_details = launch.get_command_on_unix(
        root_prefix=root_prefix,
        prefix=root_prefix + '/envs/someenv',
        command='boo',
        package_name='foobar',
        environment=None,
        cwd=os.path.expanduser('~'),
        default_scripts_path=tmp_folder,
    )

    assert '. /prefix-with-no-spaces/bin/activate /prefix-with-no-spaces/envs/someenv' in command_details['cmd']
    assert f'"{tmp_folder}' in command_details['cmd']


@pytest.mark.skipif(os.name == 'nt', reason='Skip on windows as used on unix only')
def test_get_command_on_unix_no_spaces_2(temporary_folder: str) -> None:
    from anaconda_navigator.utils import launch

    tmp_folder = os.path.join(temporary_folder, 'dieter-oms')
    root_prefix = '/prefix-with-no-spaces'
    _, command_details = launch.get_command_on_unix(
        root_prefix=root_prefix,
        prefix=root_prefix + '/envs/someenv',
        command='boo',
        package_name='foobar',
        environment=None,
        cwd=os.path.expanduser('~'),
        default_scripts_path=tmp_folder,
    )

    assert '. /prefix-with-no-spaces/bin/activate /prefix-with-no-spaces/envs/someenv' in command_details['cmd']
    assert f'"{tmp_folder}' not in command_details['cmd']
