# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel,missing-class-docstring,missing-function-docstring,no-member,protected-access

# -----------------------------------------------------------------------------
# Copyright (c) 2016-2021 Anaconda, Inc.
#
# May be copied and distributed freely only as part of an Anaconda or
# Miniconda installation.
# -----------------------------------------------------------------------------

"""Tests for login dialog."""

import typing
from unittest import mock, TestCase
import pytest
from qtpy.QtCore import Qt
from . import mixins


@pytest.mark.usefixtures('commercial_dialog_bot')
class TestCommercialLogin(TestCase):

    @mock.patch('anaconda_navigator.widgets.dialogs.login.commercial_dialogs.token_set')
    def test_set_token(self, token_set_mock):
        from anaconda_navigator import config as navigator_config

        self.widget.input_line.setText('TOKEN_MOCK')

        with \
                mock.patch.object(self.widget, 'check_text'), mock.patch.object(self.widget, 'accept'), \
                mock.patch.object(self.widget, 'api'), mock.patch.object(navigator_config.CONF, 'set'), \
                self.qtbot.wait_signal(self.widget.button_apply.clicked, raising=True):

            self.widget.check_text.return_value = True
            self.qtbot.mouseClick(self.widget.button_apply, Qt.LeftButton)

            token_set_mock.assert_called_with('TOKEN_MOCK')
            self.widget.accept.assert_called()
            self.widget.check_text.assert_called()
            self.widget.api.client_reload.assert_called()
            navigator_config.CONF.set.assert_called()

    @mock.patch('anaconda_navigator.widgets.dialogs.login.commercial_dialogs.token_set')
    def test_set_token_failed(self, token_set_mock):
        from anaconda_navigator import config as navigator_config

        with \
                mock.patch.object(self.widget, 'check_text'), mock.patch.object(self.widget, 'accept'), \
                mock.patch.object(self.widget, 'api'), mock.patch.object(navigator_config.CONF, 'set'), \
                self.qtbot.wait_signal(self.widget.button_apply.clicked, raising=True):
            self.widget.check_text.return_value = False
            self.qtbot.mouseClick(self.widget.button_apply, Qt.LeftButton)

            self.widget.check_text.assert_called()
            token_set_mock.assert_not_called()
            self.widget.accept.assert_not_called()
            self.widget.api.client_reload.assert_not_called()
            navigator_config.CONF.set.assert_not_called()

    @mock.patch('anaconda_navigator.widgets.dialogs.login.commercial_dialogs.validate_token')
    def test_check_text(self, validate_token_mock):
        self.widget.input_line.setText('TOKEN_MOCK')
        self.assertTrue(self.widget.check_text())
        validate_token_mock.assert_called()


@pytest.mark.usefixtures('anaconda_org_dialog_bot')
class TestAnacondaOrgLogin(TestCase, mixins.TestBaseLoginPage):
    pass


@pytest.mark.usefixtures('enterprise_login_page_dialog_bot')
class TestEnterpriseLoginPage(mixins.TestBaseLoginPage, TestCase):
    def test_login_finished(self):
        worker, output, error = mock.Mock(), 'TOKEN_MOCK', ''
        self.widget._finished(worker, output, error)
        with mock.patch.object(self.widget, 'accept'):
            self.widget.accept.assert_not_called()


@pytest.mark.usefixtures('enterprise_domain_set_dialog_bot')
class TestEnterpriseSetDomainPage(mixins.TestBaseSetDomainPage, TestCase):
    pass


@pytest.mark.usefixtures('team_domain_set_dialog_bot')
class TestTeamSetDomainPage(mixins.TestBaseSetDomainPage, TestCase):
    pass


@pytest.mark.skip
@pytest.mark.usefixtures('team_login_page_dialog_bot')
class TestTeamLoginPage(mixins.TestBaseLoginPage, TestCase):

    def test_login_finished(self):
        worker, output, error = mock.Mock(), '["token", "refresh_token"]', ''

        with mock.patch.object(self.widget, 'accept'), mock.patch.object(self.widget, 'check_text'):
            self.widget._finished(worker, output, error)
            self.widget.accept.assert_called()
            self.widget.check_text.assert_called()

    def test_login_finished_with_error(self):
        worker, output, error = mock.Mock(), '{"token": "Token", "message": "Unauthorized"}', ''

        with mock.patch.object(self.widget, '_track_error'), mock.patch.object(self.widget, 'check_text'):
            self.widget._finished(worker, output, error)
            self.assertEqual(self.widget.label_message.text(), '<i>Invalid Credentials!</i>')
            self.widget._track_error.assert_called()
            self.widget.check_text.assert_called()

    def test_login_finished_with_exception(self):
        worker, output, error = mock.Mock(), '{BROKEN]', ''

        with mock.patch.object(self.widget, '_track_error'), mock.patch.object(self.widget, 'check_text'):
            self.widget._finished(worker, output, error)
            self.assertEqual(self.widget.label_message.text(), '<i>Unhandled error happened!</i>')
            self.widget._track_error.assert_called()
            self.widget.check_text.assert_called()


@pytest.mark.skip
@pytest.mark.usefixtures('team_add_channels_page_dialog_bot')
class TestTeamAddChannelsPage(TestCase):
    def test_skip(self):
        with self.qtbot.wait_signal(self.widget.button_skip.clicked, raising=True), \
                mock.patch.object(self.widget.api, 'create_login_data'), \
                mock.patch.object(self.widget, 'close_window'):
            self.qtbot.mouseClick(self.widget.button_skip, Qt.LeftButton)
            self.widget.api.create_login_data.assert_called()

    def test_add_channels(self):
        with self.qtbot.wait_signal(self.widget.button_add.clicked, raising=True), \
                mock.patch.object(self.widget.api, 'create_login_data'), \
                mock.patch.object(self.widget, 'close_window'), \
                mock.patch.object(self.widget.channels_table, 'get_selected_channels'):

            self.widget.channels_table.get_selected_channels.return_value = (['default_channel'], ['channel'])
            self.qtbot.mouseClick(self.widget.button_add, Qt.LeftButton)
            self.widget.api.create_login_data.assert_called()

    def test_add_channels_fallback(self):
        with self.qtbot.wait_signal(self.widget.button_add.clicked, raising=True), \
                mock.patch.object(self.widget.api, 'create_login_data'), \
                mock.patch.object(self.widget, 'close_window'), \
                mock.patch.object(self.widget.channels_table, 'get_selected_channels'):

            self.widget.channels_table.get_selected_channels.return_value = ([], ['channel'])
            self.qtbot.mouseClick(self.widget.button_add, Qt.LeftButton)
            self.widget.api.create_login_data.assert_not_called()


@pytest.mark.usefixtures('notice_page_dialog_bot')
class TestNoticePage(TestCase):
    """Basic tests for :class:`~anaconda_navigator.widgets.dialogs.login.NoticePage`."""

    qtbot: typing.Any
    widget: typing.Any

    def test_confirmation(self) -> None:
        """Check that dialog confirmation behaves in an expected way."""
        with self.qtbot.wait_signal(self.widget._accept_button.clicked, raising=True):
            self.qtbot.mouseClick(self.widget._accept_button, Qt.LeftButton)

    def test_title_update(self) -> None:
        """Check that :attr:`~anaconda_navigator.widgets.dialogs.login.NoticePage.title` preserves value."""
        new_value: str = 'NEW_WIDGET_TITLE'
        assert self.widget.title != new_value
        self.widget.title = new_value
        assert self.widget.title == new_value
        self.widget.title = '-'
        assert self.widget.title != new_value

    def test_message_update(self) -> None:
        """Check that :attr:`~anaconda_navigator.widgets.dialogs.login.NoticePage.message` preserves value."""
        new_value: str = 'NEW_WIDGET_MESSAGE'
        assert self.widget.message != new_value
        self.widget.message = new_value
        assert self.widget.message == new_value
        self.widget.message = '-'
        assert self.widget.message != new_value

    def test_button_text_update(self) -> None:
        """Check that :attr:`~anaconda_navigator.widgets.dialogs.login.NoticePage.button_text` preserves value."""
        new_value: str = 'NEW_CONFIRMATION_BUTTON_TEXT'
        assert self.widget.button_text != new_value
        self.widget.button_text = new_value
        assert self.widget.button_text == new_value
        self.widget.button_text = '-'
        assert self.widget.button_text != new_value
