# -*- coding: utf-8 -*-

"""Tests for creating new environments with Navigator GUI."""

from __future__ import annotations

__all__ = ()

import os
import uuid

from tests.utilities import shell_utils
from tests.utilities import qt_utils


def test_create_any_environment(qt_helper: qt_utils.QtHelper) -> None:
    """Create environment with default settings."""
    environment_name: str = uuid.uuid4().hex

    # wait for main window to appear
    main_window = qt_helper.wait_for_one(
        qt_utils.OfType('MainWindow') & qt_utils.IsVisible(),
        timeout=300_000,
    )

    # wait for startup tasks (if any)
    try:
        progress_bar = qt_helper.wait_for_one(
            qt_utils.OfType('QProgressBar') & qt_utils.IsVisible(),
            timeout=30_000,
        )
    except TimeoutError:
        pass
    else:
        qt_helper.wait_for_none(
            progress_bar,
            qt_utils.IsVisible(),
            max_depth=1,
            timeout=1_200_000,
            wait_after=1_000,
        )

    # open environments tab
    environments_button = qt_helper.wait_for_one(
        main_window,
        qt_utils.OfType('ButtonTab') & qt_utils.WithText('Environments'),
        timeout=10_000,
    )
    environments_button.click()
    qt_helper.sleep()

    # wait until environments tab appear
    environments_tab = qt_helper.wait_for_one(
        main_window,
        qt_utils.OfType('EnvironmentsTab') & qt_utils.IsVisible(),
        timeout=10_000,
    )

    # click on "Create" button
    create_environment_button = qt_helper.wait_for_one(
        environments_tab,
        qt_utils.OfType('ButtonToolNormal') & qt_utils.WithText('Create') & qt_utils.IsEnabled(),
        timeout=10_000,
    )
    create_environment_button.click()
    qt_helper.sleep()

    # wait for "Create environment" dialog
    create_environment_dialog = qt_helper.wait_for_one(
        qt_utils.OfType('CreateDialog') & qt_utils.IsVisible(),
        timeout=30_000,
        wait_after=1_000,
    )

    # look for available python/r checkboxes
    checkboxes = qt_helper.find_many(
        create_environment_dialog,
        qt_utils.OfType('CheckBoxBase'),
    )

    # wait until at least one of python/r checkboxes is enabled
    qt_helper.wait_for_many(
        checkboxes,
        qt_utils.IsEnabled(),
        at_least=1,  # len(checkboxes),
        max_depth=1,
        timeout=60_000,
    )

    # enter name of the new environment
    environment_name_entry = qt_helper.find_one(
        create_environment_dialog,
        qt_utils.OfType('LineEditEnvironment') & qt_utils.IsEnabled(),
    )
    environment_name_entry.setText(environment_name)
    qt_helper.sleep()

    # click on "Create" button in the dialog
    create_environment_button = qt_helper.wait_for_one(
        create_environment_dialog,
        qt_utils.OfType('ButtonPrimary') & qt_utils.WithText('Create') & qt_utils.IsEnabled(),
        timeout=10_000,
    )
    create_environment_button.click()
    qt_helper.sleep()

    # wait until navigator finishes with creating environment
    progress_bar = qt_helper.wait_for_one(
        qt_utils.OfType('QProgressBar') & qt_utils.IsVisible(),
        timeout=30_000,
    )
    qt_helper.wait_for_none(
        progress_bar,
        qt_utils.IsVisible(),
        max_depth=1,
        timeout=1_200_000,
        wait_after=1_000,
    )

    # check environment in conda
    conda_environments = [
        item
        for item in shell_utils.conda('env', 'list', '--json')['envs']
        if os.path.basename(item) == environment_name
    ]
    assert len(conda_environments) == 1
    shell_utils.conda('env', 'remove', '-p', *conda_environments)
