# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for the :mod:`anaconda_navigator.config.structures`."""

from __future__ import annotations

__all__ = ()

import typing

import pytest


@pytest.mark.parametrize('values', [
    [(1, '+')],
    [(3, '-')],
    [(2, 'a'), (3, 'b'), (11, 'c')],
    [(7, 'x'), (5, 'y'), (3, 'z'), (1, 'w')],
])
@pytest.mark.parametrize('offset', [0, -3, 7])
def test_intervals(values: typing.Sequence[typing.Tuple[int, str]], offset: int) -> None:
    """Check the behavior of :class:`~anaconda_navigator.config.structures.Intervals`."""
    from anaconda_navigator.config import structures

    # instance to test
    instance: structures.Intervals[str] = structures.Intervals(
        *(structures.Interval(count=count, value=value) for count, value in values),
        offset=offset,
    )

    # expected values to test against
    expected: typing.List[str] = [
        value
        for count, value in values
        for _ in range(count)
    ]

    # check if instance reports expected dimensions
    assert instance.first == offset
    assert instance.last == offset + len(expected) - 1

    # check if instance returns expected values in the [first, last] range
    index: int
    value: str
    for index, value in enumerate(expected, start=offset):
        assert instance.get(index) == value
        assert instance.get(index, strict=False) == value
        assert instance.get(index, strict=True) == value

    # check if instance returns expected values outside instance boundaries
    for index in (instance.first - 100, instance.first - 10, instance.first - 1):
        assert instance.get(index) == expected[0]
        assert instance.get(index, strict=False) == expected[0]

        with pytest.raises(IndexError):
            instance.get(index, strict=True)

    for index in (instance.last + 1, instance.last + 10, instance.last + 100):
        assert instance.get(index) == expected[-1]
        assert instance.get(index, strict=False) == expected[-1]

        with pytest.raises(IndexError):
            instance.get(index, strict=True)


def test_intervals_errors() -> None:
    """Check the behavior of :class:`~anaconda_navigator.config.structures.Intervals`."""
    from anaconda_navigator.config import structures

    # empty sequence
    with pytest.raises(TypeError):
        structures.Intervals()

    # invalid values in sequence
    with pytest.raises(ValueError):
        structures.Intervals(structures.Interval(count=-1, value='a'))

    with pytest.raises(ValueError):
        structures.Intervals(structures.Interval(count=0, value='a'))

    with pytest.raises(ValueError):
        structures.Intervals(
            structures.Interval(count=0, value='a'),
            structures.Interval(count=1, value='b'),
            structures.Interval(count=2, value='c'),
        )

    with pytest.raises(ValueError):
        structures.Intervals(
            structures.Interval(count=3, value='d'),
            structures.Interval(count=0, value='e'),
            structures.Interval(count=4, value='f'),
        )

    with pytest.raises(ValueError):
        structures.Intervals(
            structures.Interval(count=5, value='g'),
            structures.Interval(count=6, value='h'),
            structures.Interval(count=0, value='i'),
        )
