# -*- coding: utf-8 -*-

"""Predicates for checking statuses of Qt widgets."""

from __future__ import annotations

__all__ = ['IsEnabled', 'IsHidden', 'IsVisible']

import abc
import typing

from . import core

if typing.TYPE_CHECKING:
    from .. import qwidget_collection


T_contra = typing.TypeVar('T_contra', contravariant=True)


class FlagPredicate(core.GenericPredicate[T_contra], metaclass=abc.ABCMeta):  # pylint: disable=too-few-public-methods
    """
    Common base for predicates with expected flag value.

    :param value: Expected value of a flag. By default - predicate expect corresponding flag to be set.
    """

    __slots__ = ('__value',)

    def __init__(self, value: bool = True) -> None:
        """Initialize new :class:`~FlagPredicate` instance."""
        self.__value: typing.Final[bool] = value

    @property
    def value(self) -> bool:  # noqa: D401
        """Expected flag value."""
        return self.__value


class IsEnabled(FlagPredicate['qwidget_collection.Item']):  # pylint: disable=too-few-public-methods
    """Check if Qt widget is enabled."""

    __slots__ = ()

    def __call__(self, instance: 'qwidget_collection.Item') -> bool:
        """Check if `instance` meets a condition."""
        try:
            return instance.isEnabled() is self.value
        except (AttributeError, TypeError):
            return False


class IsHidden(FlagPredicate['qwidget_collection.Item']):  # pylint: disable=too-few-public-methods
    """Check if Qt widget is hidden."""

    __slots__ = ()

    def __call__(self, instance: 'qwidget_collection.Item') -> bool:
        """Check if `instance` meets a condition."""
        try:
            return instance.isHidden() is self.value
        except (AttributeError, TypeError):
            return False


class IsVisible(FlagPredicate['qwidget_collection.Item']):  # pylint: disable=too-few-public-methods
    """Check if Qt widget is visible."""

    __slots__ = ()

    def __call__(self, instance: 'qwidget_collection.Item') -> bool:
        """Check if `instance` meets a condition."""
        try:
            return instance.isVisible() is self.value
        except (AttributeError, TypeError):
            return False
